package Forks::Queue::SQLite;
use strict;
use warnings;
use Carp;
use JSON;
use DBI;
use DBD::SQLite;
use Time::HiRes 'time';
use base 'Forks::Queue';
use 5.010;    #  using  // //=  operators

our $VERSION = '0.01';
our $DEBUG;
*DEBUG = \$Forks::Queue::DEBUG;

our $jsonizer = JSON->new->allow_nonref(1)->ascii(1);

sub new {
    my ($class, %opts) = @_;
    if ($opts{join} && !$opts{db_file}) {
        croak "Forks::Queue::SQLite: db_file opt required with join";
    }
    $opts{db_file} //= _impute_file();
    $opts{limit} //= -1;
    $opts{on_limit} //= 'fail';
    $opts{style} //= 'fifo';

    if (!$opts{join} && -f $opts{db_file}) {
        carp "Forks::Queue: sqlite db file $opts{db_file} already exists!";
    }

    my $exists = $opts{join} && -f $opts{db_file};
    $opts{_pid} = $$;
    # process id is tied to database handle. If process id doesn't match
    # $self->{_pid}, we must open a new process id.

    my $self = bless { %opts }, $class;

    if (!$exists) {
        my $dbh = DBI->connect("dbi:SQLite:dbname=" . $opts{db_file}, "", "");
        $self->{_dbh} = $opts{_dbh} = $dbh;
        if (!$self->_init) {
            carp "Forks::Queue::SQLite: db initialization failed";
            return;
        }
    } else {
        $self->_dbh;
    }
    return $self;
}

sub _init {
    my $self = CORE::shift;
    my $dbh = $self->{_dbh};

    my $z1 = $dbh->do("CREATE TABLE queue (
                        timestamp decimal(24,12),task text,pid mediumint)");
    return unless $z1;

    my $z2 = $dbh->do("CREATE TABLE pids (pid mediumint)");
    return unless $z2;

    my $sth = $dbh->prepare("INSERT INTO pids VALUES (?)");
    my $z3 = $sth->execute($self->{_pid});
    return unless $z3;

    my $z4 = $dbh->do("CREATE TABLE status(key text,value text)");
    return unless $z4;

    $self->_status("db_file", $self->{db_file});
    $self->_status("owner", $self->{_pid});
    $self->_status("style", $self->{style});
    $self->_status("limit", $self->{limit});
    $self->_status("on_limit", $self->{on_limit});
    return 1;
}

sub _dbh {
    my $self = CORE::shift;
    if ($$ == $self->{_pid} && $self->{_dbh}) {
        return $self->{_dbh};
    }
    $self->{_dbh} = DBI->connect("dbi:SQLite:dbname=".$self->{db_file},"","");
    $self->{_dbh}{AutoCommit} = 1;
    $self->{_pid} = $$;

    $self->{_dbh}->begin_work;
    $self->{_dbh}->do("DELETE FROM pids WHERE pid=$$");
    $self->{_dbh}->do("INSERT INTO pids VALUES ($$)");
    $self->{_dbh}->commit;
    $self->{style} = $self->_status("style");
    $self->{limit} = $self->_status("limit");
    $self->{on_limit} = $self->_status("on_limit");
    return $self->{_dbh};
}

sub DESTROY {
    my $self = CORE::shift;
    my $dbh = $self->_dbh;
    $dbh->begin_work;
    $dbh->do("DELETE FROM pids WHERE pid=$$");
    my $sth = $dbh->prepare("SELECT COUNT(*) FROM pids");
    my $z = $sth->execute;
    my $t = $sth->fetchall_arrayref;
    $dbh->commit;
    $DEBUG and print STDERR "$$ DESTROY npids=$t->[0][0]\n";
    $dbh->disconnect;
    if ($t->[0][0] == 0) {
        $DEBUG and print STDERR "$$ Unlinking files from here\n";
        unlink $self->{db_file} unless $self->{persist};
    }
}

sub _status {
    my $self = CORE::shift;
    my $dbh = $self->_dbh;
    if (@_ == 1) {
        my $sth = $dbh->prepare("SELECT value FROM status WHERE key=?");
        my $z = $sth->execute($_[0]);
        if (!$z) {
            carp "Forks::Queue::SQLite: lookup on status key '$_[0]' failed";
            return;
        }
        my $t = $sth->fetchall_arrayref;
        if (@$t == 0) {
            return;    # no value
        }
        return $t->[0][0];
    } elsif (@_ == 2) {
        my ($key,$value) = @_;
        $dbh->begin_work;
        my $sth1 = $dbh->prepare("DELETE FROM status WHERE key=?");
        my $sth2 = $dbh->prepare("INSERT INTO status VALUES(?,?)");
        my $z1 = $sth1->execute($key);
        my $z2 = $sth2->execute($key,$value);
        $dbh->commit;
        return $z1 && $z2;
    } else {
        croak "Forks::Queue::SQLite: wrong number of args to _status call";
    }
    return;
}

sub end {
    my $self = CORE::shift;
    my $dbh = $self->_dbh;

    my $end = $self->_end;
    if ($end) {
        carp "Forks::Queue: end() called from $$, ",
        "previously called from $end";
    }
    if (!$end || $end != $$) {
        $self->_status("end",$$);
#       $self->{_end} = $$;
    }
    return;
}

sub _end {
    my $self = CORE::shift;
    return $self->{_end} ||= $self->_status("end");
    # XXX - can  end  condition be cleared? Not yet, but when it can,
    #       this code will have to change
}

sub status {
    my $self = CORE::shift;
    my $dbh = $self->{_dbh};
    my $status = {};
    my $sth = $dbh->prepare("SELECT key,value FROM status");
    my $z = $sth->execute;
    my $tt = $sth->fetchall_arrayref;
    foreach my $t (@$tt) {
        $status->{$t->[0]} = $t->[1];
    }
    $status->{avail} = $self->_avail;  # update {count}, {avail}
    $status->{end} //= 0;
    return $status;
}

sub _avail {
    my ($self,$dbh) = @_;
    $dbh ||= $self->_dbh;
    my $sth = $dbh->prepare("SELECT COUNT(*) FROM queue");
    my $z = $sth->execute;
    my $tt = $sth->fetchall_arrayref;
    return $self->{avail} = $tt->[0][0];
}

sub _maintain {
    my ($self) = @_;
    return;
}

sub push {
    my ($self,@items) = @_;
    $self->_push(+1,@items);
}

sub unshift {
    my ($self,@items) = @_;
    $self->_push(-1,@items);
}

sub _add {
    my ($self,$item,$timestamp) = @_;
    my $task = $jsonizer->encode($item);
    my $dbh = $self->_dbh;
    my $sth = $dbh->prepare("INSERT INTO queue VALUES(?,?,?)");
    $dbh->begin_work;
    my $z = $sth->execute($timestamp, $task, $$);
    $dbh->commit;
    return $z;
}

sub _push {
    my ($self,$tfactor,@items) = @_;

    my (@deferred_items,$failed_items);
    my $pushed = 0;

    if ($self->_end) {
        carp "Forks::Queue: push call from $$ after end call from "
            . $self->{_end};
        return 0;
    }

    # TODO: check if queue  limit  is reached, consule  on_limit
    my $limit = $self->{limit};
    $limit = 9E9 if $self->{limit} <= 0;

    while (@items && $self->_avail < $limit) {
        my $item = shift @items;
        $self->_add($item,time * $tfactor);
        $pushed++;
    }
    if (@items > 0) {
        @deferred_items = @items;
        $failed_items = @deferred_items;
    }

    if ($failed_items) {
        if ($self->{on_limit} eq 'fail') {
            carp "Forks::Queue: queue buffer is full ",
                "and $failed_items items were not added";
        } else {
            $DEBUG && print STDERR "$$ $failed_items on put. ",
                                   "Waiting for capacity\n";
            $self->_wait_for_capacity;
            $DEBUG && print STDERR "$$ got some capacity\n";
            $pushed += $self->push(@deferred_items);
        }
    }
    return $pushed;
}

sub _wait_for_task {
    my $self = CORE::shift;
    my $ready = 0;
    do {
        $ready = $self->_avail || $self->_end;
        sleep 1 if !$ready;
    } while !$ready;
    return $self->{avail};
}

sub _wait_for_capacity {
    my $self = CORE::shift;
    if ($self->{limit} <= 0) {
        return 9E9;
    }
    my $ready = 0;
    while (!$ready) {
        last if $self->_avail < $self->{limit};
        last if $self->_end;
        sleep 1;
    }
    return $self->{avail} < $self->{limit};
}

sub shift {
    my $self = CORE::shift;
    return @_ ? $self->_pop(-1,1,@_) : $self->_pop(-1,1);
}

sub pop {
    my $self = CORE::shift;
    return @_ ? $self->_pop(+1,1,@_) : $self->_pop(+1,1);
}

sub _pop {
    my $self = CORE::shift;
    my $tfactor = CORE::shift;
    my $purge = CORE::shift;
    my ($count) = @_;
    $count ||= 1;

    my $order = "timestamp";
    if ($tfactor > 0) {
        $order .= " DESC";
    }
    my $dbh = $self->_dbh;
    my $sths = $dbh->prepare(
        "SELECT task,timestamp,pid FROM queue ORDER BY $order LIMIT ?");
    my $sthd = $dbh->prepare(
        "DELETE FROM queue WHERE task=? AND timestamp=? AND pid=?");
    my @return = ();
    while (@return == 0) {
        my $limit = $count - @return;
        my $z = $sths->execute($limit);
        my $tt = $sths->fetchall_arrayref;
        if (@$tt == 0) {
            if ($self->_wait_for_task) {
                next;
            } else {
                last;
            }
        }
        foreach my $t (@$tt) {
            my ($task,$timestamp,$pid) = @$t;
            CORE::push @return, $jsonizer->decode($task);
            if ($purge) {
		$dbh->begin_work;
                my $zd = $sthd->execute($task,$timestamp,$pid);
                if (!$zd) {
                    carp "purge failed: $task,$timestamp,$pid\n";
                }
		$dbh->commit;
            }
        }
    }
    return @_ ? @return : $return[0];
}

sub peek_front {
    my $self = CORE::shift;
    return $self->_pop(-1,0);
}

sub peek_back {
    my $self = CORE::shift;
    return $self->_pop(+1,0);
}

my $id = 0;
sub _impute_file {
    my $base = $0;
    $base =~ s{.*[/\\](.)}{$1};
    $base =~ s{[/\\]$}{};
    $id++;
    if ($^O eq 'MSWin32') {
        return "C:/Temp/fq-$$-$id-$base.sql3";
    } else {
        return "/tmp/.fq-$$-$id-$base.sql3";
    }
}

sub _DUMP {
    my ($self,$dfh) = @_;
    my $dbh = $self->_dbh;
    $dfh ||= *STDERR;

    my $sth = $dbh->prepare("SELECT * FROM pids");
    my $z = $sth->execute;
    print $dfh "\n\n=== pids ===\n------------\n";
    foreach my $r (@{$sth->fetchall_arrayref}) {
        print $dfh join("\t",@$r),"\n";
    }

    $sth = $dbh->prepare("SELECT * FROM status");
    $z = $sth->execute;
    print $dfh "\n\n=== status ===\n--------------\n";
    foreach my $r (@{$sth->fetchall_arrayref}) {
        print $dfh join("\t",@$r),"\n";
    }

    $sth = $dbh->prepare("SELECT * FROM queue");
    $z = $sth->execute;
    print $dfh "\n\n=== queue ===\n-------------\n";
    foreach my $r (@{$sth->fetchall_arrayref}) {
        print $dfh join("\t",@$r),"\n";
    }
    print $dfh "\n\n";
}

1;

=head1 NAME

Forks::Queue::SQLite - SQLite-based implementation of Forks::Queue

=head1 VERSION

0.01

=head1 SYNOPSIS

    my $q = Forks::Queue->new( impl => 'SQLite', db_file => "queue-file" );
    $q->put( "job1" );
    $q->put( { name => "job2", task => "do something", data => [42,19] } );
    ...
    $q->end;
    for my $w (1 .. $num_workers) {
        if (fork() == 0) {
            my $task;
            while (defined($task = $q->get)) {
                ... perform task in child ...
            }
            exit;
        }
    }

=head1 DESCRIPTION

SQLite-based implementation of L<Forks::Queue|Forks::Queue>.
It requires the C<sqlite3> libraries and the L<DBD::SQLite|DBD::SQLite>
Perl module.

=head1 METHODS

See L<Forks::Queue> for an overview of the methods supported by
this C<Forks::Queue> implementation.

=head2 new

=head2 $queue = Forks::Queue::SQLite->new( %opts )

=head2 $queue = Forks::Queue->new( impl => 'SQLite', %opts )

The C<Forks::Queue::SQLite> constructor recognized the following configuration
options.

=over 4

=item * db_file

The name of the file to use to store queue data and metadata.
If omitted, a temporary filename is chosen.

=item * style

=item * limit

=item * on_limit

=item * join

=item * persist

See L<Forks::Queue> for descriptions of these options.

=back

=head1 LICENSE AND COPYRIGHT

Copyright (c) 2017, Marty O'Brien.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.10.1 or,
at your option, any later version of Perl 5 you may have available.

See http://dev.perl.org/licenses/ for more information.

=cut
