use strict;
use warnings;

use Test::More tests => 2000;
use Crypt::Cipher;

my $trans = {
  "3des"       => 'DES_EDE',
  "safer+"     => 'SAFERP',
  "khazad"     => 'Khazad',
  "safer-k128" => 'SAFER_K128',
  "safer-sk128"=> 'SAFER_SK128',
  "rc6"        => 'RC6',
  "safer-k64"  => 'SAFER_K64',
  "safer-sk64" => 'SAFER_SK64',
  "anubis"     => 'Anubis',
  "blowfish"   => 'Blowfish',
  "xtea"       => 'XTEA',
  "aes"        => 'AES',
  "rc5"        => 'RC5',
  "cast5"      => 'CAST5',
  "skipjack"   => 'Skipjack',
  "twofish"    => 'Twofish',
  "noekeon"    => 'Noekeon',
  "rc2"        => 'RC2',
  "des"        => 'DES',
  "camellia"   => 'Camellia',
};

my $tv;
my $name;
my $size;

while (my $l = <DATA>) {
  $l =~ s/[\r\n]*$//;  
  $l =~ s/^[\s]*([^\s\r\n]+).*?/$1/;
  $l =~ s/\s+//g;
  my ($k, $v) = split /:/, $l;
  next unless defined $k && defined $v;
  $name = $v if lc($k) eq 'cipher';
  if (lc($k) eq 'keysize') {
    $size = $v;
    $size =~ s/bytes//;
  }
  $tv->{$name}->{$size}->{$k} = $v if $name && $k =~ /\d+/;
}

my $seq;
$seq .= pack('C',$_) for(0..255);
my $zeros = '\0' x 255;

for my $n (sort keys %$tv) {
  my $N = $trans->{$n} || die "FATAL: unknown name '$n'";
  for my $ks (sort keys %{$tv->{$n}}) {
    my $key = substr($seq, 0, $ks);
    my $bytes = substr($seq, 0, Crypt::Cipher->blocksize($N));
    for my $i (0..255) {
      next unless $tv->{$n}->{$ks}->{$i};
      my $ct = Crypt::Cipher->new($N, $key)->encrypt($bytes);
      is(unpack('H*', $ct), lc($tv->{$n}->{$ks}->{$i}), "$N/$ks/$i");
      $bytes = $ct; 
      $key = substr($ct x 100, 0, $ks);
    }
  }
}

__DATA__
Cipher Test Vectors

These are test encryptions with key of nn bytes '00 01 02 03 .. (nn-1)' and original PT of the same style.
The output of step N is used as the key and plaintext for step N+1 (key bytes repeated as required to fill the key)

Cipher: aes
Key Size: 16 bytes
 0: 0A940BB5416EF045F1C39458C653EA5A
 1: 2B20AF92A928562CF645B1B824F2E6D9
 2: FC29C3356937ECC3159D8D6EF5E883A1
 3: 4C07B5A2EF31A3229C87AB2E4DE88602
 4: 93AFA1147E793FFCC3D852695A62D008
 5: D4BCC317DC9AFE0E6C7AD1E76F79DBE9
 6: FEDB3371F3C65162AFCCDC6D18C79A65
 7: 4AF2A76F93F07C14161C16B5C176E439
 8: 00A1A596AF7CF44FD12981FA12CB1515
 9: 8013D7006AB38AEBD40D0DC10328751C
10: 81A077F3A262FA4D00D98EE4D1BEC390
11: 0CCBC99A3135F26D2BE824D633C0366F
12: CDBB5568610AD428706408B64DB66E50
13: CE94461EB0D57C8DB6AEB2BC8E8CE1D2
14: 06F14868F4298979462595C0FBF33F5A
15: FE22A7097513246074B7C8DFD57D32B2
16: 0F2D936610F6D9E32C0E624568BB8E6F
17: F32BCD92B563D475E98322E5850AC277
18: 6E6FCB72930D81469F9E05B20FD406C0
19: 42FF674CBA6C19C4AD84D42816173099
20: 41C12474A49B6B2B5E7D38E03A4DD4E0
21: F9E234E3CE3FCED184C775B6140AD733
22: 7EB5CC6B183D8B3EB4FBA4717CD8838A
23: CB6C5D78F9721E5BF8E980F0EDCAD4AF
24: B3F20EF6C26FD9301576D82DA6D50809
25: F9375037377D86599FB4F241166C43E9
26: 98BAF9AB7402479C2DA356F5DAE35D5F
27: 58D1A8E0DC3BC53FD995BB0F60F25FE7
28: 0A75C0D22D2627C97BA2A7344B9B8C74
29: 88C299B2F8C9EDAF86A301BBF534BDA7
30: 755E3A17420281F2C619588A6B521FF9
31: 0E540DD25C0C147461146E11F832A63D
32: DC5B58691C6BA5B243036A41301BD7D1
33: E9299A7336C2D8A51D6C7E2BD1B8F054
34: 78CA6F682FC649DB289DD62D28D3A22D
35: 98D96EDA081DE416083650B22BD3869D
36: E747DE96D122CE1EF6F89BDE0FAE75FF
37: E48DDF2EDDEB54C861A1E42F5B649EEE
38: C650C2CF1E903B7F9C8536A2446CA762
39: CF0BCDCE0F1FE7EB40016C1231FB2962
40: 37B1C8BE3812147E0D5D115A797663EF
41: 45DD8184581049C4B28FBC0809690C5D
42: 11B0D889F96E677EEC2E934E9F7F5398
43: CEC30BC1128A96CD506E406B5ADFAE19
44: DE67D5439BF83D5338D53F362FCF79B6
45: 724FBB2D95CBEABC568AA44941D9B6E5
46: C63F480DA3C73B2A661F1FBC3E4D1F89
47: 225CD18789D18FF09C982EF38AEF0AAF
48: B493DEC7E3D11911DEF8788102453670
49: 23E0B12A67DF025CB77CBDF9E295FCAF

Key Size: 24 bytes
 0: 0060BFFE46834BB8DA5CF9A61FF220AE
 1: 597FA00D03EDDC81C2575B4DD6B6AEFD
 2: 4881E4EF69005DCB9110BA327CAC8460
 3: FC4A968AF65FCFF45E4698455918673D
 4: 3079D7B27A3DA5C0805A61CC37109EE0
 5: 9B3F2C7C35806276E2C53826EC1B3C84
 6: FCDFCB1FD9FCF1B63E1AB6737FC154E8
 7: 4A8012AFD410D29CE2CEE0FD195EF9DA
 8: 9F4201C4174C71A3AEF8FD6822197D67
 9: DE3E5E98DA60E895389A1C17E3D50DA1
10: 20C9064A076C01D1BC121A5A2A1F913C
11: BA41A36CD24B515545B8B464B244E5BE
12: 2CC1DE9DBCAC45269C6DBBC9203095F4
13: 2ED2499CFEB30203E6305B3E1C329C4D
14: FD709FC0AB48B204C95B74AD189C8832
15: 7ED298B472C53A4CB7A3BAE588805E86
16: CB0C6FE2BA76901F9EDE752634DCC31D
17: 6C5CA6EFCF7101881507AB8770ACF1DE
18: DEC3C5209E98BBFAA469C5FE6C02A674
19: CFAC040C1198C8264679CACEAA7E9DE7
20: EF990992EBA8ECA7E5F95E3B9D69D3A4
21: 8FC1B640EB55A96D08D83D1184B77769
22: E1F3DFB9D055BCB2D6CED6DCB8361BFB
23: 6621F47057706F2A079819DBC0197B9C
24: 882611AC68778CBD6A46FB5DD4611A37
25: F35E1367A283CC641FBCE26512A8F2F1
26: 5A4A71F69056CFBAB67DDA777F5CD945
27: C446F2BFAD060A9E9E17F71B05ADABD0
28: 1F0E50F71A67FAA7D169A7A1017FFD65
29: A6A38588848915509451A2354D2AAC8E
30: 4C887574F2C5DB00ED4FBAF814A70302
31: 1B642944162A049CCA9FD0284D7AB4C3
32: 431BD9293C5BFD12F948C255C838880B
33: 32CD23A30039AE2FB80B804B905362B1
34: EBB30E07E7517580A645CD1B5F664182
35: 292F2BB28BB172620B05C7621BA347D6
36: 46C06E1223F392D57B98EFCF4C832C18
37: 451DFBAD2AA92080204F85432236A42C
38: 768D6206D2B3DD1B9C26FAA5977A6477
39: 3705F9CEBFE8F91ECE07F84578C05494
40: 085EB0DCF360F5403FF1E7402A0F7A03
41: 2A0D56F2E7C7FCE3095F511BDE4AD9A2
42: A8AB2F3643A61AF164F99FEFAE2CE1B4
43: E73FD4B1FAE0E5E6A6A31CCC2AF96386
44: 578E84FD1AA16FF350374E4FD5FDD529
45: EEAE301DD57084801DB01F8B9C4036CE
46: 1C44A93B404298327857F71962E0604C
47: B5F64CD5835C85A68DC23E26D4B8FF80
48: 6C6F97850A87088AF195D0500B3F5D78
49: 0BAB3A60A25CD1A750C2C443AA01C57A

Key Size: 32 bytes
 0: 5A6E045708FB7196F02E553D02C3A692
 1: 5F7229D6AACF0DAFE3B518C0D4ADBAB4
 2: 96477F47C0A6F482AC4036D2C60FAAD8
 3: 7F791D54914F12E9F0D92F4416EFBEC0
 4: 87DDB19415BEDC42BD361FE380553C5A
 5: 8EDB2A09DC8731DB76D9F67A03AC4D9E
 6: 269A7C08C28D5E4D9355DDBA161F862E
 7: 042A3397BA5029C443DD76755008DB2A
 8: 469C82A94BC5F7B2DF57F0CE1716EE74
 9: 5A84A93077FA19146078310035F4B7E4
10: 28CAF1C0D811F86CFD3C5EFC30DF79F9
11: 05B575D06C2D593B708F7C695CE97571
12: B7E8CACF0A0BD7F2F5DA0B09CC8B8AEC
13: 0ADDE90F66F1BCF38CEC63EFBF9DBD46
14: 9BF99E7F5B8F176DD686AF017D5196E2
15: ABC189EE80D4A4588B3D54DDACCD9778
16: A57405378580B1E8A8D877791300374C
17: D1EF03F72FAB3DB68022FC60A2CEC13D
18: 3D2406231BA17FF7CC973C5E203872DF
19: C3E07233BD101502953D6186001838E4
20: DC281C0CE02A83098C80D6C9463F3449
21: A923023D2390B2230FCE9217776AAAFC
22: 92E28E69009959FB84046C5ED1B64D1A
23: CEF8F684EC64A31C651280CDC942DFC2
24: 5A954684B22691F9CFC60442A654EF61
25: 56A38A0D93188BAA50DFAF2CB799A76C
26: 54503340C5DE26679AA5F35215DE85EA
27: E74BFAF64946DFD699583FF9C47A1EAF
28: 01F234F9868B085E9B1A2EC84738E2DB
29: BBCA3DAEAB24EF25BC7B623F4D9FD680
30: 3956C880F7F7D94ABC259D3D86157F27
31: 4672C2149054C839C537BDA1F5BBF8F4
32: CF1E9ACBEB391062593BD88C7948F64D
33: CA5B4E867AE9D8BA2D4416C908EB99F1
34: 36666180C768636CF1708CC5C85A6875
35: 53E396D2755218675494C7AA515A2310
36: C2B7D31A59A602A65E155F80353DB83D
37: 0EBCE19FF6FC03E327A2602F858D835E
38: E47CC2A5E6C7FEF185806E2CFB304D91
39: D61F15FF75E0F523FA3872132A09AF42
40: DCC25495052980986AE30756BA0417DA
41: 451BF5B7C1F1AED9F3D5E18A391EA4DA
42: 1B6B105C580083D23F3A8EACE41B7984
43: 8C2F86CD6C86B67C9EBDCAFC5720E4F8
44: 41360BDB3E4C6836BE0D15B659CEC5AA
45: F972104AD851BAE0AD963817A3F03F58
46: 396095F7C102B5A238110DD3D6D4ADFF
47: F58391AEB9A5D8BB32A3055B37556E81
48: A23789B146CE89C876F3C331901261D8
49: 2684AF345C4B13FA154E93A3E2CD2A90


Cipher: blowfish
Key Size: 8 bytes
 0: 84BF44A1442B8217
 1: 3981205BDD22C01E
 2: 0ACC5CCBA118CD07
 3: DF76980D5E089145
 4: A8503E8D849C599D
 5: 5E56574687038F5F
 6: D63296B036996F50
 7: FD2FD7A0669A9E7A
 8: BC6583720A962585
 9: 4B38C2856256103E
10: 48A4FA354DB3A8A6
11: EF97C32734BE2A10
12: A7467E9C729F8123
13: 04D2507F9C4B5854
14: 57F76A4D406B22D1
15: ED0A3B26D842C8F2
16: 047CB457E9730CD1
17: 9F13BB1A97BF5E2F
18: 628CA4F77161C95A
19: 37C7D8EF718DFD50
20: 2C9A9C655B839B1E
21: AB222D66579DBE0D
22: 57950CDEAD6FAE88
23: 67AAB3669431E403
24: 6B35C87144F6B748
25: 94C2E8A1DBC963C2
26: ECD68F56EED1F78E
27: 2E7BE0B866B1D3C7
28: 6671DCDCB3D8EED4
29: 8ACBE7A2F77FBB35
30: 0BF0AC4EAE284F93
31: 29928AE5DC8A57C6
32: 84E48C27E21264DF
33: 4EF0E943E4F48ED3
34: DA155BEFBFFD2638
35: 611EC83E0931FFBE
36: 3BDDEC15BC543A92
37: D7B9564BBAEE19FC
38: DE44907E9F0A1F11
39: C8638C0594D13614
40: 9E67F1B15418BF14
41: EDF531A083F72852
42: 7E5F8F9A72890BB3
43: 2A0B060E3EDDE9C3
44: 9B4B0F6FE6511106
45: 328658F222C7FCE4
46: F6F1B50B4F9F9C93
47: A620519E69952F5E
48: 24DA24DFE96AD818
49: 226C43435FBDA34A

Key Size: 32 bytes
 0: 46CDCC4D4D14BA2E
 1: C079641BD6584E5A
 2: 38828DF8B4C4920C
 3: B4ABCF6B78A721F3
 4: 8E7E2914CBBA708C
 5: C0EBE7002C888495
 6: C60F404DE7CF9B78
 7: B29E843E1771EF26
 8: 983033386CA6A09B
 9: 76F1F89AFDCF7130
10: BED4E2114F4376FA
11: 879A2B9D19AFAB87
12: 366201BC87532AE5
13: 6F409580FA907A64
14: F7A202F00A38863E
15: 98B0A9C79FFC27B1
16: 1CB68D9BBF8A1A8A
17: C21A2C54E5471D23
18: 76A81C3DFC149934
19: C7A0627412FC323A
20: A034684D7058E7A6
21: AC87722F27029BC2
22: 36A6C2AF10245E3E
23: 1F85B90D11217EBE
24: 9C2A0C383A4AB7D5
25: 11D5C689039CA179
26: B0B38C7077E1450B
27: C59C7DCCC3B8A1BB
28: 9BC539F29208AC24
29: 8546F17C77C60C05
30: B189C3E92AF53844
31: 3C7689163B8D2776
32: 6AFEB9A0671156A8
33: 05514E39F2990008
34: C941E31A2A1F42BF
35: 87C3777C74A730A0
36: 2861667755C8B727
37: AF75A0312433B151
38: F76939331E9C9792
39: 819FF8C81FC7C8DC
40: 31E7B07EB03D170D
41: 696E5EC1A741170E
42: 6C5BF0E0BA48FEC3
43: 6D751BCCDC475797
44: BB5A91D0CA7E60F4
45: 7F7EC0531C88B14C
46: 9F302392529B70E8
47: CAC9A1A88F09AC1D
48: 39D56A652E02D5B0
49: 13641D42BC126517

Key Size: 56 bytes
 0: 373C66BBA50EB9CC
 1: A4E8C602AE3A2CEB
 2: A59F08BA78502F32
 3: D0D4968015F4E4FF
 4: 0D3C2F291E6C2EE0
 5: 3F99F5DADAD5FD2C
 6: 5BA41EC1A506396D
 7: 0BDE3B5B50591D35
 8: 5C4A6AEFA69A115D
 9: ADABFE96D6D159E8
10: F97F0B9C88ACD5C0
11: 8882A163F0F02BB2
12: F00556C9F5A56A32
13: 257615BEC96CC228
14: D58DAEC547DD8B89
15: E677F4953EC44097
16: 20156D066DC37000
17: 6F18E01C6FDF947E
18: C8DFF24F12621237
19: 032F91C5119AE308
20: 394194AD8BC1E5CF
21: 6F24E937F3925581
22: 086A4510D95759F3
23: 073204BADF0EE942
24: 5BC8B8E402D90F43
25: A10B7E9D01DD3809
26: 22C0B183AFFDA506
27: 216306AE6DDBAF3F
28: E275E1F52430A1FD
29: C3BDB49D588D31BB
30: B860818C5923B688
31: BE1BC7A444B0E141
32: E4C4B67900DBC8DB
33: 36D7B7ECB6679A9C
34: C1EAD201EE34BEF7
35: 9ABBC877CE825B14
36: 3B211121C0C3C09A
37: BE3B34FF2E83F5A7
38: 46C2B3E628A53EAD
39: B7E7DDE16C7DFF62
40: 3C9A14C4226EBCC5
41: C5FD46242DB29704
42: D45A96723FF62201
43: BB015D10878CF70D
44: 71DB021BE398D41A
45: 06F52D237F06C260
46: 3552F47E8CCFC73F
47: 769E33828AD5D88E
48: 659861DDF080AA67
49: CF7A13782F317342


Cipher: xtea
Key Size: 16 bytes
 0: FFC52D10A010010B
 1: 9CFB2B659387BC37
 2: 7067D153B259E0D6
 3: 0A1769C085DD67A9
 4: A9D781A1A7B4B292
 5: 6FEF8300DF395062
 6: A67B66CA99B9121C
 7: 006E657E1DAD46D3
 8: 2D63322467438A5B
 9: 4F67A826126BE01D
10: 852C6FD597EBAB00
11: F8DD14F59FF44A20
12: CD4DC4E92B5CD40B
13: 802B89A3EFB75810
14: CCA7D920F69A5491
15: 0DFF98CA4F71CA0E
16: 80118F2AE4E83DE8
17: CD6935285D45D83C
18: 47B4613483889187
19: 87F3F1975B8618E3
20: 49BF15EF40C72DBA
21: F850822AD58AD1CC
22: 9701AD2EF51FD705
23: 705AE7F6FD60420B
24: E885CC84A9866B28
25: 93E0D712D27E4E22
26: 8C9CE43E517D3324
27: 31004841AF51FB0E
28: B250BEBF0E58457C
29: 78290B6D83D442E9
30: 3EC72388709CC6E2
31: 099FB875AB5CA6EA
32: B15E20B58F5E8DD0
33: A41511E198E0B1E7
34: B8B5CDD9607B6B40
35: BEF9624E922DB8AC
36: AF198FCD314D8DD4
37: 1A37E433C261EF9D
38: AB7895A2E9D41EE4
39: 4C95BE8D34A7D75B
40: 0D90A8EB03F2852E
41: 9AAD1D630D835C67
42: 6AD88003661B2C5E
43: 4FA7E2CC53EBA728
44: 862245D794441522
45: FAB262C13D245B3E
46: C0A29AA315A5721E
47: F98617BBEFA6AD6A
48: 6F84EAB462F10F36
49: 30850051303CDB96


Cipher: rc5
Key Size: 8 bytes
 0: 04F6B9B18E6828C1
 1: BEA50D165E50EA04
 2: 6F3728FE19F09B03
 3: C682C26278B372FE
 4: 78BCC81E144E1B0F
 5: B62775716366450F
 6: 5BC49690F97CBCFC
 7: 359414E9EACDE379
 8: D3331D8ECBF684FF
 9: 13129FB10EAFC82E
10: 7F29218421CC4B5A
11: FC225A4F31A75162
12: 29BF8BFDA8A15D37
13: 6854AC5BD98EEE95
14: DEF05AB6D102E992
15: 317C3EA6F0600016
16: D6F3658B2E80B77F
17: 7C1DF7ED6C92C23D
18: F8665145BAFE28C5
19: 9D8059C34B79F0EF
20: DC8D1617D3EBC7DB
21: 2D8FF59FCA19BE6C
22: 5C6956743715EA13
23: 91160BE1F4F3D4A0
24: 1D482C2359EC93F5
25: 9C21D1A3755A7251
26: E48D1BB926D51E63
27: 08A054946263F617
28: 9C900BA37199B7C7
29: 0C6C58743EC83082
30: B24197EEB5603D3D
31: CF5B735F8A492E49
32: 337C504698BBE553
33: 3A2BCCC88BE9ED51
34: F3E9241743203ABF
35: B67BCC625815C585
36: F8E96E4EEBC2566C
37: E27D162006C67C8D
38: 08CE8C1A5E3C169A
39: 0CF8AD04125EFCD8
40: 6712F9F044864FAA
41: 0FD465AFFD64085E
42: 6BA8C82B3974391F
43: A5FFF24CE9659357
44: 0947F81E1EB4970E
45: DEA966CA50243067
46: 1F1BE4866F48E39F
47: 03A7D7CE793F52C7
48: A1FADE3F801B414A
49: DE7DA6D14A50E305

Key Size: 68 bytes
 0: C8704ABBDA9624EE
 1: C719C645E301FC16
 2: 32D82553B0E35EF8
 3: C63C77EE6C2A6E36
 4: F84EDA1E77ECB5F0
 5: 382C1E72AA1FD1BC
 6: 6B00939F535F9C83
 7: 3CE0825AE10C2B0E
 8: 1F9E7738602BDD0A
 9: 9273E7933CED0B0A
10: 4CAB45EEA45C32DC
11: FD0208C6A89FB519
12: 520D8E6912E9551D
13: 5B88684544868BD5
14: 32AA2A8EE58135D4
15: 281045702DD38766
16: 26D68E073C996747
17: 23DFB9E174D86192
18: E32FD5AF5101E45C
19: 3DEFB679670A143C
20: E616394D859BFE94
21: 217B9BE0ED47DDAD
22: 4F7901A5620EA484
23: 6654C042783F5737
24: 752AA74BACF9BE65
25: 2FAEBEB8B325F89B
26: 6FEA020B058F32CB
27: 2A32682A36691665
28: 338C8AB628A30105
29: DFAE9DD082DFE38C
30: 51B451C29DBA19C4
31: A2993DA9B8C1D5FD
32: 24D92FA331E2F93A
33: 821A961C0524C89D
34: A07BF305EE8964D9
35: 981652361262C5CE
36: 3DD770C3761B695B
37: F36359AFE1C8A07C
38: BEBC629B8938F4A3
39: 2E31DC53F77898B3
40: 52E4104C4E8E6901
41: 75C912DA610525EA
42: 2F280F70963F82DE
43: D7E3FCCA75AEE5DF
44: 8EBC7E825A14A2BB
45: C1138701F64187DB
46: 1294E21ED2550DFA
47: 577820D772BE2C8E
48: 48CE93C46BFD33CD
49: 3B50D264382E03BC

Key Size: 128 bytes
 0: 236CF0A207576E8E
 1: AC12D8F1AE55F057
 2: CEC4230F747B547A
 3: 61EA1B510D033B26
 4: E064F51998E20804
 5: 6247797DF02BAEF7
 6: D25A766244063A7F
 7: 2C2B3FDDA0E07067
 8: 04EED646C3F6FF90
 9: 05487E7702865E4A
10: 6C0A92AC23ED07C5
11: 6E54E768C797F797
12: A7C53BF7B252A884
13: 731052795E12C80B
14: 3E4DAD15A826C43D
15: 10B1191B4012C2A0
16: ADD244B33AEAEF7E
17: F6CC7B5F0885E056
18: E23489F3B7BE438E
19: B0C27661692FDE4C
20: E81CE014DA769F07
21: 7A8BE0D2D52623A8
22: 082F444E00D5E127
23: AE42F684ADD1DAC7
24: 9061BA686F76A450
25: 9BEB7141B8F6F5F0
26: 38CBA9933AEF85E7
27: C66F4245901CB341
28: 8659AA47E6B06BC3
29: 357AB20DCE2DDA3E
30: 236689C2F36976D9
31: 331EFD7D5CF7AD50
32: C373526C2D44DB80
33: 79F7ACBA770F5C92
34: 64325C5A67F364F6
35: DF2F720829FF1694
36: 9EE17F47ED487BC6
37: C41067896AF4CFC5
38: 5776E1E5FBE59933
39: 07A05B1508B739E0
40: B19EF72A65B3C035
41: F8BF5FF4095C0173
42: 7F1226C6CA7B4072
43: 8A6C8F26A97DD33B
44: 62948A9A627E98AD
45: 9EC58E3F8A477D21
46: A656F285AE0684B4
47: 8489690681A53EE5
48: 940915E733721837
49: 1221956BCEE0143B


Cipher: rc6
Key Size: 8 bytes
 0: 6533F7041D69B9B883A5305180A85520
 1: 496683D6356950E8F4AF4582630BE46C
 2: CA421828FCFCEF2F042F6D81F14CBE76
 3: 92DB67F2F057858FC806E071D239F245
 4: 203CDFE0C36434AEDDBE2DA68ADC5EF0
 5: 8EB23BDBD58A279C1F3BF75199FC9E34
 6: 8FA8BB4E772E09DD1EFBE03090E77FF8
 7: 2018803BFD91D157AE20C6E8FF1894B0
 8: 267319634294F0684AFA7B26EB612B3C
 9: 108745E1F2E80864D9043582CD5550EE
10: E4F9EFE5A6C426BB719EA17174890D0A
11: EFFD4CAE649740B3309692AA19ACA227
12: EB909E6D0789F649538E7EA1686FC0F9
13: 0216851E23EDAE52D9964939DA0A7493
14: D6A9CD3429D1679D26A9599EBDE5409A
15: 5DCDECA6E89A7F0EB40649EFDE6420AF
16: B74FD556B06C07BA8D9E517348CC66CC
17: 9A22CB5B73EF1DDE68A5AEF1B1510ECC
18: 77F78557143E08A7449A75A13098FEF8
19: 548FE6700BD17D0AE247B07C2F1AB0E7
20: B7DFD8CB428A36733BBE9A51CF45C072
21: E7E8B7AA2D93E3DE99C543A473CC6760
22: 3FA5821248B0F0AEB5CF00EEF7958F5E
23: 0A655AC6C51DB33849BCDA72DAE106F1
24: 9EE93EAB01E1A1DC57B698C266469481
25: A7D398720E0ABA2D0D143D8306FD5AC8
26: 98A46C94125BD2E5600BD26EEA420F2A
27: F4789EDC3C50BC4186816F14A86403D1
28: F8AFBA8EC652EFDC3AF5EA5CFE143E16
29: CEEEBD4B6724A30E1859A5B4EF9B2B3D
30: 766715B4C4FA7CD4B365A2A67C79E48A
31: 92C5EB7BE61155D79DE0A6F55715DA22
32: 42CF0C9B2BAACB085CB1603688037D0F
33: 6C4BE816F7B573CCFA8BB6E399EEB17F
34: B6D7E606CC99D267ECCFDBC006878691
35: 2048B58B74F9A721B2E33D2EB86F5991
36: 3E458C1015ECB08CC7B8980135E71893
37: E4E28A032CF2F3C8262CD4BBE7A4CDF8
38: 701EAA449AD9E5AF81DF3F436AB25620
39: D1C3FB7C16F5249503EB389A692B112F
40: 7012790DB65526DC87F9A2BF0FBB5664
41: B782A3104FFE65DDB340F713ACFFE25B
42: A155F033E4536FB1176EBDF9EB5FEC4C
43: 8898BCC7A008127014850D5529327352
44: 8F4B3BE150FAA0371CDE69891E52A3C9
45: D371C8283F68DE983C98D98A7563B934
46: DEB679915E8F0D0B65B37918BE4596F7
47: 84D74F7FA199304A47BB37B8AF893CF0
48: 5367B0187496539F6DF6CCE0965B376D
49: 4B9C6011D43CF2D8FAFA2E7105D85024

Key Size: 68 bytes
 0: 6BBF763FF30876A4BBB68A7E7E290770
 1: 59852279E82E98B8B680E5CEE12BB99A
 2: F96989565E5B163EA483FF74ACA22DC9
 3: 221F7A410F5AD73C1C67DEBA99683E0A
 4: 55F058D1D9B8C372E2A28AB6E953A851
 5: 24A8F7E07620A55D69CC3981B92E5CCE
 6: F4D9DA95BF66FE28BA3A84E77E62822D
 7: EE7EAC2BD34DDE6F83E6D4F23C0A49D3
 8: 4218AA697FB7C5D897E66EB00A9FB489
 9: 55A8CDF8608A3B1A6B14275E2258A096
10: 18D50743982F5C8A5C528BDB5896CDFC
11: 391403B889F0CEE865893EBE9F1BF90A
12: F3CA9C30C163C510432D3207DB6967EA
13: B14B6574DF53257BE4508DBE78843B47
14: F52F1E5FD6FB19C1F5466276F9C33A97
15: 9D5AABA86E8B65E4F633B6EDE17516E8
16: 9038CF746F722DA1A0C34461359FD378
17: 398E317E9CC074C2293B59598F72EA64
18: 9D75D897D487DD2B5BC534B4B223ADD1
19: 6C6DFF734BFB9700EDD6B3CFC6E311F7
20: E27591407CA9771F227A5B6B3A77C928
21: 1618F15FFA8E2692A3B3EF8EB6151427
22: FA426AC6161F31F0D63FC9DA97A6A393
23: 1281869E9959DED2CF75F50DA7FAB66A
24: E8BF17E4B76D6DC5C1D07DC80970665A
25: 9A869B6C5EEF391C7E7C4585FFD9FF3A
26: 59564F799AFC984D39A8168D1A30C8C8
27: 1D3927AA2E2C48E6CFEF88D04ADD30DE
28: 39BF89DE1365DF15A8ABA86856ED074B
29: 0CCC4A4DEB36A000E7EB271F5EE54543
30: 26476623D35A514B8833D868426A2BE9
31: C3C85993EA15AB2D056D670707851802
32: BF5F7ED18E1320BAD536DCEDEE1A9AF7
33: 337BDC5FF0F7AD44E6A3F5D6712BD5DF
34: 7DBA76B3D9C818D0CE1A530BC79E77D2
35: 20DF55E617CD2598F18534DA7A22B555
36: B0A0C1BDF9E81B4F07F47D31A9CC8408
37: CB9586F4B27F759B9B9C6C7DB01D26A8
38: 1E79A2894906A64098AC43799CEFED38
39: 82FA120F237EB0B3A1F8B52379B8346F
40: 3DB9848603E3B1134585E5C9001D119B
41: A750875900E244996887EC995131D151
42: 12133748D3745F77C161470A921F73BD
43: A265C351694574B44517FDAD8816133F
44: 5E50CC8281C2A69FD376250D99620DD3
45: 443ABBC1AD5605A0BA05B8E6ABA5D2A1
46: 73546A87B39C54F0639EBEC118ADA007
47: 380244C822817453C75405B727D67C4B
48: 73F1E23DFF05EFAB5D39E503031C4165
49: 8030071491490590A8913AE4B5D082CC

Key Size: 128 bytes
 0: 24B06811BD97AE9512B3799E3189DCD3
 1: 92DBA6269E880F8D8A09143D792A46DE
 2: F956F459C333DFBA4C6A309C613DD661
 3: C31488EA551CC0FC8885F6817CA696FF
 4: F59634FE907F9DF9467BD1059F82DAAC
 5: 051AF11DD2FCF742169B58A786128CE7
 6: 87326A3A4A98CC15B23DFBFFC5AE16D3
 7: 58FCDE2E88A79D5682729ADB4D971142
 8: EAA787D68EB68CA79CCC6BFAC3BE9247
 9: 8BCF6980AEED36AF38B68A50DD454AF0
10: 4B0E31AE48E903DFF52939800BB16DC0
11: 19766AA929B40840715D53D9170C986F
12: F9CAEB36F03CE7B3BB363AC7EB3ACF99
13: C8E34A6BDEDA4DB836DF3D863D02A6EB
14: 370547CEA441FDCBAFD281A8653BE2D4
15: 77E0F33343158A8C3AC3C6D14FD69431
16: 7A985B1368F842C644538C654D081FD3
17: 60E0C8933F43D40003030196E8E881AC
18: 3473474B58AE6BC7582ADD7AE24711B7
19: 75936C8D46F6D5AF5C7EE0E8DCEB5AB2
20: 4A04F619FB0E05F7B07C43F4B9F2E134
21: FD53A5A7F4F0B9D933E38D9F07AC88CD
22: F62EE43B20F582AC77879AD7E66FCCAC
23: 4436AD771624896683D7D29E8879F89F
24: F88F3C0EF2B9FD0CA997BEF17787DA2F
25: FF5576F42CE9A0665A1D6A2420A691D0
26: C077D6AEBA65B61CD1A9AAE17FCFC04D
27: 84D0D7C52D6DB3979BC3F6F34456CB91
28: F163121D9EB7569CA7DE3B7619E0BE51
29: 727D23FB43215467B57DC67A8890CF26
30: 60BA577F3C6627267D7B33E693FB7BCB
31: 82C66B23586CCEA2AE1480B469B3F3C3
32: A65092726D6CF2F77CE00648E2D117B0
33: EC30662CBA891A3380B846DA6C64024E
34: CE1B253FBCE36B217ED1EFBAAAD2E783
35: 9D963CD5E65A9ECD2DAEE93B6C6C1178
36: 1B8E3D07E7BD4BB4248B6A7DF8935980
37: DBC3FD5888B80C4CEFC6C5E170E271CE
38: 307CA8CDDFE5DA152B66E10346BB2E1C
39: 8858250F933650D978B403A4504EA581
40: F06005FA6E56E0C0D96988A3FAD359FC
41: 816CBE37FDE3719804DBFD093E3FD87D
42: 4878C07B127D696214393DDC66F5EB36
43: EFBA6045243050C0D8D82046B17008E8
44: 3D30C3E5892D32BA3C685DC5B186E959
45: D4A4C41F0E6687E4021FB90D7A8A9F81
46: FE1057B2013308C4CE839B4186F43B4C
47: D7333AC65F66ED6D4BB8D069E265020F
48: 33C262F58BF0D91DF2047E799BAA5F37
49: B960A18764D7A6E17FA1F88487EFF192


Cipher: safer+
Key Size: 16 bytes
 0: 7D42607880B424C27E5A73A34FECE686
 1: 1A99D469F1946EA46523083FBAA79CBB
 2: 831385E400FD681C8A30AAD9477F1ABB
 3: 5C1BB8F259CDEC2C1BE0B803C7D9447F
 4: C57C1CBB18D87FCF72A87B247392752D
 5: 1625183B0C595A33FE0E8AE0DCE40ADE
 6: 4AF3A9D6733DC4FFF3422AA5BE5ADC94
 7: 853133894C87A23318DFAD2B247FBFF3
 8: 8C7F68E01A8413D19B9A479246E54722
 9: 8620898ECD3BF91A47CC54E6D9987FA7
10: 33F12ABB7CC6A9041543A2073AEDFFA7
11: A096E46F2834F79C096D0B655EDC9A63
12: 3DD0D7824A87C9F5D8D25F5AF57E70B7
13: 6B7C99E5CD29AC1C5A8D66AB460E5AD5
14: 95A9F6009AB4DD2AC7E8E45F36D91E9C
15: 60CCEFC6630329C341782B17365995A2
16: 0276C96A7B1191BC16C8A9C98468DB05
17: 1F352CB77C21139C058837B8194E3E64
18: 2DB8E340F58844705F217551782F6B4D
19: 34E99832E0722C5AE8F0CA1A50E9E7E2
20: 7E1538DC10C1F56C3723A87BFD127743
21: 36B9714A8ACDC8B8A17E85E2803A8C88
22: 11848329B0DB9DC7768C07D95F0CF662
23: 93ECEDEB4C6369AC56AF1F49345720A6
24: A3ED7F9D17067C2650728E266B623124
25: F33574590B435D1DDBBA925F0D0EA8AD
26: 87E542DBD40DCBD80C4AB52555C264C1
27: 6D806991AB8E3604C8267AC1EBEC2E21
28: 4B7333F87EBB46BB2A8ECD392C85A12B
29: 4FF49ACA62898F558AC065B66CAD0234
30: 62DE7B2133B09544EDD0DF66DC4F5E2A
31: 82195B39FF7B8A85D7F0EE52D19E510F
32: 24FA56176A4F0B37F851CBAB176C9702
33: 85FA9230D9B93CDCC0752FC738211703
34: D441132032BDAC6715F4453CBC2434D8
35: 438AB9BEA8A900368D84EF870EAF7E84
36: 433BE5BFE1529BFA7C5688CFE3BD4DE5
37: 2A79FB6F37AA08533445B8BEA5098EA2
38: B9C986EE45D204B6A1CA152944912C9F
39: 8289C9F78760D02AA71873FD97E2ECB8
40: 48B0D1244523165055BE9A5E95CF852E
41: 471E211E5E784C2AF476DB3CB555CF35
42: F290CBEB1F1009D250F7B12E044B80C3
43: 1B9796D80C3976FE3071B1C01154D42E
44: A80E21A1A1007B69E8BCAE728BBE6A92
45: 652058EF0FAF8549424F998669660931
46: 89418FB4740E9801F6DFFEDC593FA32E
47: 907561A69CFA117F167A52B88777D89C
48: EA2EB4B1EE739320F99894B476A8A51E
49: E627E64AAB6E2FF6B99643A7FBB84BFC

Key Size: 24 bytes
 0: E8975F94A8B1392FBA78CBDDCC8E8F08
 1: 708CEFB68A0281AEA424B3D4698D2F2B
 2: 21A0DE56545BC950FCE4DF209C96CE6F
 3: F2CA4103B703264D46CBC09E13D5B8EE
 4: 2892101077FEE427C434CCFBBAB598B5
 5: C2F191CC5C681CBFC098B264C479B2AD
 6: 308C3B794C8A7971BBA96FE4C727F48E
 7: 8A4F9D4463B5DC4DD461ED0763CDAEA2
 8: B7E1BBBE455AEDF18329A6EECD6E222C
 9: C80DAAE7FBDF56DE05A897FBDBB53DEC
10: 6A3D38758BF0390156F22F83C20F0262
11: CA493DF771E37A93822D6117ED14B60C
12: 623012748826A08F3C59B71FF3D66327
13: A283BCB126B9795D306B129035BCC2DC
14: 3790A6704BB0F119139A0264F7E8B2C8
15: 9B369BBC095428EBD712517B2C4D06F0
16: 0F488018162193ADB11E4C39CFDEE0DC
17: 8AFB7C6FD7D6DD64C2C77DA3A0D91EE2
18: B8BEFA241BA339BF6F059464C533F9F0
19: E76141C8CD54200FAB2F8C2B76AF5FEE
20: 80B4FE57851C0240D81E78DA8200F195
21: 8BF1C690EF5FCE7ADC74E31C24F83F5E
22: D30C4F78703BDE91750E0E49FA0E8982
23: 86C5D1E0B88EF0AF9B899850510000EB
24: FDE727442BCC0305A7B06E6EE179D836
25: 0B4A719342F9226FA680796887A98FA5
26: 980D4BE9AF3E3CF9D4558478D1DD03AB
27: 03ECD11992D3D5864B8D1987966BA544
28: 8DBC2931D7D17657BF38E3931F486850
29: 76AE069E39FA7308BBF507ABE35BC1E8
30: 9541B59CE18EA255CDC06DFD3FFCD1C1
31: 5A382748AE3641ABF6D9CA24A35B2002
32: 9B7A49DCC2CFC5A6078AB41E6F62B4CD
33: 91B2EAC878E5628DBCC7C876B91B73D1
34: 31125CFFC41A0D3528FB16BAE761C47A
35: 916D2A769DA002ADCA8C87F8241BB50D
36: 681C3F601EE5D82F9B040932F2FB4AEF
37: 6B6F32E5EAC2F424074C33F7C988D5FE
38: D15A5FDC2A4956DE61BA6E1C98867242
39: 0747BCFE1B979E947EED5225FAFCA44F
40: 133B43C85CCBC360DF8649BBBD2EB45B
41: 052D943062A6D475D30100EA412C30EE
42: BD6401C591D585F4A82CDCF678679D5B
43: F95D1A5E338F56760C425D972D67B57B
44: 9B1569308608CA00BB1E2BC9E20289A7
45: B6454C52C64F64D69F1D36D2D224A222
46: 529B5B013AE3F37E38BE340D1D200A64
47: 1B65904F177212AC8E9ED4D8DB287810
48: CD5CAC56236E0B9A25A5D306F12D8C4B
49: 01DF7E1D0F5F7A5DAA0B379388425126

Key Size: 32 bytes
 0: 7FBC212996ECE9CA2C4D5BD88FA0B3D9
 1: EA3D788C25CF3BE7F4101EDECF23455B
 2: BD0B98481366AE0869ABA11DB18D1E95
 3: 53393E2B757C90489EB6B47F67837337
 4: E1D350640CCA6C13648C3B57BE00B950
 5: 951E1EF99E6DE72744A9D8D9EBFBA94E
 6: 433E4D4E64B41818097BD5E2EBA01D8E
 7: 8FCBD07E303B381B2935FB82CA0CBF13
 8: CF46569005BD968B9310149E892B4D69
 9: F1B672657C2657AD53BFFE2BA5DDE1D2
10: 0035337210703240F9CF2F5A9184FDB7
11: 773951841F77DCF8A6730109DEDF3E9A
12: E47DC0FB381DB86EBD208A0D649E7B03
13: 0D9E34ADB257146EAB95AF14636301D2
14: AB5D5C106E52AC7662C26F9F27F2CD55
15: 6938F205DC23C3500B7723281E9F626F
16: 3CABD52558D7F418CAF6B41CEC334DAD
17: D2192F1E5AFC3B989C42C396B3764862
18: 59D32E3A41141C6CAA2A8F44FD587D94
19: 483CFECF65D71CB2994E1C12A1A7F907
20: 8F086AD81B1FD5B13195EDB4DAB7DC73
21: EFEB1328CE7AE6A63E7D8B3ECA9E12B9
22: 362AAE308B3BBA588DBCFBB7197E693C
23: B817A136EB30CD127B5015A85A7B6E95
24: 03796E7F244CC916BE02AF61E5FEC42F
25: 5854F2889CFF44B0688718C23B0A822D
26: 0F772AC6E37364AA7B844AEACB33F4A2
27: B3E95F5195BA646DAF333AA89130451F
28: 911A32AF7CC44309B37D75E158A4AB18
29: 232CFE228EB72A949616B307C2BEED15
30: 7C8989F135B8DE6FD26C7792B8354F35
31: E79231779BFB9F792BD179C0625280A8
32: 015F6CCAE8A1275A2E836A207D8254EA
33: 4EB94709245CE1FBF7C6D9587FA92D40
34: 63D82005320F974EFDC4C021FB73ABB5
35: 0F15B2E8E389C2D59289D7DA41ABD07D
36: CEE7FBBF766540D4E39903D0494DB797
37: FB564C18A15D53108C1954FCD3518FC1
38: A67C5F4A4A95AF2BD8E4FC1182B2EEBB
39: 0D354E664C35B9364E4EE9DB8DE0CA76
40: 3295826D52F3B980B50EFF3E9317F1CB
41: BC65592A9C0BADD84F363A175EE6BC54
42: 58DE762ADA621FE2A7A6A83F68E93213
43: AD774FC8402F7DDBB881B703EC836057
44: F1C95AD5E004AF35AE315AE88A2577FA
45: 968775A2C3485875B024B008D96182EC
46: 623E736238B5800ACD9B67D76C84D236
47: 1C5E9F65D43343D743E50C80D0E0F648
48: A62E4A197E15CF90C2569DC38D9FC915
49: 165B139BE483A71381B9A8E93D9473DA


Cipher: twofish
Key Size: 16 bytes
 0: 9FB63337151BE9C71306D159EA7AFAA4
 1: 8CC5A931DEC29B4C7D85595D24FF8405
 2: E867FC0E144FDEA24FEA88D22D8776EA
 3: B450A2969C67D93E1AE3A4093BA4C48F
 4: 7AEA41F9956149F336612E868E42B3C4
 5: F201FBB730E6E58CF9E5AD1DC4844C4C
 6: 13D8869E66412C6C288B5D7F37F2D94A
 7: CD57DDDDB782C0A04C40E47D920799DC
 8: 65371C8ABC919EC09004A7D63D9302BF
 9: CC31DFD3B7DCCC86866CC79D45A89C3F
10: 541D548D311714EF8661BFA721568D16
11: 269F3AA2D2D10DBD3DD3992BFEE23C05
12: F86DA5D73AFBA94A9D592D02B5855D25
13: EAD8D785B50E036437E39F3B27DB4657
14: 2AD0A13C493B9F3EDD259E11A495B205
15: C05F9D72AA384F05B28A3519AF218CA9
16: D442072F588D38AC8B9D447762E9FCF3
17: FDD3BFB91EFD127196FF9F19ADADBF28
18: F56717661B424E41D7DE1CD13E21DF89
19: 0F6C952D9BE6CA49B5147EFD44455A92
20: 6C214935726364F2766BE7B4C2B64362
21: 5D5316D7E057FF481CCC10C7452D1A17
22: 56C78DBD802CC9B040446A3EFF3F12AC
23: A38CEADA8448DBE29C2D11DF2A487715
24: CB2F786AB8063350F3FAE11EC8C65A5B
25: F5B7298B6F558E2C4FCC11744AD22653
26: 01BF89C1B48C5F6918FC6BDC10B12A21
27: A031F25AAFF294EE79220BC76E73E52E
28: 42C94B50E12499DA35F14BB6BB6E684D
29: FD68B6840DC9A271CDE2032EF0200295
30: A9863C1B04B3FE3945D29966F67B87E2
31: 6226D4CEEC1E8AEC1B7B3E13D80A83FF
32: 6100A71B1E3ABBBA48A9ED280DD1617E
33: 5CE93A26D4EFF0CC7DFA2DD43A511871
34: 282D165BFBA0F7F229161BE533BFC4D9
35: E6AC479970891392972B2845C949A068
36: 4E4A887368F8443BE51FA7CD16CF0B87
37: 121AFC81AA2750572B35D100BDC34DB5
38: 7C41FA7E0A18A87E44BE488F331B35E0
39: C8847D295E1F505C04E2F5CE2CBF5E00
40: 4686EE8628BC1BBB92CE825F04B1D5E8
41: 397DFACD19C283B3FC27D3FCBE952254
42: 815B6C69608B5A379E3C9E67FB1BA15A
43: A73E72B912EB3AA4929E9EAF73A448BB
44: 5BC4E2C88512BCD55408CC5AEAD15A91
45: EF20B2BF297456DED1F4AB7BE0747902
46: 3D876135E19BB56B98050450C6B0FD06
47: D68100E1BAD07B438786337C0957471D
48: CE85A91938049B5E21C983F7C45ECA3E
49: 9FACEFFB9D08BB65DDC34E3C575B8442

Key Size: 24 bytes
 0: 95ACCC625366547617F8BE4373D10CD7
 1: 99AEB25CCE73B3796C351A19791ACD52
 2: 56B9D6689830275C1A3E4266B3D56D53
 3: 2F5F822E11F087BCB8A0F428414855A0
 4: 65400F729990FE175AAA894BCFBB1A59
 5: 947BA33D20194BBB0D4904023FB32FFB
 6: 116B0C6744D215AE98DEB1F9FF1D36C0
 7: BA6964C101FA044ED06877E46D540850
 8: A36B18526FA0004CF557C99A3AC1423A
 9: 573099674B8AFC2DD6424A2E4C581B89
10: F46169CFE9496A2696A93EEB7EC561FB
11: 2C64BC052898F3F3A2B78F2591F2BF1E
12: E8A0D40B08585459199DD6ECC108A490
13: 47927989BE5EB55B9A0F294C04FF035F
14: 54A3024E3AD86005A2C44E4B8BDBBEFB
15: D0AD51D1DADFAD7ED0EBCC756DBCDCC9
16: 5DE698B7014C34AA6F0099CBB5937A20
17: 9BA30F49470C6DB9632C5EDE34F8EE73
18: 0BDF558A2AE9298288E60616442D3408
19: 25F6DD23BA4E090E1CFFA6EE3487AFA7
20: DAC5FB99E299D2672F79F0C38E177D83
21: 58CB113430895C9890D96EA60E3DDC23
22: 48A0771F0049B776A44AE8877B57EFFB
23: 2F12B26A4BF7065076417530CDEE14CC
24: AA6ADCB0176D5324C4C1FFD640D502EE
25: 384B52A66F4C7A70ED36091BC3FEA09C
26: 2AFE7ACF071C6B0FD69930A09D6DD5E5
27: 9A2DB9A5E7712A5BFB095D2C09453CA3
28: 716C0EF522A13EA05A48F16BAD3FD10A
29: 44AB46F3CCFD02BDD2C77A332293A4D9
30: CE6AB27A0F60F410B1B3CACD9AB923A8
31: 69EAFAFC171C55D1D90ED1C6E1F3A48F
32: 5EEEB0B7833121AD7D27BCFAF2D4F8ED
33: 47133445A4EBCC60E27B29FCC160FA75
34: 9F1BFEB9715A20D5FA7BA3BFF1A27BBC
35: 516D4615F07756B4DBE7D37EBBF4684E
36: B88744E53073BDA0F1B115E3DB224E83
37: 1B77C3D067BBE00420450BA5CD9A83CA
38: 94B87AC96F8CBFF36B01A68E0651E591
39: 52ACE87A1A8E46655935536FB3701290
40: B406BB632D5B7985400EC78D621C8801
41: 20F9ABCBF97A9917EC6C5DE3CB2C925B
42: 539A8AF920833F9E81A20A6D10543176
43: B79AFB8BB39B0351094F37C6EC93F8A9
44: 5830BD21289FED3F95C3E3740AC6C5BF
45: 86C4AF5839ECB9860B175642ADA32895
46: A6427E5E08CEA2A50B8426B063AEE189
47: 2E872129B5BC5F535BCE2B81F013A103
48: 2203EB9B2BF51FC2025549D0BF1924A7
49: 6A5E383A4FC78F6E03B9B276F81195BE

Key Size: 32 bytes
 0: 8EF0272C42DB838BCF7B07AF0EC30F38
 1: 9F8801E94E41A7DC875665663BFA7912
 2: EBE2CA6B45A0BEE83B54402E1A843A3B
 3: F6C5A1187AEF4B5A88E97E866CD757A1
 4: B3E62CD473E53812EDF9ECE875F39F5B
 5: D8C38B1EC23264BB7C42B5589C1300B2
 6: BE315EB75B944EC9E51F5EAE65F70BD2
 7: D4275409941A44993037364886B60598
 8: FC34F1D4E9C4A634C1349F6D4E9AB94E
 9: BE712387C59A40A8A35F854333A0302E
10: 1F1FE164834BABC71DBFDFCCA7D2B4B6
11: BB2413CCB5347B5157651819E698D988
12: 6EB5523A968ECE965D9AA4B975D7C2EF
13: B5DD49AB7E269F9D8516FB57EB47D57D
14: 74F5D81856F192D49A70B3743945BFC0
15: 95437BB00D57CD88BD52DE0A66D934C6
16: AE4804A975D67C6B6F99176F407AAA3C
17: 5E5B2FB9B2A028A5467B56F8BDBA6980
18: 8C617FF1F9C50A36BE2EC19A629BA96B
19: E3401F7CBE177A1D224117894E7EA08A
20: F8451D9DD31A08BE828FA9AF39708921
21: 5BE66DD577826804817B85A07BCEDE28
22: E426227780943AA1A830B7E7D7F7CA0A
23: B39C7277C3A5CA21897563DBD8DD6D94
24: FA9992385396F959841D1E0E68CCE00D
25: E1DE89B1DD5CC31685558A51CC731E6C
26: 64618455C46C6FF117F19FF300B3B557
27: 882758C02B3C11D406A21012977D4BF8
28: F948B62F8038D3A3AFB73041B2F263AE
29: AE3BF626020D2877052B90B31E70B8A4
30: F1C6DBBC166985C9EC2E1A3A61BD8E75
31: 82C343FA36B6D4E9E7AF6D0B7741FB09
32: 0BFB756EC55AC63BEA71E4A8187C2B10
33: F1941AD10BE60DAD3FBA33CB899B63A3
34: 18236A39CD34743DE7B60B2575A9B204
35: AA37FBC2525F74710D7561D316E8D90B
36: 413E0F72C2B349FE2691554F77162B5C
37: 5B9E6F98B2CA0F637E463BE4A6EFD39E
38: 1B4A4CA36DC60D51BA981392D6070379
39: B1E26163A90F339E33F86D252EFBAB99
40: BB98F9F844FA81B25ECC89A8482404BE
41: CE142F512A42A28F4788847B971AA7E9
42: C5CE782936F3D28C69C2BD45FD7BC117
43: 9B6E142582E0A309EDB550DED51238B0
44: 0D9D80C01612977FF3A2C7A982D0894A
45: A7630C752B1F787B07C382693334C6AF
46: 9F24990F270D575881C746481A59C245
47: C38B5E11479C200B5ACE1D6522FC6B1F
48: 99118D8114D24B6559CC5D9456F1BEDB
49: F8B974A4BC134F39BE9B27BD8B2F1129


Cipher: safer-k64
Key Size: 8 bytes
 0: 533F0CD7CCC6DDF6
 1: C3CD66BB1E5E5C17
 2: 079DFD68F6AF9A79
 3: 84EB4922264A1204
 4: 31F3A7D739C7E42C
 5: 381F88FB46E1DCA2
 6: CAF4AC443E50EF47
 7: 2914E255DA9BDDBB
 8: A160A24120E4FECC
 9: F748C6009FFBC465
10: 8B3CB5784846D2B0
11: 4F98C1621473399B
12: B486B0BC365ABEE9
13: 314EAB2B4E9F7840
14: 613FE3637968A8FE
15: 28935352361E1239
16: 0DCB090233B8EB3C
17: CF0BC7F307586C8B
18: 64DF354F96CB0781
19: D2B73C6BAACA7FB1
20: 638FCEEF49A29743
21: 204C4E0E0C0A8B63
22: F041EF6BE046D8AA
23: 76954D822F5E2C32
24: 6700C60971A73C9E
25: 80019293AA929DF2
26: 8EF4DE13F054ED98
27: 41DDF9845ABA2B7A
28: B91834079643850C
29: 8F44EC823D5D70DC
30: EC2FF8DE726C84CE
31: 25DF59DC2EA22CB5
32: FC1130B511794ABB
33: ED3259359D2E68D4
34: D7773C04804033F6
35: C1A32C114589251C
36: 51647E61EE32542E
37: B95A8037457C8425
38: 4F84B3D483F239EE
39: 458401C3787BCA5E
40: F59B5A93FD066F8A
41: 1450E10189CC4000
42: 0F758B71804B3AB3
43: 51B744B271554626
44: B55ADA1ED1B29F0D
45: 585DF794461FEBDA
46: 3790CC4DCA437505
47: 7F7D46616FF05DFA
48: 6AE981921DFCFB13
49: FE89299D55465BC6


Cipher: safer-sk64
Key Size: 8 bytes
 0: 14A391FCE1DECD95
 1: 16A5418C990D77F4
 2: EE33161465F7E2DD
 3: AB85A34464D58EC4
 4: 3D247C84C1B98737
 5: D88D275545132F17
 6: 00B45A81780E3441
 7: 6830FAE6C4A6D0D3
 8: 93DF6918E1975723
 9: 15AB9036D02AA290
10: 0933666F0BA4486E
11: 93F42DEE726D949C
12: 756E7BA3A6D4DE2E
13: 4922DCE8EED38CFD
14: 8EC07AFBD42DF21C
15: E82BEBCFB1D7C6B4
16: B3EDB4CB62B8A9BA
17: 5521307CA52DD2F3
18: 54B5D75512E1F8F3
19: 1A736293F2D460A8
20: 778C71384545F710
21: CBC041D3BF742253
22: 9C47FC0FDA1FE8D9
23: B84E290D4BF6EE66
24: FC3E514CE66BB9E3
25: E8742C92E3640AA8
26: 4DA275A571BDE1F0
27: C5698E3F6AC5ED9D
28: AC3E758DBC7425EA
29: B1D316FC0C5A59FD
30: 2861C78CA59069B9
31: E742B9B6525201CF
32: 2072746EDF9B32A6
33: 41EF55A26D66FEBC
34: EC57905E4EED5AC9
35: 5854E6D1C2FB2B88
36: 492D7E4A699EA6D6
37: D3E6B9298813982C
38: 65071A860261288B
39: 401EEF4839AC3C2E
40: 1025CA9BD9109F1D
41: 0C28B570A1AE84EA
42: BFBE239720E4B3C5
43: 09FB0339ACCEC228
44: DFF2E0E2631B556D
45: ECE375020575B084
46: 1C4C14890D44EB42
47: EA9062A14D4E1F7F
48: 82773D9EEFCAB1AB
49: 516C78FF770B6A2F


Cipher: safer-k128
Key Size: 16 bytes
 0: 4D791DB28D724E55
 1: 53788205114E1200
 2: 4472BCCAF3DDEF59
 3: FE9B3640ED11589C
 4: 4DDD7859819857D7
 5: 6BF901C4B46CC9DB
 6: 930DBFC0DE0F5007
 7: E89F702158A00D82
 8: BEB661953BF46D50
 9: 6F0DA64C0FD101F9
10: 4EBBCE4E5A37BED8
11: 996EAA0AF92A09AC
12: AED6BB9522E0B00F
13: DF9C643624A271B4
14: 2E5C789DD44EF0CF
15: 86A5BA1060177330
16: 2385DBA4DEBEB4A3
17: 82E2FC765722094D
18: B3CA2161757695EF
19: F8A4C6081F3ABC06
20: 6422316E1BEFFAC8
21: C178511BFBFF380E
22: 049B8CBEDE5942A9
23: 0E181292C1B1DEFC
24: C347BA0632A49E55
25: 32FDA46669714F99
26: 0523743E30C16788
27: 782BE96A93769ED0
28: 9F99C9E8BD4A69D8
29: 104C094F120C926D
30: 1F7EA3C4654D59E6
31: 90C263629BC81D53
32: 1803469BE59FED9E
33: 1478C7C176B86336
34: 362FE111601411FF
35: 6428417432ECC3C8
36: D74C42FCC6946FC5
37: 1A8F3A82C78C2BE6
38: EE22C641DC096375
39: 59D34A0187C5C021
40: F68CC96F09686A30
41: CF8C608BDCC4A7FC
42: D2896AB16C284A85
43: 8375C5B139D93189
44: 0F0462F9D8EBAED0
45: C3359B7CF78B3963
46: E4F7233D6F05DCC9
47: 8533D1062397119B
48: 4B300915F320DFCE
49: A050956A4F705DB9


Cipher: safer-sk128
Key Size: 16 bytes
 0: 511E4D5D8D70B37E
 1: 3C688F629490B796
 2: 41CB15571FE700C6
 3: F1CBFE79F0AD23C8
 4: 0A0DC4AA14C2E8AA
 5: 05740CF7CD1CA039
 6: 24E886AD6E0C0A67
 7: EEF14D7B967066BC
 8: 6ABDF6D8AF85EAA0
 9: 0EB947521357ED27
10: BDD2C15957F9EC95
11: 0989B87A74A2D454
12: 04C793BA2FAB7462
13: 3DAD2FACDDFA3C45
14: D1194935CC4E1BD7
15: BAC0A2C8248FF782
16: 7DD5894A82298C64
17: A59F552A4377C08B
18: 8DDDE41AB4586151
19: 7CC4261B38FFA833
20: E99204D6584158EC
21: AACC8ED0803CB5C4
22: C105CA72A7688E79
23: 3D662FDC35B88C09
24: A4BCEDC0AE99E30E
25: EAECF9B6024D353C
26: 214651A3D34AFF40
27: 807099325F9D73C2
28: 45EC21AEB6B90A24
29: DCED39526687F219
30: 2CC248E301D3101D
31: C7F37AB8570BA13C
32: BB9B31A34A39641B
33: 5314570844948CAC
34: 4581F837C02CD4F4
35: 4E036B1B62303BF3
36: 7B3B88DE1F5492A4
37: CEF2865C14875035
38: 14DE8BEE09A155DE
39: 3AA284C74867161B
40: 3616B4607369D597
41: 07512F57E75EDEF7
42: 710D1641FCE64DC2
43: DB2A089E87C867A2
44: A192D7B392AA2E2F
45: 8D797A62FBFE6C81
46: E52CE898E19BF110
47: 72695C25158CB870
48: 29F945B733FB498F
49: 27057037E976F3FB


Cipher: rc2
Key Size: 8 bytes
 0: 83B189DE87161805
 1: 7DCB9C9E50D15508
 2: C724D535853CDE79
 3: 113AFD4BA7D3D7E3
 4: CFA06CFB93C2280C
 5: B3F291C1AAD9CB07
 6: 606F74D9AAD4FA71
 7: 1CC3F7AD551C5F89
 8: 43F8772BA6C6B60D
 9: 2F933E12F57689DD
10: 2BC1AF0D5571D17E
11: 4123AAFABDB254E5
12: 413E0AD5C709DCE0
13: 6B3CF01A7542BD2F
14: 1E5E2CA0CD605999
15: D0F7C9DC472A0709
16: 00482798857A2BB9
17: EED583440CFA8B48
18: DFB377FE1EE5E055
19: 30511C4C565E8F75
20: F74A72482B43B99E
21: 1EE4EA7849B0B070
22: DB7A4ACF022706FD
23: 2D7EBABC8C8E4DD4
24: 6F0369BF66A8B637
25: 59E5D13B247EE0F6
26: C7BAB430AA89A5FE
27: AE0F03746D38138B
28: 942DF0B523D02482
29: 929CE0963CFA46B1
30: F8C68A91DC53B7CC
31: 5735395C63E15094
32: 7821605C18D83D42
33: DEC892FD743BA6DC
34: 77AC80C1177963D3
35: 3E223EB4EA297456
36: AF63B231D671D9DC
37: 665CA287AF32E92C
38: E451EAB337DC1EB6
39: 95B179EC950992CA
40: B8937115492802AE
41: 355A6E5EDF40A939
42: 353E77D4A5A28D79
43: C958FA5903F921B8
44: C334E296BCB24ABE
45: 4F37F7F652FE31ED
46: 77304A655B03ED67
47: 3548A4209ACB6EE2
48: 696E712B1911B695
49: E4B63641D22A3DDD

Key Size: 68 bytes
 0: 7ED68E8B30A7D5DA
 1: 867E18AE64B783EE
 2: 032E554D6AAD7055
 3: 26BD785D0DDAD48B
 4: FFBD4009ABF53D03
 5: A71006E1E30C3855
 6: 92638EE741BE65B5
 7: FC8C5F28CB38C83D
 8: F03F145CBCC4CF80
 9: A28A7C63F67DDE7B
10: 3089486A2247B72A
11: CDD6E6BA5ED53A8D
12: B75A2DE8CB96A267
13: F74D72A2CD68CEF5
14: 3865AC8D170EEDBA
15: B1B520CE5FC2BA39
16: 658DACFDD701B4EA
17: 6B5C1DA9484FCEDF
18: E3CDFB5755BDFFC1
19: 56DAFF7E9A908880
20: B6F8B970F68BC2BD
21: 7A00DEE6977A91F2
22: 6C0CE4FD2D02B36C
23: 8EDA10D868504648
24: 76FB106689E66EA7
25: 67F45BB28A20E110
26: 5F3207099AF93B07
27: F5E382F7266AB342
28: 0E31AC2E4CEFFBDC
29: 8DBA1B2FC6980FF0
30: 311368E62EC2A9E2
31: 50DD1F7A319727EB
32: F0DE146C9ECF5662
33: 81CE0842CE223F15
34: 4C442535A8BC9AD2
35: 06EE8869DB91EBDA
36: 4078E7CAC29DCEE7
37: 115D619FB705F289
38: 3D3F8A380DCB8FB1
39: 9044E5AB8049D21A
40: 66327F00B07CFC76
41: 811AB23A4AD3F132
42: D4A507E96BB39BC2
43: 51C9E4C9318F87D9
44: D2255C13DBD09E88
45: ECB76BCB961F812D
46: 83D7E39727BBBEC5
47: 415540F0AE34DD65
48: 232D20F9E188E2C7
49: EE37261ABA270B22

Key Size: 128 bytes
 0: 8A8F8E5C5A04C73B
 1: B197CF922883CE71
 2: 8AF3F896460CC597
 3: 755F313AEB18D3B8
 4: F1DB52688BB879A8
 5: 29D68EA6456B1CF9
 6: BE7C444267A7925D
 7: 46A7BF7DED575933
 8: E2C7AD7C902E5E15
 9: 90A38AE1DD69C2EA
10: AA95FA050CD3388C
11: 23822B6AD5B83018
12: 8FD42F0DBFF3FEE1
13: 645098BC94FDE21B
14: 7E43D43EAC50E992
15: 2F540FC66A9E0F43
16: 453E52EA7B2D1F92
17: F6F731E8C5A32C54
18: B1E89646504E4D7C
19: AB8168452A7984E1
20: 044BB0758DB5435B
21: E9BAE7C99A152BFF
22: B758F70708B6597E
23: 23A1EFD0AA925D7E
24: CA60DD174CBA23DC
25: 5E916F2DF7B6B3CF
26: F2723A9BFD82BB62
27: 2BC381F6C048687E
28: 573BFD71896A4133
29: 03DF7250C3D69801
30: 8639060454669BCB
31: E31945F0A87221AB
32: AA39447BBF0A77EA
33: 174F1B65BF6A34A3
34: 2712F966022A9589
35: B6358876D6D56D16
36: 2A92C131E68B77BE
37: 040C6935F4CFC43B
38: F23503C74121C660
39: EDD120883F1F813D
40: AFC6500D34BD9D33
41: 963117444417BDD3
42: 2FC3A58861B4A58E
43: CFDB70ED8BCD1173
44: 91B75760CF38B8D5
45: 93A59048B78B3415
46: 7E60C5E75225D45F
47: D4212C6422878FFA
48: DDD1B241E0E0EF6E
49: 20337DB931078078


Cipher: des
Key Size: 8 bytes
 0: E1B246E5A7C74CBC
 1: 9262AFD8AD571E2E
 2: 36D2067D960EB870
 3: 319834DC66705192
 4: B4508A7C5F012C93
 5: CAD1DECDDEE81342
 6: AE9E1CBB71C719E6
 7: D7FBB1CDAFD5B2C1
 8: BE1A70564E3BFB5A
 9: 96D9EC1407A1BD34
10: 05E4B9AF3A4DABB3
11: 0DC77419E1F12C02
12: F73E62369190A5E3
13: 830C1CA7820ABA2D
14: D2574B6AEED0A4F4
15: BC08782E293546A1
16: A35DCC9AAD1EBFB3
17: 79B2224667B33706
18: F9462FFD2808233A
19: D6421CD0D9697DC5
20: 57CB2173D67E7001
21: DBE2DB0BDC07644F
22: 014E72E7E99C969F
23: 37901B1963D0B29B
24: 8E12C23929125686
25: 73AA9E2A60C327A1
26: 54A19D07D941EAC2
27: ADB8CBBAEE1848D6
28: 3997E866119856B5
29: 4D647526FE7E1E27
30: D574FE7342104F93
31: B84383E34A790E11
32: 322214BE9B93BB6F
33: D4C8E0B7AA139D68
34: 2B9747CD280E48C8
35: F92EB2E3711FEE2C
36: 5CEE84E124B7882B
37: 82427488597FF618
38: B1E8B313D2DC76CF
39: 03E237CD40D7F214
40: 8C8DC1299293E15D
41: D6C7463FE86D4EF8
42: CF1550CACE647779
43: B998B3D32B69F00B
44: 1B94C342C3D91107
45: ABD4551B27F58BE8
46: 2B24D978D1BFB3DA
47: 85361D36950635CB
48: 448009F38F1DBB4A
49: 6B901B2B79B6950C


Cipher: 3des
Key Size: 24 bytes
 0: 58ED248F77F6B19E
 1: DA5C39983FD34F30
 2: 0BD322177B935920
 3: 9D093F3174EDBAE3
 4: 756B1F2CDF02B791
 5: 57C31C2A913C1126
 6: CF936257517C53FA
 7: 5F06305849E5E158
 8: 9A85DFD788C59D19
 9: 6392279BBE29DC1F
10: 76B0AF835D79C4E8
11: 7073F36DB1E31464
12: 276258605F2CAB3B
13: 3B69E97811FDA996
14: 3E6E491D2862A1F3
15: F8F3F68BDB006520
16: FD376E66D401A097
17: CA2FE47825804996
18: 6F5C256F84FD14AF
19: D7B07F5C40FF0CDE
20: 73F2E074F1324496
21: 0B2A35016F24BD21
22: B040D2E3B311C193
23: 3938DEA347112E2E
24: 9D7C1703CEC0BFAA
25: A07949F76BDFAF68
26: 43087EEF52564C4C
27: 11132B599734AF0E
28: 62A04C819FDD9A8C
29: B74F0E5649D33690
30: 8E77E5009B0AA322
31: 5174134B9A1889B9
32: 053E33441D11AE63
33: 01EF381013F86E4C
34: BCA358DEF35DFD60
35: 5908A88513E2E5A0
36: A3214C8367E04B05
37: B2CBBE851A54BE9C
38: B271817F63B3B76A
39: 08AFBF82ABB97D8A
40: 2CE02ED014186B86
41: 63F3011C97F6E990
42: C392351F432738D9
43: 0534DDA50E685725
44: 1509456871F5CC20
45: 2383029935901342
46: 8714F1A53CCB213A
47: 2491B2FD3CE6A7CB
48: 4BB02399D85BB66E
49: B8AC2CDFF7AC22C1


Cipher: cast5
Key Size: 5 bytes
 0: 9B32EF7653DAB4E6
 1: 48AEB06B1BDB2397
 2: B530927183622D58
 3: 0ECC8F24BA742216
 4: F6352F9B6461D82C
 5: AF6315AE98E12A71
 6: C364D4B506EF28B9
 7: 817061BEDF5E0A5D
 8: C19DE7B1F2066C04
 9: A6E1345E3AA1B79D
10: 06B036B04785428F
11: 244DAB3F7223F567
12: B9CF3791F6C79F4A
13: 86C5A02AF0517C5E
14: A16E3198F1317E04
15: CF72A44C01E36DDD
16: 199C72ECD5E632ED
17: 0BC66BF05EB7887A
18: AE1F696F3D6B7952
19: 685C92084F600885
20: DBC1353A95AD2097
21: F481E98CB36CAB3B
22: 8F1C06A482C70BB6
23: EA087739954A9CE5
24: 6D0AD727D8E4EF9D
25: 61CA0F7965FEE246
26: 0D664CA16BA785DA
27: 2359D363755605B9
28: 6B6E3A216ABFB55A
29: 6FBCCF7B0342D3C9
30: 3431F4572E2CBE57
31: 36D84FCE6D5D8FE4
32: C234F6AD41E34E76
33: 522F12E8D0617263
34: AD8199B6E94D4E74
35: 56DEC7C21C83B2AD
36: 22CDBFBC5B476E69
37: 70BAD497381F378D
38: 4584F15DBC0EB7F3
39: DE0603CEE19BCFCD
40: EA8D582C1CE62FC9
41: 4299C28A912CE568
42: 7208AB000E3FA9D4
43: 7AAE6CB398D01665
44: 33A6AA42B6D7949C
45: 2AEC5640AD535D69
46: 74D37D9DD7B67248
47: A5300FFF5CF85833
48: 834F83398D591C38
49: 85C787A8B7E519DB

Key Size: 10 bytes
 0: 95827CB504BD43C7
 1: 8FBF4EBCB8413ABF
 2: 5CFF411BECED9971
 3: CEE2AEB4415E0A5D
 4: BB3A8DF7C54FA88F
 5: D508B933EF335111
 6: 993745722EF0D8D3
 7: 04EFB233AA88E796
 8: 478A7DCEAF243F90
 9: 269CC3D138ED48E7
10: 88EBD14D2F999C89
11: B7441626D4487A20
12: 46A6E2CE6C66058E
13: 60687D2D5381757F
14: 885D05ABBF187B89
15: 5032A7ECD3D51521
16: 50BAF36BC5C14A8B
17: 8E805499569FBB0E
18: F8359B18AF3E69C5
19: F24E415CB4D2AA95
20: 361805D4E45B56B4
21: 3172336F01E3530C
22: 333A551E0A03C4A3
23: E2B991995A2D2962
24: 067CEEDD8F213B67
25: FEC3F306851F8616
26: 4B80DAE6AB11894F
27: 250C26E21A8273A2
28: 313F2A505915C909
29: 42E0DC3D4816B38D
30: 9FAEEF0510DEE721
31: 3BB5F5EF74B4CD7E
32: 0FBC9007F462BEAC
33: B9D1467B0D7A1043
34: D9B991C9348DF321
35: 061D577806C50742
36: 48AEA67AAAB6FA23
37: 22F7910383BDA87C
38: 9987087EDBA56FD8
39: 2FCC8341B69BAA14
40: 29DEDB6C2A433F50
41: E067D2746B99A9CB
42: A5C9CB975A856D57
43: AAFEFD3A82D6185B
44: BBE8952CC93CCCC8
45: FC08CE0934EF2E25
46: E44E642DBA7CF3F0
47: CC26F0E8E85AB372
48: D95D63B8389082E0
49: BCA941C921B91E16

Key Size: 16 bytes
 0: 20B42D77A79EBAE5
 1: 96CF6C91E5183CA2
 2: BD87E77A38DDB4E2
 3: E7454CA30B69DE2D
 4: 888F278D219384EE
 5: 972CB887CDE920F8
 6: 49BEC1E7913F3CAE
 7: 96A81B37FEF63CA5
 8: 408DD23A6DA940FC
 9: DA86E92BB735755F
10: 2032F2D972F228BD
11: 8F9EF7DEEF74EFEA
12: 547C92025DCAF9F4
13: 80CD440DFF2EA62A
14: 7D6141D102E1B941
15: 307596ABF5C9A6B2
16: 82E3F1B17EBD52FE
17: 5917DDD11EDB55A3
18: 2909F77166F24F9F
19: 88BDE9D686328942
20: 8F987B737F6A011A
21: A74B3D1D6433B9F4
22: DA8F95DE949482EC
23: 953BA9B26B9AC476
24: 76A52FE193CBFAF9
25: 4BB7D2597A78F2D8
26: 5C8BE951251F4B1D
27: 6E8AB32A4A279D84
28: BB94BC9937B42848
29: FF0EE686F97BF8DB
30: 4146656AB1477E13
31: 1BFCA7053E6DB8AC
32: 4B9A6A349BFA926E
33: 3B5F6FDD127B0A87
34: 53C996E956598599
35: 62C142E63C28B5EE
36: BBB71D6548F32693
37: 107576AA26352455
38: DE32E31FFE02B6F9
39: 4C5DB81D1715FF5C
40: 8E5C706206F493A6
41: 4BBC51E184A67C92
42: AAE216B413DE2A06
43: 77AE26F467233B06
44: E8680D0E71F6AAD6
45: 7061DCED4BC94F78
46: 06772D63818C7E86
47: EE5B9CFC06CBD639
48: 5784B3EFCDC28DD4
49: 4F962107A2EF843C


Cipher: noekeon
Key Size: 16 bytes
 0: 22C082F55D7F6D861B11C36911BE694F
 1: 0485388F24B147918116347E942BCF4A
 2: 47388A4B060617B21134D3B4EB1CABCA
 3: AA8866CFB9D7507CC67A7F271AEF11E0
 4: F6A078AEF1BDF8B621A76CB732804FF3
 5: 8301F76E39A4E8C8AC38A7751B26DD31
 6: 5BE06821E7B23277B808143F36BABDE0
 7: E326A3A32F4F0D8A4FA94877997DA11B
 8: 2BA7773B55F90B5399C11EA80D6CADEF
 9: E64776D92B81770E51E4E2F44688A59D
10: E987ED52D4C33B2668BB9DCF0889D5AB
11: 351F5BC075D06BC6977D31A442CCC2B6
12: 645468E2497FA5EB913C04032457C1DF
13: 10CFDBEC689B01FB969AA2C760F76CCB
14: 0BC5B171A3B727B9594238EC522F72F0
15: 887D105D54D8EAABABC892F04F3455C0
16: 53CC30B5F16713AC77205B0F194FED59
17: CD63AD99CC0D5F34D67C363F99F7CF1E
18: 59BE7B22114383FE8491304FB291D2BC
19: 4B107C8D37CD46EF1DB68ECF4588FEF3
20: 46034C755D278E368305D1133BA6B4FA
21: E2472AC6D4048AB59E126930F6476D06
22: 821014CDA5084A85058F1D556854D33D
23: F67C3FB5CB1271B454810FEE632F7EE8
24: 57705CB352AF1A8B342E1E555C9DAEAA
25: 72AB36C1A8D3C2111330D0EF78726227
26: 1931783D7E3DD6A33962BAD6962D8A33
27: 06029A07CA801027D97BFAFF4719FB89
28: D78B7E4E3083A60610C42BFC03810590
29: 3CA3B14C5741A43F1FF5AF2179684DBA
30: D1BCC52AE476999E25391E7FFDC59C81
31: 1E102DBAA4224ED5E32515A59A07EDAA
32: 81BE227D2663DBB733F9CB5018AED67C
33: 92C5A77D5D62A16C031DA0BD968FBAC0
34: 9EC8E61B543BE73AAD711A9F58C86790
35: B6A1FD059A7D8D73C143C17D97E4C177
36: 0316ED78EA520EE98BB568413A390E44
37: BEFEE68550E2FAFC4AECBE309031BEFD
38: D394CBCC38A47482B2B6900BD68D6540
39: C58F2EE6C493BD1EB41DEB88A169D240
40: 0A45FFA6D6E888B1F6E95E388818C6AE
41: 8A9CAD2C511F284CE1D77167E5D23456
42: 577CB9155A69CA34213FFD15E03D54F4
43: 2AB7DD760EB7DDDD3883A6966B9D44D2
44: 4564DC5318B0A940CBBC3C1607804B70
45: 0E9F42D9C2AC03694CC2E82BA3C4BBBF
46: A49089D9FD9E13DF35B0490E59A9B7C9
47: D58B3008003D6C8D556D7D76180691FF
48: 1FBC6D5F3F1B0E599DED48FF7A63CB76
49: 077533478FABE8AD5DC2B9E96E7CC6CB


Cipher: skipjack
Key Size: 10 bytes
 0: F62E83484FE30190
 1: 03B4DFE5423A117B
 2: 8CE4DAA2307CF018
 3: 77D8C958DAE4336D
 4: 00D367D5C1FC95D8
 5: C1F1305A5B01A474
 6: C3956225C846F695
 7: 2A8977DC186749A3
 8: 433AC6B56AE5C200
 9: 10489A7E87F803CE
10: F176DF022D02D136
11: 1395AE1C0C00AA1B
12: 0C1C3FF32E93F789
13: 901EAAD562EE92DF
14: 59D55D9EE3EA0154
15: D9135CE0BBF68AC7
16: 90A8E4A8E76349A3
17: C04ED52AA69D1ED0
18: 19E842698B5008A4
19: 26FCA0FA3AA7718D
20: 62635FD1A542C7C0
21: 5A3695398C979E40
22: 34998BB72108D89F
23: F889CF892998D689
24: 2C6A4D61F468F19C
25: EC70D59FC906349B
26: B95F11FD098B34A6
27: 32F86206BB4E525B
28: E6BE51063B60CB9A
29: 8964E47BAC22F995
30: B1C297883819747B
31: F2AE1F39F55FB6C2
32: E633EA2DE342767E
33: AF1F0ECBCA788A28
34: 6A898F4407696B27
35: CD9CB5374EA080BD
36: 15881B0200AE6A42
37: 579D05E5F5DE7840
38: 86F8C683D23EB976
39: FDAC7DC6C8F7777D
40: 10D6F7641409F027
41: FCDAA0872D1EC61A
42: 7A353991A81344DC
43: 43661187956D3F8D
44: 5190FDFB904A78F0
45: EF651E67F65CCD57
46: 5E539C61748BDE3D
47: E11E23BA8BEBA42E
48: BAEF0956173B32AD
49: 0AAB29DF65861F4C


Cipher: anubis
Key Size: 16 bytes
 0: 30FF064629BF7EF5B010830BF3D4E1E9
 1: DD7A8E87CFD352AF9F63EA24ADA7E353
 2: 0D0BE8F05510EBD6A3EC842E5BD9FC2A
 3: 330F09581FDC897B3FE6EC1A5056A410
 4: 30349D965F43C295B9484C389C4D942C
 5: 9225343F0056BC355060C0282C638D02
 6: E3A85D41B5337533C4D87730948A9D4E
 7: 09DA0DDB65FF431081CAB08A28010B76
 8: 6C0D0BD6CEAFB9783B31023FD455DAC6
 9: FBE6F26B7CA322A45312856D586DE2EE
10: 1F269EC072D0FBA72E87CA77F8B983FB
11: CFFAE9ADE3006BD511ED172D42F16D05
12: 73F0E9DE89F4C7541506F052D181BAC2
13: FCFA3E2E89FF769834295C77431EF7CE
14: 0452360383D56F827C81263F6B0855BC
15: 40744E07299D6A2A210BE5598835221B
16: 2F0FC61148C36F4C7B42DF274AD0DDE0
17: 2EA0E9BE9E4E4DF85488FE6E7CFCD6E3
18: 0AD1254FA64C3996BBD485D41A3687A0
19: 5B55988652DF200348A114F802FD3C03
20: C32906AF76934C1436CA60BAD58A0C66
21: 59D87987DE9DD485C4537F3A95A164A0
22: 0A706ADF488D84632C96F4BEC43D9FA8
23: 0B74E0CDD14D984B37491E2D9FA63CAE
24: 47CB1827D151A60473E67BD5D233102F
25: F455B4B665D3D0AFB25FDE4A3312AFF6
26: F9A0649421D45DF604206854F681DBDB
27: 21477F5546339E4B6D8215368EE9F884
28: 577640F23CA73345701B0906DFABA4B7
29: 89F8D08A6E173759020DD7301E0FE361
30: 44EF7AF7043FD4B8112345CEE42BC969
31: D7CF0CE04A57253F4C63CABC4A5CB034
32: AF73D3F4CED32593B315E27079131D22
33: F6E603E3455359FE43A3B83AAF3AF0C5
34: DCC3FB557F2C301B631DEF499097E4FD
35: 8285A25CF6F7E701644708E12081C62C
36: EC702DD0293F4C646B1C9C2606762816
37: 289491E5A65DCA605B78E88DA8A9F8AB
38: D82FBC14452BE34C5840DAD81FC2A65E
39: B88A340EB1BF8D5ADE6A4E6C16104FC8
40: C9FC3D70D2BA26C4059BD3D34134264C
41: 18CE3D2920E3BDEFA91C369E9DE57BF4
42: 50917AE58278E15A18A47B284D8027A3
43: BDA6F9DE33704302CE056412143B4F82
44: C287898C1451774675EB7A964C004E0D
45: 3BDE73E0D357319AB06D3675F1D3E28D
46: 30FF4326C89C0FFE4D31D2E92CC0BF9B
47: F69816F304ED892232F220F290320A8D
48: 1368153F1A54EFF8D61F93A2D6AF21E3
49: 06DD274894B6EDF3159A1403F47F09C7

Key Size: 28 bytes
 0: 7828B1997D3D050201DC6EE45C8521B5
 1: 0D77F896F9CEF16DAAFCF962C2257AAE
 2: 89C27B0623F5EECCA38BAE1AD86AE156
 3: 44EC09834052009CC3CD66E1BA11AF01
 4: F922BFDB03FB186A069C1E7B48222E3D
 5: 277F7971955D8984AAECF287C32B8211
 6: E77ED0144A3ED827B71453B91562FE25
 7: 1760EFD04477AE527BC37F72C8BBBCAE
 8: 26259425ACD58207AE328B3F1A217AC1
 9: 0876C4DC51D22657C4121E9067C2C3BA
10: 0214981592C9CEDD4D654F84AF1793A5
11: 3E11FA027BC4F15048D27B187062259A
12: 24E7D61BB21EA90B5282B43AAFB0DBDC
13: 688F56ECB45B7C242000653460F04A23
14: DFA587501A875ACDE8687A04AE404861
15: 4C21CC3FBB768CC9AF2242FA206FE406
16: 5CA0B03FA7751DEBBE70CB21AA61765A
17: 4879B3AC26270C422645B9CA29CAD8BB
18: 24F941E1B9AF84C18D03885EAACE16E3
19: 05E163A0150123C2664131A81B20AFC1
20: D606CAA85362E23598E5B8BD60C60506
21: 33BD0AE751019BB751C151AE47BD5811
22: 75DA523F5F793F90034144A3599DC5E6
23: CD4709B56521EA306F5AD95CCA878183
24: 6A4EC2EDDEBBBFEB62C1F13F7A59BF20
25: 2A36272DC4EFDFC03F4DCF049ED2ADFF
26: FD4F3904E8E37E7C31508E5829482965
27: BA64BAE1C2ABB8599A31B245DBAD1153
28: 757E0151783A50FC92AE55861DCD797D
29: 5E63BDA3217ECB544972CA14A9074DA5
30: E52F1195921767FA2410BA095EA5C328
31: 6D7E42D67E329D669299B5A590017E8D
32: 0516F6F7D99ADE5DC42E635BB5832E80
33: 57FB4E6B82ED2A3091248DCEF9C27F14
34: 25231D0E9B96534977D2F2AF93DD10AB
35: 847C4C524A586568D19EFA3ECA343F1C
36: 52448814064E0F33A4EA89368C2E1ACC
37: 461275466FAA7BC16ABAD9EC459BD67A
38: 16C8324A383A00DA06DBEC419B69C551
39: 5F26F7CF715FF2649DCC3C71EB6B92DF
40: 575363411FB07C067CD4357A1CD1D695
41: AB70F08BAB51C5F57139A107EE858A12
42: 887F62AE3D700EC5323EDA231C6B4C48
43: 7B9851B01DC9083293F3B226690A54F4
44: 36E03DF51C574E35EF2077DB7A49548E
45: E238A564246B163F97EDD733A235EDEB
46: 30679CE080915DC3BFA91D0DAFF5E82E
47: 7C2E8145D803D4FE18EE32995AAC16B0
48: 24D6F61ECC87206804885D33BFA7B2CA
49: 1F4F81751CB3FAFDC9F9C27E639F370B

Key Size: 40 bytes
 0: 31C3221C218E4CA1762B0DE77B964528
 1: 0B6E4BD937773597647FFE0A3859BB12
 2: 67A116E5F762619DE72F99AD1562A943
 3: B6A841663FB466ACAF89C8DA5BA080F0
 4: 0442708BF804642B9B1C69F5D905817E
 5: BC77391EAB530B96CA35319E510DB306
 6: AED37991A50AECB70C1B99137D5B38F2
 7: 8735F7AF0BF6C5C7E3C98021E83A31EE
 8: A614243B1B871D80BDCE4A23AD00F9FA
 9: 16AC67B139A92AD777871C990D3DA571
10: B1774A2A12A8CAB25D28A575B67CEF5D
11: 4C9B1A120BC6A33C62AF903FEEC3AF5F
12: 7B128F00480E497C5754EE333457EE5E
13: AB56D578229492B95ED309C0EC566658
14: 42FAF577855FEDB3446D40B4B6677445
15: 84E0C19B4A4512001F663E22D3184F0A
16: 8B01680D049F5A9421BA9BED100CC272
17: 2B1D70B92A5DF12CE0FA6A7AA43E4CEE
18: C7F61340D1B2321A1884E54D74576657
19: 153C07C56B32530866722C4DEAC86A50
20: 2EACBEFC4A29D1250EEAFD12A1D4AE77
21: FCCB40B0997E47512295066F1A0344DD
22: C149A543345E2A1B8249F71CB9F903A4
23: 3FD0688A8D0BE5F06F157C234C29BF9A
24: 6A3F813F396D77C7F4641ECC3E0BF3AA
25: E2888B9D2A6D819367F61C5792866A8F
26: 1A8A000F91AF4E600DDD88E098BD938B
27: 2283E758C04548EF8C37FA9F5700A7AD
28: 4FD6D8E1678D2B85520B96C038C582BF
29: D13C0B228F792EF88F09ED192C571029
30: 1A2A06B1987BE0DADA4B558AE5E6A128
31: 097B0460C47F1801986F5706A69EB01C
32: DD17BAC0737515C6386ECA6A6D6C02B6
33: 5989BD1D46FD6EC14D4C55D5D6D17F99
34: 431002E0224BD34B0B93988356C19E7C
35: 37DB7570296DCCE45ABDDE36EBE4731D
36: 4731DE78EEBAA1D02568EEEA2E04A2F5
37: 1F879753A7964AF44C84FD5765D8E080
38: 54F120726F68EA4B0501365CD2A84759
39: 366E43BB744C615999E896D01A0D1D0E
40: 18747BD79F1D0529D09CAC70F4D08948
41: 4F9854BAE0834A0C5FD12381225958F2
42: 7C14ADF94A0B61828996D902E4CCFF3E
43: 242F0E9CE96E4E208A9E0C5D76F8E698
44: 27EE179E2A9301B521B2C94ED3D36A77
45: 892C84A5E77E88A67F5F00F3597F4C04
46: FC7880D7860E90DE17E935700FC8C030
47: BC49373F775BF9CD6BDC22C87F71E192
48: 365646D0DE092AF42EC8F12A19840342
49: 62D0E9C210A20ECD2FF191AD3495DE6F


Cipher: khazad
Key Size: 16 bytes
 0: 9C4C292A989175FC
 1: F49E366AF89BD6B7
 2: 9E859C8F323666F9
 3: 349EC57A02451059
 4: 59E34CF03134A662
 5: 436C16BAB80E3E2D
 6: 81C35012B08A194C
 7: 056CCC9991C1F087
 8: 0A59F24C4715B303
 9: 3C2CFF98AE8500FD
10: 9136C3FCC332D974
11: FA3FA726E6BEBA65
12: DD84E4F9F39FB7EE
13: A3F397CC9FB771F5
14: E2D6ECC1F40A51C7
15: 6704A1A705163A02
16: BD820F5AF7DEEB04
17: E21E37CC122027FF
18: E319085D8E2C1F4F
19: 0DDFE55B199A49A9
20: B70F39CCCB2BA9A6
21: 3F2F25723AED2E29
22: 751FACD5F517AB2F
23: D32CE55FBF217CE9
24: 91393018EA847012
25: D50F1C54BABE7081
26: C73350FBC5B3A82B
27: E9A054F709FD5C57
28: 94BD5121B25746D4
29: EE19F88B28BEB4B7
30: CE6845FD13A3B78A
31: 566729D0183496BC
32: DC0E1D38CB5E03A8
33: 251AD2B2842C75E3
34: D344AC41190F3594
35: 579B956A36ADA3A8
36: 5F83D3AFEE9A6F25
37: 2D3FF8708A03C600
38: 32A732C7BEEBB693
39: F437276FAA05BB39
40: 58DDD4CD0281C5FD
41: ECC2C84BD8C0A4DC
42: BAB24C2CEFE23531
43: 5244BFA3E2821E7D
44: A4B273E960946B2C
45: 039376D02A8D6788
46: D3EB7074E3B05206
47: 89C18FFA26ED0836
48: 1F05A2D2D78927D9
49: 0133E1745856C44C


Cipher: camellia
Key Size: 16 bytes
 0: ED18D83F3153160C5A6D01AC3717515C
 1: 1012886CCDF3FFD25E588BA10D6CE363
 2: D25562F6943EBE3A7E0EF28D33CF091E
 3: C26FDC4539DD1E6D0330B5836AB24420
 4: E14A50CE727B74B8CEBEB284FEF3C810
 5: AABFD72D334F594344C617EF8E8F5741
 6: E8D941419ABE88060835E9BD375455BB
 7: ED863784E1590139A2CA50D77450300A
 8: 545FCF42030BD764724C3EF5C139B038
 9: 08C194E007FAA99997D855A759D10743
10: 3899D3731500C79D2945AFC2980B4C17
11: 2720FA4B402AB7F1B019AF6248702369
12: 3FF6C3C90AB4141DEE5FF30EA2047F73
13: BB5BAF7545AA774C7AA5A58568F96832
14: 66349C52709EDE0EE34AB6501B420C7C
15: E1E93D923504A5421BAEA5F1D61D4C9A
16: 3C07DFD64B2407BB7575A905F3F31E83
17: 0FC569AC89ED790F69BBD1E998700C97
18: 6B6F390AFA1052BD2E8DB0DC261E4D26
19: CBEA83ED55DA9DED95B87F2BBBEAC37D
20: CE005DECECB98F5937D5ED26FD83154E
21: 738301D76316EC4173F124A9C9D6577A
22: D00A1E40CFB5F2B8FD2C0714580FAD50
23: 7EBF497C78B72E646EB72A326F1D5C4B
24: 7E0023900F6000D00737242DA8F2E1B1
25: 0F7737E715BEF0DEA503E355394540A9
26: 15452DD70DEBF45BEF39782CDB8BB086
27: E7464917B3AF060BC763D8959DDF90C1
28: CBE4B90FF8C66672122D53585198773B
29: B7262E6CAA2C14B18EE374DF922CDB98
30: 01E695E3CD87A2FD4B9C49D08D032DAD
31: AA1686BA0B5C5688D0F370C6E2BFA43C
32: 9448BA348E8E904992C3F4233C226B22
33: A1DCD1CB810DFB46BDCE6FBE3A192560
34: 4345D200A309FA8C5A0CE9EC60EE506C
35: 54C7F64D9B411BF90B283ED62686D28F
36: E347E882EC2635081547612B1D9589D1
37: 36D44CC101B37BB6F6AF68C3FEA3A7B7
38: F38C2D5B921965D2AFFDBF4EC5BCEC19
39: F7ED6BF85782F0526301BD1CD1624E67
40: 7959C134BFC85CA176550EA689F81054
41: A8FC96504C437F0EFD0BDF6CCEF516D2
42: 6B88D1A06D7C8C74379FEFE2D6A7C895
43: 39C21AA165F4A71A161971D89CA5DC32
44: CC123C40071BF02D282DC83D2AC18226
45: 0780A63741AE47CD03FA99A74C320E33
46: DFB0831BA27AA0750701439603B8A805
47: 0C783CBA4ECD9EEE1F91838259831187
48: 1456624438B22555B08D59CA50D6E95D
49: D5F463D983A9A6FE9A0B47C245596D40

Key Size: 24 bytes
 0: 1D1DAF85EA5CAE19F5F5EA1DC61E5B83
 1: DDAC7FCF2C2F275C7041E7821AAC84A3
 2: 591091C3755816AAEB9170D5DF77A0B3
 3: C4BC965CDC20E6FC039F07DA2CD10BE3
 4: CD8DA54FC48524EDCFEF985C0C39C961
 5: 14FA12F39AC3D701A958765B4499FFAC
 6: 2BBEA5F3AA140CFFED9F1EB2BC969D56
 7: 5F73CA8BF641770D6833A43947D9A5C3
 8: 3E872D303B882284AB02393D43137450
 9: 01EF55D4CE182FA03216A83A5128F761
10: 915C2F5793692A6D118D865783317C58
11: 4368A442B61D6F12D5447F1CB8854714
12: 3477ECB27ECFF2D7108ED1297DE80F86
13: 89C875CB55C1CE80FF2D430921FADB05
14: C5AAFE7A4588D4D9039E4552B3FC9B02
15: BF1E7509405AB219B540BDD0D3DE7528
16: 7E5CC85B6563099B902638B7E0D09728
17: FF04D2350647F117F81DA037A9E11946
18: EA294A53395A20B391B11AB9F97262F3
19: 448C801307E9405F740623BA55A45639
20: 62032AE6EB01322233FB321B2D6A8C38
21: 79A54FFB9CA25AE315BA0E7B6E59EA99
22: EDE7E634C396926876A49DB3C0E261E1
23: E9DA5106B8BD33391C28407E9B3758BD
24: D8EAF9F744E060695AD1F55F85AF3D76
25: F1E61F0F467C0785B6053332129114EA
26: 3119CACB24B012F3B96EFAD3FB856AFB
27: 97753ACDAFD6224E5D289BF76673A73A
28: 8D5912FFFD628736C64B3DE01DF1E674
29: 8951CEDB758DF5EA5D2A78B2A08480EE
30: 3C0FC9DFD8CF79A5F9F75CC43B1A9247
31: 4C7047481FE0849EA0416BDC00A52321
32: 97034388AE8553570366EDFB9F6D618F
33: F16BCC0FB2B77CCBDC5EF7AB2233599D
34: 6D94D041196F43F0224B1DAC84165E7C
35: 313C6BA0AD767259860DCF8003F2F5A2
36: C5F835DCF63D1C40E56DBAC7ADCE7F3C
37: DAFAFF6BB46EA9280562E5DDFA793BA8
38: 5C8C0570B06C595E296DD4A9FB864FCE
39: 72B433F78D7CA638C2ADA09D99CFB769
40: B6D7A6C47339743E9739D35D0F08A25D
41: 6CFD73F9E9781FFCE53C69AD2EF11E03
42: B7F0BA994EF90642B80FDD798666D752
43: DD49766125316ED4F546B246A2CFA23A
44: 8ED53D6CEF3CFB9DB0147F02656EDA35
45: 95690401D61C84A013EC6D25CCAC5CD1
46: 7693648B4A6CA804B6F01AE67816746C
47: F08C5898CE7970C41A5F8C05882CAB8B
48: 91EC0EC1CF839B58009E6CAAB3FD67A0
49: 853DFA14A029EB8FB8D693B0A65306A1

Key Size: 32 bytes
 0: 5F77DC44E5E6701E8755C1FA176E2434
 1: 5C1F70FC144C66D82D8F21DD2A0BA54E
 2: A98317BC656475F83E83062A69A17EF6
 3: D5B8C0DB1095E65D49CEC82D78FD4C7E
 4: 37A537292409ABE5B922DD97EC0F6CA4
 5: C7FD40883DE6BBC6059327DA586AD96E
 6: F4D19C443A2195B66085DACA7EFFDADF
 7: 6F12FD74B4D25C9F2856CAA1BA32461E
 8: DFC00046F41BC27684321B980BF68F6E
 9: 4A8BECB6A8D57002FCC6FE08B6D31118
10: 859562FB3727E535BD4A914907822545
11: EBA65EA3BD622DC044CA5384E568C65F
12: 79C16A751FBE22340F30462600724324
13: 8F4FB71B5B3E0C1DB870B4BC81E995D0
14: 4B82E7E8D64D8EF9D78DA944B292CED9
15: D873F8D7125A63EBB04473F7331B1975
16: 2FA25AF9E8D5A4DC82CAD98505E5DA60
17: C80C24625096E6E9852A6F9EE12735BB
18: 10D4434CB795DC06E926CFA3B43D2368
19: 070795AEA2765A443213F9CA909DF6C4
20: 7184D2F5644306FB6DD55F1C90C111CA
21: F4FAEDF12FB40DE7CE7B08121A340557
22: 86CE014AA863FD3030A26E6F8C178673
23: 5A46BF2B3F14D5FEA884C3361EA87ED3
24: 456584515D983D17ED4F3AE944BFB2C4
25: E1E8F394691C2A9123023A8EE3FCBBEF
26: AC73E8BD1758850DEDAA3817B01E6353
27: 15AE5395CBC3371F81A6F5B05C52671F
28: F15AA72D34C4E0EEF8DDDDA90D9A9539
29: 3325E709043735898EA242E94D169112
30: 044AB447754DADD4E2709FEE08D5CEA2
31: E02DD5E86D32B3A6CC7F0016375AEC5F
32: 790278BD19E2860618E24DC69993F92B
33: F776D24FD90A43A78D000CFC1189E56A
34: A3EE4A3D121280750F7C70E55DD40FF4
35: 32928BBBF98DF4B9E107599DFB30364F
36: B3E9296B529118B656D27AFF0F4D1A55
37: 4668FD77100255C3406281EC813719AE
38: 16F9FF27B26F13300DB8DEE2EDD023AA
39: 9295F8435D688D12BE631A31B2531482
40: D86917DF41ED4342C0ABF10628DBD1B4
41: 1F5215B987C3F079769292E65D04B823
42: F68B98BD2F12AACEBE78666AA83CA7D0
43: 09BB635B67279F5A6B1D5C5D880A1357
44: AE4ABBCC1D35CD8C4C254111D5F27158
45: 5552B3E39DE67F759799A686222EE4EC
46: 1CA439434B9FD2F24561A32A0A2A79C5
47: 0E33BE7CE3B9A5CFF00A73BD27DFE9EF
48: 6B7056FDC97983173D6B3D5BFC9B09B8
49: DA293A4CB96FE3608CFFD89B927C9ED6
