#!perl -w

use strict;
use Test::More;

use Text::Xslate;

{
    package MyArray;
    use Any::Moose;

    use overload
        '@{}' => sub{ shift->items() },
        fallback => 1,
    ;

    has items => (
        is  => 'ro',
        isa => 'ArrayRef',

        default => sub { [] },
    );

    package MyKV;
    use overload
        '%{}' => sub{ shift->items() },
        fallback => 1,
    ;

    sub new {
        my($class, %args) = @_;
        return bless [ $args{items} || {} ], $class;
    }

    sub items { $_[0][0] }

}

my $tx = Text::Xslate->new(
    #verbose => 2,
);

my @set = (
    # enumerable
    ['<: $a.size() :>', { a => [] },        '0', 'for array'],
    ['<: $a.size() :>', { a => [0 .. 9] }, '10'],

    ['<: $h.size() :>', { h => {} },        '0', 'for hash'],
    ['<: $h.size() :>', { h => {a => 1, b => 2, c => 3} }, '3'],

    ['<: nil.size() :>', { }, '', 'nil.size() returns an empty string'],

    ['<: $a.join(",") :>', { a => [] },        ''  ],
    ['<: $a.join(",") :>', { a => [1, 2, 3] }, '1,2,3'],
    ['<: $a.join(",") :>', { a => ["foo","bar","baz"] }, 'foo,bar,baz'],

    ['<: $a.reverse()[0] :>', { a => [] },        ''  ],
    ['<: $a.reverse()[0] :>', { a => [1, 2, 3] }, '3'],
    ['<: $a.reverse()[0] :>', { a => ["foo","bar","baz"] }, 'baz'],

    ['<: $a.reverse().join(",") :>', { a => [] },        '', 'chained'],
    ['<: $a.reverse().join(",") :>', { a => [1, 2, 3] }, '3,2,1'],
    ['<: $a.reverse().join(",") :>', { a => ["foo","bar","baz"] }, 'baz,bar,foo'],

    ['<: $a.sort().join(",") :>', { a => [] },        '', 'sort'],
    ['<: $a.sort().join(",") :>', { a => ['b', 'c', 'a'] }, 'a,b,c'],
    ['<: $a.sort().join(",") :>', { a => ['a', 'b', 'c'] }, 'a,b,c'],

    # kv
    ['<: $h.keys().join(",") :>', { h => {} }, '', 'keys'],
    ['<: $h.keys().join(",") :>', { h => {a => 1, b => 2, c => 3} }, 'a,b,c'],

    ['<: $h.values().join(",") :>', { h => {} }, '', 'values'],
    ['<: $h.values().join(",") :>', { h => {a => 1, b => 2, c => 3} }, '1,2,3'],

    [<<'T', { h => {a => 1, b => 2, c => 3} }, <<'X', 'kv' ],
<:
    for $h.kv() -> $pair {
        print $pair.key, "=", $pair.value, "\n";
    }
-:>
T
a=1
b=2
c=3
X

    [<<'T', { h => {a => 1, b => 2, c => 3} }, <<'X', 'reversed kv (pairs as struct)' ],
<:
    for $h.reverse() -> $pair {
        print $pair.key, "=", $pair.value, "\n";
    }
-:>
T
c=3
b=2
a=1
X

    [<<'T', { h => {a => 1, b => 2, c => 3} }, <<'X', 'reversed kv (pairs as objects)' ],
<:
    for $h.reverse() -> $pair {
        print $pair.key(), "=", $pair.value(), "\n";
    }
-:>
T
c=3
b=2
a=1
X

    ['<: $o.size() :>', { o => MyArray->new() },                   '0', 'for object'],
    ['<: $o.size() :>', { o => MyArray->new(items => [0 .. 9]) }, '10'],

    ['<: $o.keys().join(" ")   :>', { o => MyKV->new(items => { foo => 42 }) }, 'foo'],
    ['<: $o.values().join(" ") :>', { o => MyKV->new(items => { foo => 42 }) }, '42'],
    [<<'T', { o => MyKV->new(items => {a => 1, b => 2, c => 3}) }, <<'X', 'object as kv' ],
<:
    for $o.kv() -> $pair {
        print $pair.key, "=", $pair.value, "\n";
    }
-:>
T
a=1
b=2
c=3
X

    # builtin filters
    ['<: $value | raw :>', { value => "<em>Xslate</em>" }, "<em>Xslate</em>", 'raw as a filter'],
    ['<: raw($value) :>',  { value => "<em>Xslate</em>" }, "<em>Xslate</em>", 'raw as a functiun'],

    ['<: $value | html :>', { value => "<Xslate>" }, "&lt;Xslate&gt;", 'html'],
    ['<: $value | dump :>', { value => "<Xslate>" }, qr/&lt;Xslate&gt;/, 'dump'],
    ['<: $value | dump | raw  :>', { value => "<Xslate>" }, qr/<Xslate>/, 'x | dump | raw'],

    ['<: $value | html | html :>', { value => "<Xslate>" }, "&lt;Xslate&gt;", 'x | html | html'],
    ['<: $value | html | raw  :>', { value => "<Xslate>" }, "&lt;Xslate&gt;", 'x | html | raw (-> html)'],
    ['<: $value | raw  | html :>', { value => "<Xslate>" }, "<Xslate>", 'x | raw | html (-> raw)'],

    ['<: html($value) == "&lt;Xslate&gt;" ? "true" : "false" :>',
        { value => '<Xslate>' }, 'true'],
    ['<: raw($value) == "&lt;Xslate&gt;" ? "true" : "false" :>',
        { value => '<Xslate>' }, 'false'],

    ['<: 1 ? raw($value) : html($value) :>',
        { value => '<Xslate>' }, '<Xslate>'],
    ['<: 1 ? html($value) : raw($value) :>',
        { value => '<Xslate>' }, '&lt;Xslate&gt;'],

    ['<: 0 ? raw($value) : html($value) :>',
        { value => '<Xslate>' }, '&lt;Xslate&gt;'],
    ['<: 0 ? html($value) : raw($value) :>',
        { value => '<Xslate>' }, '<Xslate>'],

    ['<: raw :>',
        { value => '<Xslate>' }, qr/\b CODE \b/xms, 'raw itself'],
    ['<: html :>',
        { value => '<Xslate>' }, qr/\b CODE \b/xms, 'html itself'],
);

foreach my $d(@set) {
    my($in, $vars, $expected, $msg) = @$d;
    if(ref $expected) {
        like $tx->render_string($in, $vars), $expected, $msg or diag $in;
    }
    else {
        is $tx->render_string($in, $vars), $expected, $msg or diag $in;
    }
}

done_testing;
