package Import::Into::As;
use strict;
use warnings;

our $VERSION = '0.000002';

use Import::Into;
use Carp qw/croak/;

sub import::into::as {
    my $class = shift;
    my ($target, $rename, @args) = @_;

    # need to bump level by 2, 1 for the call to _get_imported_syms,
    # another for the call to import::into
    my $bump_level = 1;

    my ($dest, $level);

    if (ref $target) {
        $dest = $target->{package} if exists $target->{package};
        if (exists $target->{level}) {
            $level = $target->{level};
            $dest  ||= caller($level);
            $target->{level} += $bump_level;
        }
    }
    elsif ($target =~ m/^\d+$/) {
        $level = $target;
        $dest  = caller($level);
        $target += $bump_level;
    }
    else {
        $dest = $target;
    }

    croak "unable to find destination package!"
        unless $dest;

    # Stash away any current subs with names that conflict iwth new imports
    # that should be renamed. Also remove them form the namespace so that we
    # know if a sub gets put into the name that we put it there.
    my %old_subs;
    for my $name (keys %$rename) {
        $old_subs{$name} = $dest->can($name);
        _purge_sub($dest, $name);
    }

    # Do the import
    $class->import::into($target, @args);

    # Make a copy of all the subs that were imported, keyed by their new names
    my %new_subs = map { $rename->{$_} => $dest->can($_) } keys %$rename;

    # Restore original subs, purge imported names that should not be kept.
    for my $name (keys %$rename) {
        my $sub = $old_subs{$name};

        if ($sub) {
            no strict 'refs';
            no warnings 'redefine';
            *{"$dest\::$name"} = $sub;
        }
        else {
            _purge_sub($dest, $name);
        }
    }

    # Put the new subs in place under their new names
    no strict 'refs';
    *{"$dest\::$_"} = $new_subs{$_} for keys %new_subs;
}

sub _purge_sub {
    my ($dest, $name) = @_;
    no strict 'refs';

    local *GLOBCLONE = *{"$dest\::$name"};
    my $stash = \%{"${dest}\::"};
    delete $stash->{$name};
    for my $slot (qw/HASH SCALAR ARRAY IO FORMAT/) {
        *{"$dest\::$name"} = *GLOBCLONE{$slot} if defined *GLOBCLONE{$slot};
    }
}

1;


__END__

=pod

=encoding UTF-8

=head1 NAME

Import::Into::As - Wrapper around Import::Into that lets you rename subs.

=head1 DESCRIPTION

This wrapper provides a new function C<import::into::as> which works just like
C<import::into()>, except that you can rename the subs being exported around.

This is a wrapper around L<Import::Into>, which is an awesome tool that lets
you import any module into any other module. The problem is that not all
modules you are importing from have the ability to rename exports. This is not
a problem if you are exporting from modules that use L<Sub::Exporter> or
L<Exporter::Declare>, but it is a problem if you want to rename a sub exported
from L<Exporter>.

B<Note:> If you are exporting from modules that allow you to rename the exports
in their own syntax then you should use that. This module will be slower than
using the exporters rename syntax when available.

=head1 SYNOPSYS

    use Import::Into::As;

    # Export 'foo' and 'baz' from Some::Exporter into Destination::Package.
    # Rename 'foo' to 'bar' in the process.
    Some::Exporter->import_into_as('Destination::Package', {foo => bar}, qw/foo baz/);

=head1 METHODS

=head2 $package->import::into::as($target, \%renames, @arguments);

=over 4

=item $package

Package to export from.

=item $target

This can be a target package, a call level, or a hashref, see
L<Import::Into/METHODS> for an expanded explanation of C<$target>.

=item \%renames

This must be a hashref where the keys are the names of a subs exported by the
package, and the values are the new names you want them to have.

    { 'export_name' => 'new_name', ... }

=item \@arguments

Import arguments for the package.

=back

=head1 LIMITATIONS

=over 4

=item Only subs can be renamed (this may change in the future)

It would not be difficult to add renaming support for other types, but there
has not been any need yet.

=back

=head1 SOURCE

The source code repository for Import-Into-As can be found at
F<http://github.com/exodist/Import-Into-As>.

=head1 MAINTAINERS

=over 4

=item Chad Granum E<lt>exodist@cpan.orgE<gt>

=back

=head1 AUTHORS

=over 4

=item Chad Granum E<lt>exodist@cpan.orgE<gt>

=back

=head1 COPYRIGHT

Copyright 2015 Chad Granum E<lt>exodist7@gmail.comE<gt>.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

See F<http://dev.perl.org/licenses/>

=cut
