use v6.c;

use Array::Agnostic;

class Array::Sparse:ver<0.0.2>:auth<cpan:ELIZABETH>
  does Array::Agnostic
{
    has %!sparse;
    has $.end = -1;

#--- Mandatory method required by Array::Agnostic ------------------------------
    method AT-POS(Int:D $pos) is raw {
        if %!sparse.EXISTS-KEY($pos) || $pos < $!end {  # $!end cannot change
            %!sparse.AT-KEY($pos)
        }
        else {                                          # $!end will change
            Proxy.new(
                FETCH => -> $ { %!sparse.AT-KEY($pos) },
                STORE => -> $, \value is raw {
                    self!find-end if $pos > $!end;
                    %!sparse.ASSIGN-KEY($pos, value);
                }
            )
        }
    }

    method EXISTS-POS(Int:D $pos) {
        %!sparse.EXISTS-KEY($pos)
    }

    method BIND-POS(Int:D $pos, \value) is raw {
        $!end = $pos if $pos > $!end;
        %!sparse.BIND-KEY($pos,value)
    }

    method DELETE-POS(Int:D $pos) {
        if %!sparse.EXISTS-KEY($pos) {
            if $pos == $!end {
                my \result = %!sparse.DELETE-KEY($pos);
                self!find-end;
                result
            }
            else {
                %!sparse.DELETE-KEY($pos);
            }
        }
        else {
            Nil
        }
    }

    method elems() { $!end + 1 }

#---- Optional methods for performance -----------------------------------------
    method ASSIGN-POS($pos, \value) {
        $!end = $pos if $pos > $!end;
        %!sparse.ASSIGN-KEY($pos,value)
    }

    method CLEAR() {
        %!sparse = ();
        $!end = -1;
    }

    method !find-end(--> Nil) {
        $!end = %!sparse.elems
          ?? %!sparse.keys.map( *.Int ).max
          !! -1;
    }
}

=begin pod

=head1 NAME

Array::Sparse - class for sparsely populated Arrays

=head1 SYNOPSIS

  use Array::Sparse;

  my @a is Array::Sparse;

=head1 DESCRIPTION

This module adds a C<is sparse> trait to C<Arrays>.  Its only use is to
provide a more memory-efficient storage for arrays that are B<very> big
(as in millions of potential elements) but with only a very limited of
elements actually given a value (maximum about 5 %).  Unless memory is
of the most importance, if you populate more than 5% of the keys, you will
be better of just using a normal array.

=head1 AUTHOR

Elizabeth Mattijsen <liz@wenzperl.nl>

Source can be located at: https://github.com/lizmat/Array-Sparse .
Comments and Pull Requests are welcome.

=head1 COPYRIGHT AND LICENSE

Copyright 2018 Elizabeth Mattijsen

This library is free software; you can redistribute it and/or modify it under the Artistic License 2.0.

=end pod

# vim: ft=perl6 expandtab sw=4
