package OpenInteract2::Action::News;

# $Id: News.pm,v 1.6 2003/07/01 05:36:03 lachoy Exp $

use strict;
use base qw( OpenInteract2::Action::CommonSearch
             OpenInteract2::Action::CommonDisplay
             OpenInteract2::Action::CommonUpdate
             OpenInteract2::Action::CommonAdd
             OpenInteract2::Action::CommonRemove );
use DateTime;
use DateTime::Format::Strptime qw( strptime );
use Log::Log4perl              qw( get_logger );
use OpenInteract2::Constants   qw( :log );
use OpenInteract2::Context     qw( CTX );
use OpenInteract2::Util;
use SPOPS::Secure              qw( SEC_LEVEL_WRITE );

$OpenInteract2::Action::News::VERSION = sprintf("%d.%02d", q$Revision: 1.6 $ =~ /(\d+)\.(\d+)/);

sub home {
    my ( $self ) = @_;
    return $self->generate_content(
                    { section_list => $self->_get_sections } );
}

sub latest {
    my ( $self ) = @_;
    my $request = CTX->request;
    my $num_items = $self->param( 'num_items' )
                    || $request->param( 'num_items' )
                    || $self->param( 'default_list_size' );

    # For cache...
    $self->param( num_items => $num_items );

    my %params = ( show_box  => $self->param( 'show_box' ),
                   num_items => $num_items );
    my $where = "active_on <= ? AND active = ? ";
    my @values = ( OpenInteract2::Util->now, 'yes' );
    my $news_class = CTX->lookup_object( 'news' );
    my $items = eval {
        $news_class->fetch_group({ where => $where,
                                   value => \@values,
                                   order => 'posted_on DESC',
                                   limit => $num_items })
    };
    $params{news_list} = $self->_massage_news_list( $items );
    return $self->generate_content( \%params );
}


# This overrides OI2::Action::CommonSearch...

sub search {
    my ( $self ) = @_;
    my $request = CTX->request;

    $self->param_from_request( qw( num_items option section ) );
    unless ( $self->param( 'date' ) ) {
        my %date    = ( day   => $request->param( 'news_day' ),
                        month => $request->param( 'news_month' ),
                        year  => $request->param( 'news_year' ) );
        if ( $date{month} and $date{year} ) {
            $self->param( date => join( '-', map { $date{ $_ } }
                                                   sort keys %date ) );
        }
    }

    my $now_string = OpenInteract2::Util->now;
    my %params = ( section => $self->param( 'section') );
    my $where = " active_on <= ? AND active = ? ";
    my @values = ( $now_string, 'yes' );

    $where .= ( $self->param( 'option' ) eq 'older' )
                ? ' AND ( expires_on < ? ) '
                : ' AND ( expires_on IS NULL OR expires_on > ? ) ';
    push @values, $now_string;

    if ( $self->param( 'section' ) ) {
        $where .= ' AND ( section = ? )';
        push @values, $self->param( 'section' );
    }

    if ( $self->param( 'date' ) ) {
        $where .= 'AND ( posted_on >= ? AND posted_on < ? )';
        my ( $dy, $mo, $yr ) = split /\-/, $self->param( 'date' );
        if ( $mo > 0 ) {
            my $search_date = strptime( $self->param( 'date' ),
                                        '%d-%m-%Y' );
            push @values, $search_date->clone
                                      ->subtract( days => 1 )
                                      ->strftime( '%Y-%m-%d' ),
                          $search_date->add( days => 1 )
                                      ->strftime( '%Y-%m-%d' );
        }
        else {
            my $begin_date = join( '-', $yr, $mo, 1 );
            push @values, $begin_date;
            my $end_date = (strptime( $begin_date ))->add_duration( months => 1 );
            push @values, $end_date->strftime( '%Y-%m-%d' );
        }
    }

    my $news_class = CTX->lookup_object( 'news' );
    my $items = eval {
        $news_class->fetch_group({ where        => $where,
                                   value        => \@values,
                                   column_group => 'listing',
                                   order        => 'posted_on DESC' })
    };
    $params{news_list} = $self->_massage_news_list( $items );
    $params{section_list} = $self->_get_sections;
    return $self->generate_content( \%params );
}

sub _massage_news_list {
    my ( $self, $news_list ) = @_;
    my $log = get_logger( LOG_APP );

    # Only grab the first 'chunk' of the news item, as split up by the
    # separator; also create information about the user who posted the
    # story

    my %posters = ();
    foreach my $news ( @{ $news_list } ) {
        if ( $news->{news_item} ) {
            ( $news->{tmp_content} ) = split '<!--BREAK-->', $news->{news_item};

            # If there is template content in the news item, process it

            if ( $news->{tmp_content} =~ m|\[\%| ) {
                $log->is_debug &&
                    $log->debug( "Processing template content ",
                                 "for news item [$news->{title}]" );
                $news->{tmp_content} =
                    $self->generate_content( {}, { text => $news->{tmp_content} } );
            }
        }

        # Now grab relevant user information, caching as we go (since
        # the same people tend to post news)

        unless ( $posters{ $news->{posted_by} } ) {
            my $user = eval { $news->posted_by_user() };
            my $poster_info = {};
            if ( $@ ) {
                $poster_info->{login_name} = 'admin';
                $poster_info->{user_id}    = undef;
            }
            else {
                $poster_info->{login_name} = $user->{login_name};
                $poster_info->{user_id}    = $user->id;
            }
            $posters{ $user->{user_id} } = $poster_info;
        }
        $news->{tmp_poster_info} = $posters{ $news->{posted_by} };
    }
    return $news_list;
}


sub _display_form_customize {
    my ( $self, $template_params ) = @_;
    $template_params->{section_list} = $self->_get_sections;
}

sub _display_add_customize {
    my ( $self, $template_params ) = @_;
    $template_params->{section_list} = $self->_get_sections;
}

sub _add_customize {
    my ( $self, $news, $save_options ) = @_;
    $self->_set_defaults( $news );
}

sub _update_customize {
    my ( $self, $news, $old_data, $save_options ) = @_;
    $self->_set_defaults( $news );
}

sub _set_defaults {
    my ( $self, $news ) = @_;

    $news->{active_on} ||= DateTime->now;
    $news->{posted_on} ||= DateTime->now;
    unless ( $news->{expires_on} ) {
        my $expire_days = $self->param( 'default_expire' );
        $news->{expires_on} = DateTime->now->add( days => $expire_days );
    }

    # substitute <p> for hard returns where needed

    $news->{news_item} =~ s/(\r\n\r\n)(?!(<p|<pre|<blockquote))/$1<p>/g;

    # If the image URL wasn't set to something real, clear it

    if ( $news->{image_url} =~ m|^http://\s*$| ) {
        $news->{image_url} = undef ;
    }

    # Set other defaults

    $news->{posted_by}   ||= CTX->request->auth_user->id;
    $news->{image_align} ||= 'left';
}

# TODO: Move this to OI2::Action::Notifyable or something
sub notify {
    my ( $self ) = @_;
    my $request = CTX->request;
    my @news_id  = $request->param( 'news_id' );
    my $email    = $request->param( 'email' );
    if ( ! $email or ! scalar @news_id ) {
        return '<h2 align="center">Error</h2>' .
               '<p>Error: Cannot notify anyone about an object when no ' .
               'ID/email is given.</p>';
    }
    my $news_class = CTX->lookup_object( 'news' );
    my @news_list = ();
    foreach my $nid ( @news_id ) {
        my $news = eval { $news_class->fetch( $nid ) };
        push @news_list, $news   if ( $news );
    }
    my $rv = $news_class->notify({ email   => $email,
                                   subject => 'News notification',
                                   object  => \@news_list,
                                   type    => 'news' });
    if ( $rv ) {
        return '<h2 align="center">Success!</h2>' .
               '<p>Notification sent properly!</p>';
    }
    return '<h2 align="center">Error</h2>' .
           '<p>Error sending email. Please check error logs!</p>';
}


# Fetch all the news items for display
sub show_summary {
    my ( $self ) = @_;
    my $log = get_logger( LOG_APP );

    my $sections = eval {
        CTX->lookup_object( 'news_section' )
           ->fetch_group({ order => 'section' })
    };
    if ( $@ ) {
        $log->error( "Failed to get news sections: $@" );
        $self->param_add( error_msg => "Failed to get news sections: $@" );
    }
    my $news = eval {
        CTX->lookup_object( 'news' )
           ->fetch_group({ order => 'posted_on DESC' })
    };
    if ( $@ ) {
        $log->error( "Failed to get news objects: $@" );
        $self->param_add( error_msg => "Failed to get news items: $@" );
    }
    return $self->generate_content(
                    { section_list => $sections, news_list => $news } );
}


sub edit_summary {
    my ( $self ) = @_;
    my $log = get_logger( LOG_APP );

    my $request = CTX->request;
    my $news_class = CTX->lookup_object( 'news' );
    my @do_edit = $request->param( 'do_edit' );
    my ( $success, $attempt );
NEWS:
    foreach my $news_id ( @do_edit ) {
        next unless ( $news_id );
        $attempt++;
        my $news = eval { $news_class->fetch( $news_id ) };
        if ( $@ ) {
            my $msg = "Failed to fetch news object [$news_id]: $@";
            $log->error( $msg );
            $self->param_add( error_msg => $msg );
            next NEWS;
        }
        $news->{title} = $request->param( "title_$news_id" );
        $news->{section} = $request->param( "section_$news_id" );
        $news->{active}  = $request->param( "active_$news_id" );
        eval { $news->save };
        if ( $@ ) {
            my $msg = "Failed to save news object [$news_id]: $@";
            $log->error( $msg );
            $self->param_add( error_msg => $msg );
        }
        else {
            $success++;
        }
    }
    $self->param_add(
        status_msg => "Attempted: $attempt; successful updates $success" );
    $self->clear_cache();
    return $self->execute({ task => 'home' });
}

# Get all sections, or add an error message to the action

sub _get_sections {
    my ( $self ) = @_;
    my $sections = eval {
        CTX->lookup_object( 'news_section' )
           ->fetch_group({ order => 'section' })
    };
    if ( $@ ) {
        my $msg = "Failed to retrieve news sections: $@";
        $self->param_add( error_msg => $msg );
    }
    $sections ||= [];
    return $sections;
}


1;
