package OpenInteract2::Action::SystemDoc;

# $Id: SystemDoc.pm,v 1.5 2003/06/08 19:55:31 lachoy Exp $

use strict;
use base qw( OpenInteract2::Action );
use Data::Dumper qw( Dumper );
use OpenInteract2::Constants qw( :log );
use OpenInteract2::Context   qw( CTX DEBUG LOG );
use Pod::POM;

$OpenInteract2::Action::SystemDoc::VERSION = sprintf("%d.%02d", q$Revision: 1.5 $ =~ /(\d+)\.(\d+)/);

my ( %POD_CACHE );

sub list {
    my ( $self ) = @_;
    return $self->generate_content(
                    {}, { name => 'system_doc::system_doc_menu' } );
}


sub package_list {
    my ( $self ) = @_;

    # Grab the repository for thiw website and fetch a list of all the
    # packages so we can and find the 'doc/titles' document in each, or
    # so we can read the files from 'doc/' directly

    my $parser   = Pod::POM->new;
    my $pkg_list = CTX->repository->fetch_all_packages();
    my %pkg_docs = ();
    foreach my $pkg ( @{ $pkg_list } ) {
        my $pkg_id = join( '-', $pkg->name, $pkg->version );
        my $pkg_dir = $pkg->directory;
        my $doc_files = $pkg->get_doc_files;
        my @doc_titles = ();
        foreach my $doc_file ( @{ $doc_files } ) {
            $doc_file =~ s/^$pkg_dir//;
            DEBUG && LOG( LDEBUG,  "Found doc [$doc_file] in [$pkg_id]" );
            my $content = $pkg->read_file( $doc_file );
            unless ( $content ) {
                DEBUG && LOG( LWARN, "Package failed to read content ",
                                     "for [$doc_file]" );
                next;
            }
            my $pom = $parser->parse_text( $content );
            my ( $title ); # = "Unknown doc from $pkg_id";
            foreach my $head1 ( $pom->head1 ) {
#                if ( $head1->title eq 'NAME' ) {
                    $title ||= $head1->content;
#                }
            }
            DEBUG && LOG( LDEBUG, "Doc [$doc_file] -> [$title]" );
            push @doc_titles, [ $doc_file, $title ];
        }
        $pkg_docs{ $pkg_id } = \@doc_titles;
    }
    return $self->generate_content(
                    { package_docs => \%pkg_docs },
                    { name => 'system_doc::package_doc_listing' } );
}

# TODO: Get SPOPS|OI2::Manual stuff in here

sub module_list {
    my ( $self ) = @_;

    # Now sort INC and chop up the files into packages

    my %this_inc = %INC;
    my @top = ();

    my $count = -1;
    my $curr_parent = undef;
    foreach my $full_pkg ( sort keys %this_inc ) {
        next unless ( $full_pkg =~ /\.pm$/ );
        my ( $first ) = split /\//, $full_pkg;
        if ( $first ne $curr_parent ) {
            $count++;
            DEBUG && LOG( LDEBUG, "First item != parent: ",
                                  "($first) / ($curr_parent)" );
            $curr_parent   = $first;
            $curr_parent   =~ s/\.pm$//;
            $top[ $count ] = [ $curr_parent, [] ];
        }
        DEBUG && LOG( LDEBUG, "Found package $full_pkg" );
        push @{ $top[ $count ]->[1] }, _colonify( $full_pkg );
    }
    DEBUG && LOG( LDEBUG, "# module parents found: ", scalar @top );
    return $self->generate_content(
                    { module_list => \@top },
                    { name => 'system_doc::module_listing' } );
}


sub _colonify {
    my ( $text ) = @_;
    $text =~ s|\.pm$||;
    $text =~ s|/|::|g;
    return $text;
}


sub _uncolonify {
    my ( $text, $is_pod ) = @_;
    $text =~ s|::|/|g;
    my $ext = ( $is_pod ) ? '.pod' : '.pm';
    return "$text$ext";
}


sub display {
    my ( $self ) = @_;
    my $request = CTX->request;
    my ( $pod_file, $html_file, $text_file, $title, $error );

    # If this is a package, display the doc

    my $package_spec = $request->param( 'package' );
    if ( $package_spec ) {
        my ( $package_name, $ver ) = split /\-/, $package_spec;
        my $doc = $request->param( 'doc' );
        my $repos = CTX->repository;
        if ( $doc =~ /\.(html|txt|pod)$/ ) {
            my $full_filename = $repos->find_file( $package_name, $doc );
            DEBUG && LOG( LDEBUG, "Found [$full_filename] in [$package_name]" );
            $pod_file  = $full_filename  if ( $doc =~ /\.pod$/ );
            $html_file = $full_filename  if ( $doc =~ /\.html$/ );
            $text_file = $full_filename  if ( $doc =~ /\.txt$/ );
            $title = "Display Document from Package: $package_name";
            $error = "Cannot find package documentation: $doc";
        }
    }
    else {
        my $module = $self->param( 'module' )
                     || $request->param( 'module' );

        # ewww! ick!
        # TODO: Can we programmatically use Pod::Perldoc to do this?
        $pod_file = $POD_CACHE{ $module } || `perldoc -l $module`;

        chomp $pod_file;
        DEBUG && LOG( LDEBUG, "Found [$pod_file] from [$module]" );
        unless ( $pod_file ) {
            $pod_file = $INC{ _uncolonify( $module ) };
            DEBUG && LOG( LDEBUG, "Found [$pod_file] from %INC" );
        }
        if ( -f $pod_file ) {
            $POD_CACHE{ $module } = $pod_file;
        }
        $title = "Display Module: $module";
        $error = "Cannot find module: $module";
    }

    my ( $content );

    if ( -f $pod_file ) {
        DEBUG && LOG( LDEBUG, "Trying to view pod in [$pod_file]" );
        my $parser = Pod::POM->new();
        my $pom = $parser->parse( $pod_file );
        unless ( $pom ) {
            return '<p>Error parsing POD: ' . $parser->error() . '</p>';
        }

        eval { require OpenInteract2::PodView };
        if ( $@ ) {
            return "<p>Error trying to load POD viewer: $@</p>";
        }
        $content = OpenInteract2::PodView->print( $pom );
        $content =~ s/^.*<BODY>//sm;
        $content =~ s|</BODY>.*$||sm;
    }

    elsif ( -f $html_file ) {
        eval { open( HTML, $html_file ) || die $! };
        if ( $@ ) {
            return "<p>Error opening HTML file: $@" ;
        }
        local $/ = undef;
        $content = <HTML>;
        close( HTML );
        $content =~ s/^.*<BODY>//sm;
        $content =~ s|</BODY>.*$||sm;
    }

    elsif ( -f $text_file ) {
        eval { open( TEXT, $text_file ) || die $! };
        if ( $@ ) {
            return "<p>Error opening TEXT file: $@";
        }
        local $/ = undef;
        $content = <TEXT>;
        close( TEXT );
        $content = qq(<pre class="systemDocText">$content</pre>);
    }

    else {
        return '<p>Error $error.</p>';
    }

    unless ( $content ) {
        return '<p>Filename found but no content in file.</p>';
    }
    return $self->generate_content(
                    { content => $content },
                    { name => 'system_doc::doc_display' } );
}

1;

__END__

=head1 NAME

OpenInteract2::Action::SystemDoc - Display system documentation in HTML format

=head1 SYNOPSIS

=head1 DESCRIPTION

Display documentation for the OpenInteract system, SPOPS modules, and any
other perl modules used.

=head1 METHODS

C<list()>

List the OpenInteract system documentation and all the modules used by
the system -- we display both the C<OpenInteract> modules and the
C<SPOPS> modules first.

B<package_list()>

B<module_list()>

B<display()>

Display a particular document or module, filtering through
L<Pod::POM|Pod::POM> using
L<OpenInteract2::PodView|OpenInteract2::PodView>.

Parameters:

=over 4

=item *

B<filename>: Full filename of document to extract POD from.

=item *

B<module>: Perl module to extract POD from; we match up the module to
a file using %INC

=back

=head1 TO DO

B<Get more meta information>

System documentation needs more meta information so we can better
display title and other information on the listing page.

=head1 BUGS

None known.

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters E<lt>chris@cwinters.comE<gt>
