package OpenInteract2::Action::NewUser;

# $Id: NewUser.pm,v 1.7 2003/06/06 05:21:14 lachoy Exp $

use strict;
use base qw( OpenInteract2::Action );
use OpenInteract2::Constants qw( :log );
use OpenInteract2::Context   qw( CTX DEBUG LOG );
use SPOPS::Secure qw( :level :scope );
use SPOPS::Utility;

$OpenInteract2::Action::NewUser::VERSION = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);

use constant REMOVAL_TIME => 60 * 60 * 24; # 1 day

sub display {
    my ( $self ) = @_;
    return $self->generate_content(
                    {}, { name => 'base_user::new_user_form' } );
}

sub add {
    my ( $self ) = @_;
    my $request = CTX->request;
    my $login = $request->param( 'requested_login' );
    my $email = $request->param( 'working_email' );

    $self->_validate_login_and_email( $login, $email );
    my ( $plain_pass, $crypted_pass ) = $self->_create_password;
    my $new_user = $self->_create_new_user( $login, $email, $crypted_pass );
    $self->_add_new_user_metadata( $new_user );
    $self->_send_user_email( $new_user, $plain_pass );

    # TODO: Create method in OI2::Session set_user_id() so we don't
    # have to know the 'user_id' key (??)

    $self->response->return_url( $self->create_url({ TASK => '' }) );
    if ( $self->param( 'autologin' ) eq 'yes' ) {
        $request->auth_user( $new_user );
        $request->auth_is_logged_in(1);
        $request->session->{user_id} = $new_user->id;
    }
    return $self->generate_content(
                    { email => $email },
                    { name => 'base_user::new_user_complete' } );
}

sub _validate_login_and_email {
    my ( $self, $login, $email ) = @_;
    unless ( $self->_validate_email( $email ) ) {
        my $msg = 'The email address you entered is invalid.';
        $self->param_add( error_msg => $msg );
        $self->param( requested_login => $login );
        die $self->execute({ task => 'display' });
    }

    unless ( $login ) {
        my $msg = 'You cannot create an account without entering a login!';
        $self->param_add( error_msg => $msg );
        $self->param( working_email => $email );
        die $self->execute({ task => 'display' });
    }

    my $user_class = CTX->lookup_object( 'user' );
    my $user = eval {
        $user_class->fetch_by_login_name( $login, { skip_security => 1,
                                                    return_single => 1 } )
    };
    if ( $@ ) {
        LOG( LERROR, "Error fetching dupecheck user: $@" );
    }
    if ( $user ) {
        my $msg = 'The username you requested is already in use.';
        $self->param_add( error_msg => $msg );
        $self->param( working_email => $email );
        die $self->execute({ task => 'display' });
    }
    return undef;
}

sub _validate_email {
    my ( $self, $email ) = @_;
    return undef unless ( $email );

    eval "require Email::Valid";
    unless ( $@ ) {
        DEBUG && LOG( LDEBUG, "Email::Valid loaded, using for validation" );
        return Email::Valid->address( $email );
    }

    DEBUG && LOG( LINFO, "Email::Valid NOT loaded, trying ",
                         "Mail::RFC822::Address" );
    eval "require Mail::RFC822::Address";
    if ( $@ ) {
        LOG( LERROR, "Email::Valid NOT loaded and Mail::RFC822::Address ",
                     "NOT loaded , cannot validate email. (You should ",
                     "not have been able to install this package...)" );
        my $msg = 'Insufficient modules to check email validity.';
        $self->param_add( error_msg => $msg );
        die $self->execute({ task => 'display' });
    }
    return Mail::RFC822::Address::valid( $email );
}

sub _create_password {
    my ( $self ) = @_;
    my $plain = SPOPS::Utility->generate_random_code( 12, 'mixed' );
    my $crypted = ( CTX->server_config->{login}{crypt_password} )
                    ? SPOPS::Utility->crypt_it( $plain ) : $plain;
    return ( $plain, $crypted );
}

sub _create_new_user {
    my ( $self, $login, $email, $password ) = @_;

    # Now, create an entry in the user table; note that we set the removal
    # date to now plus whatever REMOVAL_TIME is set to

    my $new_user = CTX->lookup_object( 'user' )->new;
    $new_user->{login_name} = $login;
    $new_user->{email}      = $email;
    $new_user->{password}   = $password;
    $new_user->{theme_id}   = CTX->server_config->{default_objects}{theme};
    $new_user->{removal_date} = OpenInteract2::Util->now(
                                        { time => time + REMOVAL_TIME });
    eval { $new_user->save };
    if ( $@ ) {
        LOG( LERROR, "Failure to create new user: $@" );
        $self->param_add( error_msg => "Failed to create your record: $@" );
        die $self->execute({ task => 'display' });
    }
    return $new_user;
}

sub _add_new_user_metadata {
    my ( $self, $new_user ) = @_;

    # Ensure that the user can read/write his/her own record!

    eval { $new_user->set_item_security(
                              { class     => ref $new_user,
                                object_id => $new_user->id,
                                scope     => SEC_SCOPE_USER,
                                scope_id  => $new_user->id,
                                level     => SEC_LEVEL_WRITE }) };

    # Log the failed security set, if it happens...

    if ( $@ ) {
        LOG( LERROR, "Failed to set security so that new user ",
                     "[$new_user->{login_name}] can see her record: $@" );
        my $msg = "Error setting security information on your record: $@";
        $self->param_add( error_msg => $msg );
        die $self->execute({ task => 'display' });
    }

    # ...otherwise, mark the user as the creator of his/her own record

    $new_user->log_action_enter( 'create',
                                 scalar( $new_user->id ),
                                 scalar( $new_user->id ) );
}

sub _send_new_user_email {
    my ( $self, $new_user, $plain_password ) = @_;
    my $request = CTX->request;
    my $server_name = $request->server_name;

    # If that worked ok, send the user an email with the password created

    my %email_params = ( login       => $new_user->{login_name},
                         password    => $plain_password,
                         server_name => $server_name );
    my $message = $self->generate_content(
                              \%email_params,
                              { name => 'base_user::new_user_email' } );
    eval { OpenInteract2::Util->send_email({
                message => $message,
                to      => $new_user->{email},
                subject => "Account information for $server_name" }) };
    if ( $@ ) {
        LOG( LERROR, "Cannot send email! $@" );
        $self->param_msg(
            error_msg => "Could not send email to you with generated " .
                         "password: $@" );
        die $self->execute({ task => 'display' });
    }
}

1;

__END__

=head1 NAME

OpenInteract2::Action::NewUser - Display form for and process new user requests

=head1 DESCRIPTION

This handler takes care of creating a new user record on request,
creating a temporary password for the new user and notifying the user
on how to login. It does some preliminary checks on the email address
to ensure it is at least valid. We also set a date on the temporary
account creation so a simple cron job can cleanup abandoned attempts.

=head1 METHODS

B<show>

Displays the form for creating a new account, plus any error messsages
that might occur when processing the request (in I<edit()>).

B<edit>

Creates the user account and notifies the user with the temporary
password as well as the fact that the account will be removed in 24
hours if he/she does not login.

B<Important>: This routine tries to validate the email address using
either L<Email::Valid|Email::Valid> or if that is not found,
L<Mail::RFC822::Address|Mail::RFC822::Address>. If neither of these
modules is found then the email address cannot be validated and the
user cannot register.

=head1 TO DO

Nothing known.

=head1 BUGS

None known

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters E<lt>chris@cwinters.comE<gt>
