package OpenInteract::Page::File;

# $Id: File.pm,v 1.8 2002/12/13 21:41:14 lachoy Exp $

use strict;
use File::Basename qw();
use File::Copy     qw( cp );
use File::Path     qw();
use File::Spec;

# Use this to mark the beginning and end of the "good" content in a
# page in the filesystem; this allows you to use an HTML editor to
# create the content and to save a full html page to the filesystem

my $BODY_DEMARCATION = '<!-- OI BODY -->';


# Read in the content from a file

sub load {
    my ( $class, $page ) = @_;
    my $R = OpenInteract::Request->instance;

    my $full_location = $class->_create_file_location( $page );
    unless ( -f $full_location ) {
        $R->scrib( 1, "File ($full_location) does not exist! Bailing on load()..." );
        return undef;
    }

    $R->DEBUG && $R->scrib( 1, "File ($full_location) exists. Trying to read file." );

    open( STATIC, $full_location ) || die "Cannot access file: $!";
    local $/ = undef;
    my $content = <STATIC>;
    close( STATIC );

    $R->DEBUG && $R->scrib( 1, "File read ok. Scanning for valid content then returning." );

    # Only use the information between the $BODY_DEMARCATION tags (if
    # they exist)

    $content =~ s/$BODY_DEMARCATION(.*)?$BODY_DEMARCATION/$1/;

    # If the page still has <body> tags, only use the information
    # between them

    $content =~ s|<body>(.*?)</body>|$1|i;

    return $content;
}

# Wrap this sucker in an eval {} -- if there's an error, the old file
# is still in place (even if that was nothing); if there's no error,
# everything is consistent

sub save {
    my ( $class, $page, $content ) = @_;

    return unless ( $content );

    my $R = OpenInteract::Request->instance;
    my $full_location = $class->_create_file_location( $page );
    $R->DEBUG && $R->scrib( 1, "Trying to save content to [$full_location]" );

    my $tmp_location = $full_location;
    if ( -f $full_location ) {
        $R->DEBUG && $R->scrib( 1, "File already exists; writing content to temp file." );
        $tmp_location = "$full_location.tmp";
        if ( -f $tmp_location ) {
            unlink( $tmp_location )
                    || die "Cannot remove old temp file: $!";
        }
    }

    # Ensure the directory where this will go exists

    $class->_create_location_path( $full_location );

    open( NEW, "> $tmp_location" ) || die "Cannot open temp file for writing: $!";

    $R->DEBUG && $R->scrib( 1, "Content isa: ", ref $content );

    if ( ! ref $content ) {
        print NEW $content;
    }

    elsif ( ref $content eq 'SCALAR' ) {
        print NEW $$content;
    }

    else {
        my ( $data );
        binmode $content;
        while ( read( $content, $data, 1024 ) ) {
            print NEW $data;
        }
    }

    close( NEW );
    $R->DEBUG && $R->scrib( 1, "Wrote content to file ok." );

    if ( $full_location ne $tmp_location ) {
        $R->DEBUG && $R->scrib( 1, "Trying to delete old content file and rename temp file." );
        unlink( $full_location )
                || die "Cannot remove old content file: $!";
        rename( $tmp_location, $full_location )
                || die "Cannot rename temp file to content file: $!";
        $R->DEBUG && $R->scrib( 1, "Old file removed, new file renamed ok." );
    }

    return $full_location;
}


sub rename_content {
    my ( $class, $page, $old_location ) = @_;
    my $R = OpenInteract::Request->instance;

    $R->DEBUG && $R->scrib( 1, "Trying to rename file from [$old_location] ",
                               "to [$page->{location}]" );
    my $full_old_location = $class->_create_file_location( $old_location );
    my $full_new_location = $class->_create_file_location( $page );

    # Ensure the directory where this will go exists

    $class->_create_location_path( $full_new_location );

    eval {
        cp( $full_old_location, $full_new_location )
                || die "Cannot copy [$full_old_location] to [$full_new_location]: $!";
        unlink( $full_old_location )
                || die "Cannot remove [$full_old_location]: $!";
    };
    die "Failed to rename file: $@" if ( $@ );
    $R->DEBUG && $R->scrib( 1, "Renaming of file from [$full_old_location] to ",
                               "[$full_new_location] ok. Now check object" );

    # Ensure the 'size' and 'mime_type' fields are set properly

    my ( $is_changed );
    unless ( $page->{size} ) {
        $page->{size} = (stat $full_new_location)[7];
        $is_changed++;
    }
    unless ( $page->{mime_type} ) {
        $page->{mime_type} = $page->mime_type_file( $full_new_location );
        $is_changed++;
    }

    unless ( $is_changed ) {
        $R->DEBUG && $R->scrib( 1, "Object size/MIME type not changed." );
        return 1;
    }

    $R->DEBUG && $R->scrib( 1, "Trying to set object size/MIME type" );
    eval { $page->save };
    if ( $@ ) {
        $R->scrib( 0, "File renamed ok, but size and mime_type not set: $@" );
    }
    else {
        $R->DEBUG && $R->scrib( 1, "Object size/MIME type set ok" );
    }
    return 1;
}


sub remove {
    my ( $class, $page ) = @_;
    my $full_location = $class->_create_file_location( $page );
    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 1, "Trying to delete content file ($full_location)" );
    return 1 unless ( -f $full_location );
    unlink( $full_location ) || die "Cannot remove stale content file: $!";
    $R->DEBUG && $R->scrib( 1, "File deleted ok" );
    return 1;
}


# $item can be either a $page object or a scalar with a location

sub _create_file_location {
    my ( $class, $item ) = @_;
    my $location = ( ref $item ) ? $item->{location} : $item;
    my $R = OpenInteract::Request->instance;
    my $full_location = File::Spec->catfile( $R->CONFIG->get_dir( 'html' ),
                                             $location );
    $R->DEBUG && $R->scrib( 1, "Trying to find filesystem location for [$full_location]" );
    return $full_location;
}


sub _create_location_path {
    my ( $class, $location ) = @_;
    my $R = OpenInteract::Request->instance;
    $R->DEBUG && $R->scrib( 1, "See if [$location] exists or needs created" );

    my $dirname = File::Basename::dirname( $location );
    if ( -d $dirname ) {
        $R->DEBUG && $R->scrib( 1, "Path [$dirname] [$location] already exists" );
        return 1;
    }

    eval { File::Path::mkpath( $dirname, undef, 0775 ) };
    unless ( $@ ) {
        $R->DEBUG && $R->scrib( 1, "Path [$dirname] for [$location] created ok" );
        return 1;
    }
    $R->scrib( 0, "Cannot create path [$dirname]: $@" );
    die $@;
}

1;
