package OpenInteract::Handler::SystemBoxes;

# $Id: SystemBoxes.pm,v 1.11 2002/01/23 17:23:14 lachoy Exp $

use strict;
use SPOPS::Secure qw( :level );

@OpenInteract::Handler::SystemBoxes::ISA     = qw();
$OpenInteract::Handler::SystemBoxes::VERSION = sprintf("%d.%02d", q$Revision: 1.11 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::SystemBoxes::author            = 'chris@cwinters.com';
$OpenInteract::Handler::SystemBoxes::default_method    = 'handler';
@OpenInteract::Handler::SystemBoxes::forbidden_methods = ();
%OpenInteract::Handler::SystemBoxes::security = (
 handler        => SEC_LEVEL_NONE,
 templates_used => SEC_LEVEL_READ
);

my @BOX_KEYS = qw( name weight title template package class method code );

my %USER_BOX_DEFAULTS = (
       template => 'user_info_box',
       package  => 'base_box',
       weight   => 1,
       title    => 'User Info',
       name     => 'user_info_box' );

my %ADMIN_BOX_DEFAULTS = (
       template => 'admin_tools_box',
       package  =>  'base_box',
       weight   => 10,
       title    => 'Admin Tools',
       name     => 'admin_tools_box' );

my %TEMPLATES_USED_BOX_DEFAULTS = (
       class    => 'OpenInteract::Handler::SystemBoxes',
       method   => 'templates_used',
       weight   => 11,
       title    => 'Templates Used',
       name     => 'templates_used_box' );

my %LOGIN_BOX_DEFAULTS = (
       template => 'login_box',
       package  => 'base_box',
       weight   => 1,
       title    => 'Login',
       name     => 'login_box' );


sub handler {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance;
    my $conf = $R->CONFIG;
    my $box_config = $conf->{box};

    # First deal with the user info box, and any other boxes that
    # depend on the user being logged in

    if ( $R->{auth}{logged_in} ) {
        $R->DEBUG && $R->scrib( 1, "Generating box information for user info box" );
        push @{ $R->{boxes} }, $class->_assign_params({
                                           action   => 'user_info_box',
                                           defaults => \%USER_BOX_DEFAULTS,
                                           params   => { user => $R->{auth}{user} } });

        # Now, if the user is a member of the site admin group, add the
        # 'Admin Tools' box and the 'Templates Used' box.

        my $site_admin_id = $conf->{default_objects}{site_admin_group};
        my $supergroup_id = $conf->{default_objects}{supergroup};
        foreach my $group ( @{ $R->{auth}{group} } ) {
            next unless ( $group->{group_id} =~ /^($site_admin_id|$supergroup_id)$/ );
            $R->DEBUG && $R->scrib( 1, "User is in site admin group:  generating info for ",
                                       "admin_tools_box and templates used box" );
            push @{ $R->{boxes} }, $class->_assign_params({
                                             action   => 'admin_tools_box',
                                             defaults => \%ADMIN_BOX_DEFAULTS });
            push @{ $R->{boxes} }, $class->_assign_params({
                                             action   => 'templates_used_box',
                                             defaults => \%TEMPLATES_USED_BOX_DEFAULTS });
            last;
        }
        return undef;
    }

    # Next do the non-login box, and any other anonymous stuff

    $R->DEBUG && $R->scrib( 1, "Generating information for login box" );
    push @{ $R->{boxes} }, $class->_assign_params({ 
                                         action => 'login_box',
                                         defaults => \%LOGIN_BOX_DEFAULTS });
    return undef;
}



sub _assign_params {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $action = $R->lookup_action( $p->{action}, { return       => 'info',
                                                    skip_default => 1 } );
    $action ||= {};
    my %params = ();
    foreach my $key ( @BOX_KEYS ) {
        $params{ $key } = $action->{ $key } || $p->{defaults}{ $key };
    }
    $params{params} = $p->{params};
    return \%params;
}


# Simple handler to put the list of templates used into a more
# manageable form -- note that we can handle template objects and
# 'package::name' strings in \@{ $R->{templates_used} }

sub templates_used {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance;
    my %tmpl_by_package = ();
    my %tmpl_used = ();
    foreach my $template ( @{ $R->{templates_used} } ) {
        my ( $package, $name, $template_id, $full_name );
        if ( ref $template ) {
            ( $package, $name ) = ( $template->{package}, $template->{name} );
            $template_id = $template->{template_id};
        }
        else {
            ( $package, $name ) = $R->site_template->parse_name( $template );
            if ( $package and ! $name ) {
                $name    = $package;
                $package = '(site)';
            }
            $full_name = $template;
        }
        next if ( $tmpl_used{ $package }{ $name } );
        $tmpl_used{ $package }{ $name }++;
        push @{ $tmpl_by_package{ $package } },
                            { name      => $name,
                              id        => $template_id,
                              full_name => $full_name };
    }
    return $R->template->handler( {}, { templates_used => \%tmpl_by_package },
                                  { name => 'base_box::templates_used_box' } );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::SystemBoxes -- Generate default boxes that appear on all pages

=head1 SYNOPSIS

 # Just define the 'system_boxes' key in your server.perl:

 {
   box => {
     handler => 'OpenInteract::Handler::Box',
     system_box_handler => 'OpenInteract::Handler::SystemBoxes',
     ...
   },
   ...
 }

 # In the main box handler (normally 'OpenInteract::Handler::Box' 
 # or '<MyWebsite>::Handler::Box')

 my $system_box_class = $R->CONFIG->{box}{system_box_handler};
 $system_box_class->handler();

=head1 DESCRIPTION

This handler defines the boxes that can appear on every
page. Currently, these include:

B<user_info> (if logged in)

Displays username, full name and any other information or links you
want your users to see about themselves on every page.

B<login_form> (if not logged in)

Displays a username/password form and a link to create a new account.

B<admin_tools> (if in 'site admin' group)

Displays links for site administrators.

B<templates_used> (if in 'site admin' group)

Displays a listing of the templates used for a request along with
links so you can quickly jump to one and edit it.

=head1 CONFIGURING BOXES

Since each box is a component and components are defined through the
Action Table, you can define information for each of these boxes in
the file C<conf/action.perl> for this package. (The information for
each should already be defined, so if you want to modify any of the
values there feel free.)

This handler also comes with default information where appropriate in
case the Action Table entries are not found. This ensures these boxes
are always seen.

=over 4

=item *

B<title>: Specify the title under which the box content will appear

=item *

B<weight>: Define the "gravity" of the box (between 1 - 10 -- 10
sinks, 1 floats). Tied weight scores are sorted by the box name.

=back

In addition, you need to specify how the box will be created. Choose
from one of the following items:

=over 4

=item 1

B<template>: Define which template will be used

B<package> Define the package in which the template is found.

=item 2

B<class>: Class to be used

B<method>: Method to be used in B<class>

=item 3

B<code>: Code reference to be executed which returns the box
content. (Doing this from the action table limits the utility of this
method, but it can still be useful.)

=back

=head1 BUGS 

None known.

=head1 TO DO

B<More?>

Any other system-wide boxes needed?

=head1 SEE ALSO

L<OpenInteract::Component>, L<OpenInteract::Handler::Box>

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
