package OpenInteract::Handler::Theme;

# $Id: Theme.pm,v 1.5 2002/01/02 02:43:55 lachoy Exp $

use strict;
use SPOPS::Secure qw( :all );

@OpenInteract::Handler::Theme::ISA     = qw( OpenInteract::Handler::GenericDispatcher SPOPS::Secure );
$OpenInteract::Handler::Theme::VERSION = sprintf("%d.%02d", q$Revision: 1.5 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::Theme::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Theme::default_method    = 'listing';
@OpenInteract::Handler::Theme::forbidden_methods = ();
%OpenInteract::Handler::Theme::security          = ( 
 listing => SEC_LEVEL_READ,
 show    => SEC_LEVEL_WRITE,
 edit    => SEC_LEVEL_WRITE,
 remove  => SEC_LEVEL_WRITE, 
);

use constant MAX_TEMP    => 10;
use constant MAIN_SCRIPT => '/Theme';

# List the available themes

sub listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $params = { main_script => MAIN_SCRIPT, 
                   error_msg => $p->{error_msg} };
    my $items = eval { $R->theme->fetch_group({ order => 'title' }) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 403 } );
        $items = [];
    }
    $R->DEBUG && $R->scrib( 1, "Found ", scalar @{ $items }, " themes to show." );
    $params->{theme_list} = $items;
    $R->{page}->{title} = 'OpenInteract Themes';
    return $R->template->handler( {}, $params, 
                                  { db      => 'theme_list', 
                                    package => 'base_theme' } );
}



sub show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $params = { main_script => MAIN_SCRIPT, 
                   temp_max    => MAX_TEMP,
                   error_msg   => $p->{error_msg} };
    my $theme_id = $R->apache->param( 'theme_id' );
    my $theme = $p->{theme};
    $theme  ||= eval { $R->theme->fetch( $theme_id ) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 404 } );
        $params->{error_msg} .= "Cannot fetch requested theme. See error log for details.";
    }
    $theme  ||= $R->theme->new;
    $theme->{parent} ||= 1   unless ( $theme->{theme_id} == 1 );
    $params->{theme} = $theme;
    
    # Get all the property objects associated with this theme (including
    # ones it inherits) and put them into a sorted list.
    
    my $properties = eval { $theme->discover_properties };
    my @property_list = map { $properties->{ $_ } } sort keys %{ $properties };
    
    # Grab all the other theme objects so we can choose a parent
    
    $params->{parent_list} = eval { $R->theme->fetch_group({ 
                                                order => 'title' }) };

    # Mark every property that's from a different theme than this one.

    foreach my $tp ( @property_list ) {
        if ( $tp->{theme_id} != $theme->{theme_id} ) {
            $tp->{tmp_different_theme} = 1;
        }
    }
    $params->{property_list} = \@property_list;
    
    $R->{page}->{title} = 'Edit a Theme';
    return $R->template->handler( {}, $params, 
                                  { db      => 'theme_form', 
                                    package => 'base_theme' } ); 
}


sub edit {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $apr = $R->apache;
    my $theme_id = $apr->param( 'theme_id' );
    $R->DEBUG && $R->scrib( 1, "Trying to modify ID <<$theme_id>>" );
    my $theme = eval { $R->theme->fetch( $theme_id ) };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 404 } );
        $p->{error_msg} .= "Cannot fetch requested theme. See error log for details.";
    }
    
    $theme ||= $R->theme->new;
    foreach my $field ( @{ $R->theme->field_list } ) {
        next if ( $field eq 'theme_id' );
        $theme->{ $field } = $apr->param( $field );
    }

    eval { $theme->save };
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 407 } );
        $p->{error_msg} .= 'Cannot save theme object! See error log for details. No properties modified. ';
        $p->{theme} = $theme;
        return $class->show( $p ); 
    }
  
    # Now work with the individual properties. First, set the theme_id
    # just in case we created a new one

    $R->DEBUG && $R->scrib( 1, "Theme saved ok. Moving onto properties." );
    $theme_id = $theme->{theme_id};

    # Get the existing list or properties

    my $properties = eval { $theme->discover_properties };

    # Now go through and see if any were changed

    my @errors = ();

PROPERTY:
    foreach my $tp ( values %{ $properties } ) {
        my $tp_id = $tp->{themeprop_id};
        my $status = $apr->param( "status_$tp_id" );
        my $changed = ( $status eq 'changed' );
        my $removed = ( $status eq 'removed' );
        next PROPERTY unless ( $changed or $removed );

        # Grab this property's theme ID for comparison later

        my $tp_theme_id = $tp->theme->{theme_id};
        if ( $changed ) {
            $tp->{value}       = $apr->param( "value_$tp_id" );
            $tp->{description} = $apr->param( "desc_$tp_id" );
            if ( $theme_id != $tp_theme_id ) {
                $R->DEBUG && $R->scrib( 1, "Property $tp->{prop} needs to be cloned/modified." );
                my $save_tp = $tp->clone({ theme_id => $theme_id });
                eval { $save_tp->save };
                if ( $@ ) {
                    push @errors, "Cannot add new version of property ($tp->{prop}): $@";
                }
            }
            else {
                $R->DEBUG && $R->scrib( 1, "Property $tp->{prop} needs to be edited in place." );
                eval { $tp->save };
                if ( $@ ) {
                    push @errors, "Cannot update property ($tp->{prop}): $@";
                }
            }
        }

        elsif ( $removed ) {
            $R->DEBUG && $R->scrib( 1, "Property $tp->{prop} needs to be removed." );
            if ( $theme_id != $tp_theme_id ) {
                push @errors, "Cannot remove property $tp->{prop} since it belongs to another theme.";
            }
            else {
                eval { $tp->remove };
                if ( $@ ) {
                    push @errors, "Cannot remove property ($tp->{prop}): $@";
                    $R->scrib( 0, "Cannot remove theme property ($tp->{prop}): $@" );
                }
            }
        }
    }
  
    foreach my $i ( 1 .. MAX_TEMP ) {   
        my $status = $apr->param( "status_temp$i" );   
        $R->DEBUG && $R->scrib( 2, "Status of temp value $i: $status" );
        next if ( $status eq 'unchanged' or $status eq 'removed' );
        my $new_tp = $R->themeprop->new;
        $new_tp->{theme_id}    = $theme_id;
        $new_tp->{prop}        = $apr->param( "prop_temp$i" );
        $new_tp->{value}       = $apr->param( "value_temp$i" );
        $new_tp->{description} = $apr->param( "desc_temp$i" );
        eval { $new_tp->save };
        if ( $@ ) {
            push @errors, "Cannot save new property ($new_tp->{prop}): $@";
        }
    }

    # Clear out the existing properties so that show() will refetch them
    # and reflect any changes we just made

    $theme->{tmp_properties} = undef;
    $p->{theme} = $theme;
    $p->{error_msg} = ( scalar @errors ) ? '<p>' . join( "\n<p>", @errors ) : undef;
    $R->{page}->{return_url} = '/Theme/';
    return $class->show( $p ); 
}


sub remove {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    if ( my $theme_id = $R->apache->param( 'theme_id' ) ) {
        my $theme = eval { $R->theme->fetch( $theme_id ) };
        if ( $@ or ! $theme ) {
            return $class->listing( { error_msg => 'Cannot remove theme -- object not created properly.' } );
        }
        eval { $theme->remove };
        if ( $@ ) {
            $p->{error_msg} = 'Cannot remove theme object! See error log for details...';
        }
    }
    return $class->listing( $p );
}


1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Theme - Create, edit and remove themes

=head1 DESCRIPTION

Available actions:

B<listing>

List all available themes.

B<show>

Display the details of a theme and list its properties in an editable
form. Denote the properties whose values are inherited.

B<edit>

Enter changes to theme details and make any necessary modifications to
theme properties. Note that if the theme you edit inherits from
another theme, any property changes will be added to the them you edit
rather than modified in the parent.

For instance:

 Properties before edit:

 Theme: Main

   Property: bgcolor   #ffffff

 Theme: Sub (inherits from main)

   Property: bgcolor not defined

Changing the 'bgcolor' property under the 'Sub' theme to '#000000'
will add the property to Sub so it will no longer inherit from 'Main':

 Properties after edit:

 Theme: Main

   Property: bgcolor   #ffffff

 Theme: Sub (inherits from main)

   Property: bgcolor   #000000

Also, note that removing a property from a theme will remove it from
that theme only -- inheriting themes will still have it defined.

B<remove>

Remove a theme from the system.

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
