package OpenInteract::Template::Text;

# $Id: Text.pm,v 1.6 2001/02/01 05:24:33 cwinters Exp $

use strict;
use Text::Template ();
use OpenInteract::Template;

@OpenInteract::Template::Text::ISA     = qw( OpenInteract::Template );
$OpenInteract::Template::Text::VERSION = sprintf("%d.%02d", q$Revision: 1.6 $ =~ /(\d+)\.(\d+)/);

use constant DEBUG => 0;

sub parse {
 my $class       = shift;
 my $tmpl_config = shift;
 my $tmpl_vars   = shift;
 my $p           = shift;
 my $R = OpenInteract::Request->instance;
 my $text = $p->{text};
 $text  ||= $class->read_file( $p->{file} )  if ( $p->{file} );
 if ( ! $text ) {
   $R->throw( { code => 201, type => 'template', system_msg => 'No text or filename to parse!' } );
   return '[[error processing directive]]';
 }

 # Get the default info for all templates
 $class->default_info( $tmpl_vars );

 $text = ( ref $text ) ? $$text : $text;
 my $tmpl = new Text::Template( type => 'STRING', source => $text, %{ $tmpl_config } );
 return $tmpl->fill_in( hash => $tmpl_vars );
}

1;

=pod

=head1 NAME

OpenInteract::Template::Text - Provide a template wrapper for the Text::Template module

=head1 SYNOPSIS

 my $tmpl_class = $R->template;
 my $html = $tmpl_class->parse( {}, { key => 'value' },
                                file => 'filename.tmpl' );

=head1 DESCRIPTION

Just feed this class the following information. The first two
parameters are always the same:

B<tmpl_config> (\%)

Configuration options for the template. Note that you can set
defaults for these at configuration time as well.

B<tmpl_vars> (\%)

The key/value pairs that will get plugged into the template

The last parameters depend on where you are getting the 
text to fill in. You can either pass:

 file => 'filename'

or:

 text => $scalar_with_text (or \$scalar_ref_with_text)

In either case, you will get a return value of the text 
with the template keys filled in. Since we are using 
Text::Template here, the keys may be fairly complex. 
Some people consider this a good thing :) 

=head1 METHODS

This class has only one method:

B<parse( \%, \%, % )>

Parses the text and returns the text with the values in the
second hashref plugged in.

=head1 TO DO

Need to see what performance for this module is like.


=head1 BUGS

B<Lack of testing>

This module has never really been tested, as Template Toolkit moves
closer to becoming the 'official' templating module for OpenInteract.

=head1 SEE ALSO

Text::Template:  http://www.plover.com/~mjd/perl/Template/

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut