#!perl -T

use 5.008;
use strict;
use warnings;

my $verbose = 0;

#### nothing to change below
use Data::Dump qw/pp/;
use utf8;

our $VERSION='0.22';

use File::Temp;
use File::Spec;

use Test::More;
use Test2::Plugin::UTF8; # rids of the Wide Character in TAP message!

use Data::Random::Structure;
use Data::Random::Structure::UTF8;

use Data::Roundtrip qw/:all/;

my $tmpdir = File::Temp::tempdir(CLEANUP=>0);

my $randomiser = Data::Random::Structure->new(
	max_depth => 50,
	max_elements => 200,
);
ok(defined $randomiser, 'Data::Random::Structure->new()'." called.") or BAIL_OUT;

# first test the sub fix_recursively()
my $td = fix_scalar('abc:xyz"\\aa"\'');
ok($td !~ /[:"'\\]/, 'fix_scalar()'." : called and it works for bad characters.") or BAIL_OUT("no it did not:\n$td\n");
# with utf8
$td = fix_scalar('αβγ:χψζ"\\αα"\'');
ok($td !~ /[:"'\\]/, 'fix_scalar()'." : called and it works for bad characters.") or BAIL_OUT("no it did not:\n$td\n");

my $randomiser_utf8 = Data::Random::Structure::UTF8->new(
	max_depth => 50,
	max_elements => 200,
);
ok(defined $randomiser, 'Data::Random::Structure->new()'." called.") or BAIL_OUT;

my ($outfile, $FH, $perl_data_structure, $json_string, $yaml_string, $perl_data_structure_from_file);

for my $trials (1..2){
  # NOTE: randomiser_utf8->generate() will output to STDERR
  #       I don't know how to generate string-UTF8
  #       it is harmless and it shows that all works ok
  for $perl_data_structure (
	$randomiser->generate(),
	$randomiser_utf8->generate()
  ){
	ok(defined $perl_data_structure, "random perl data structure created.");

	fix_recursively($perl_data_structure);

	# YAML does not like quoted strings to contain escaped quotes

	$json_string = Data::Roundtrip::perl2json($perl_data_structure);
	ok(defined($json_string), "json string created from perl data structure.");

	$yaml_string = Data::Roundtrip::perl2yaml($perl_data_structure);
	ok(defined($yaml_string), "yaml string created from perl data structure.");

	# write JSON to file
	$outfile = File::Spec->catdir($tmpdir, 'out.json');
	ok(open($FH, '>:utf8', $outfile), "open tmp file '${outfile}' for writing JSON string.") or BAIL_OUT;
	print $FH $json_string;
	close $FH;

	# read JSON from file
	$perl_data_structure_from_file = Data::Roundtrip::jsonfile2perl($outfile);
	ok(defined($perl_data_structure_from_file), "jsonfile2perl() : called and got defined result.");
	is_deeply($perl_data_structure, $perl_data_structure_from_file, "jsonfile2perl() : result agrees with what we saved before.");

	# write YAML to file
	$outfile = File::Spec->catdir($tmpdir, 'out.yaml');
	ok(open($FH, '>:utf8', $outfile), "open tmp file '${outfile}' for writing yaml string.") or BAIL_OUT;
	print $FH $yaml_string;
	close $FH;

	# read YAML from file
	$perl_data_structure_from_file = Data::Roundtrip::yamlfile2perl($outfile);
	ok(defined($perl_data_structure_from_file), "yamlfile2perl() : called and got defined result.") or BAIL_OUT("--begin YAML:\n${yaml_string}\n--end YAML string.\nfailed for above YAML string");
	is_deeply($perl_data_structure, $perl_data_structure_from_file, "yamlfile2perl() : result agrees with what we saved before.") or BAIL_OUT("--begin perl_data_structure:".perl2dump($perl_data_structure_from_file)."--end perl_data_structure_from_file\n\n--begin perl_data_structure_from_file:\n".perl2dump($perl_data_structure_from_file)."--end perl_data_structure_from_file.");

	if( $verbose == 0 ){
		my @tests = Test::More->builder->details;
		for my $test (@tests) {
			if(! $test->{ok} ){
				# a test failed, rerun with verbose on
				$verbose = 1;
				last;
			}
		}
	} else { last }
  }
}	

# cleanup only on success
File::Temp::cleanup;

done_testing;

# the randomiser produces strings with these characters
# which seem to confuse YAML::Load()
# so we are traversing the data structure and changing all
# scalars: array items, keys, values
# to remove these characters
sub fix_scalar {
	my $instr = $_[0];
	$instr =~ s/[:'"\\]+//g;
	return $instr;
}

sub fix_recursively {
	my $item = $_[0];
	my $r = ref $item;
	if( $r eq 'ARRAY' ){
		foreach my $at (@$item) {
			if( $r eq '' ){ $at = fix_scalar($at) }
			fix_recursively($at);
		}
	} elsif( $r eq 'HASH' ){
		foreach my $ak (keys %$item) {
			$item->{ fix_scalar($ak) } = fix_scalar($item->{$ak});
			delete $item->{$ak};
		}
		foreach (values %$item) {
			fix_recursively($_);
		}
	} elsif( $r eq '' ){
		$item = fix_scalar($item);
	} else { die perl2dump($item)."do not know this ref: '$r' for above input." }
}
