#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebDyne::Install.
#
#  WebDyne::Install is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: Constant.pm,v 1.21 2006/05/28 14:38:09 aspeer Exp $

#
#  WebMod::RTM Constants
#
package WebDyne::Install::Constant;


#  Pragma
#
use strict qw(vars);


#  Vars to use
#
use vars qw($VERSION $REVISION @ISA %EXPORT_TAGS @EXPORT_OK @EXPORT %Constant);


#  External modules
#
use Sys::Hostname qw(hostname);
use File::Find;
use File::Spec;
use IO::File;
use Cwd qw(realpath);
#use Getopt::Long qw(:config pass_through);
#not supported with 5.6.0, just use for the moment:
use Getopt::Long;


#  Other Constants
#
use WebDyne::Constant;


#  Need to null out some Fcntl functions under Win32
#
use Fcntl;
sub Fcntl::O_NONBLOCK { 0 }
sub Fcntl::O_NOCTTY { 0 }


#  Optionally can get httpd_bin from command line
#
GetOptions('HTTPD_BIN=s'=>\ (my $httpd_bin));


#  Local constants
#
our $Httpd_Bin=$httpd_bin || &httpd_bin();
our $Httpd_Config_hr=&httpd_config();
my  $ServerRoot;


#  Version information in a format suitable for CPAN etc. Must be
#  all on one line
#
$VERSION = eval { require WebDyne::Install::VERSION; do $INC{'WebDyne/Install/VERSION.pm'}};


#  Revision information, auto maintained by CVS
#
$REVISION=(qw$Revision: 1.21 $)[1];


#------------------------------------------------------------------------------

#  Name of user httpd runs under
#
my ($apache_uname, $apache_gname, $apache_uid, $apache_gid);
foreach my $name (qw(apache apache2 www wwwrun httpd httpd2)) {
    unless ($apache_uid) {
	if ($apache_uid=getpwnam($name)) { $apache_uname=$name }
    }
    unless ($apache_gid) {
	if ($apache_gid=getgrnam($name)) { $apache_gname=$name }
    }
}


#  Hash to hold some temp vars
#
my %temp;


#  Real deal
#
%Constant = (


    #  Binary
    #
    HTTPD_BIN				  =>	$Httpd_Bin,


    #  Config file templates and final names, delimiter if inserted into master httpd.conf
    #
    FILE_WEBDYNE_CONF_TEMPLATE		  =>	'webdyne.conf.inc',
    FILE_WEBDYNE_CONF			  =>	'webdyne.conf',
    FILE_APACHE_CONF_TEMPLATE		  =>	'apache.conf.inc',
    FILE_APACHE_CONF_DELIM		  =>	'#*WebDyne*',


    #  Where perl5 library dirs are sourced from
    #
    FILE_PERL5LIB			  =>	'perl5lib.pm',


    #  Get apache directory name
    #
    DIR_APACHE_CONF			  =>	&dir_apache_conf(),
    DIR_APACHE_MODULES			  =>	&dir_apache_modules(),


    #  Need apache uid and gid, as some dirs will be chowned to this
    #  at install time
    #
    APACHE_UNAME			  =>	$apache_uname,
    APACHE_GNAME			  =>	$apache_gname,
    APACHE_UID				  =>	$apache_uid,
    APACHE_GID				  =>	$apache_gid,


    #  Server config
    #
    %{$Httpd_Config_hr},


    #  WebDyne constants
    #
    %{WebDyne::Constant->hashref()},


   );


#  Windows overwrites some entries
#
my %win=(

    DIR_LOGROTATE_CONF			 =>	'/',
    FILE_LOGROTATE_CONF			 =>     File::Spec->devnull,

   );
($^O eq 'MSWin32') && (map {$Constant{$_}=$win{$_}} keys %win);


#  Get absolute binary file location
#
sub httpd_bin {


    #  If in Win32 need to get location of Apache from reg. Not much error checking
    #  because not fatal if reg key not found etc.
    #
    my ($path,$name_bin);
    if ($^O=~/MSWin[32|64]/) {
	require Win32::TieRegistry;
	my $reg_ix=tie(
	    my %reg, 'Win32::TieRegistry', 'HKEY_LOCAL_MACHINE\Software\Apache Group\Apache');
	my $version=(sort {$b cmp $a} keys %reg)[0];
	$path=($ServerRoot=$reg{$version}->{'ServerRoot'}) ||
	    # last resorts. blech
	    'C:\Apache;C:\Apache2;C:\Progra~1\Apache;C:\Progra~1\Apache2'.
	    'D:\Apache;D:\Apache2;D:\Progra~1\Apache;D:\Progra~1\Apache2'.
	    'E:\Apache;E:\Apache2;E:\Progra~1\Apache;E:\Progra~1\Apache2';
	$name_bin='Apache.exe';
    }
    else {

    	#  Add some hard coded paths as last resort options, will work if su'd to root
    	#  without getting root's path
    	#
	$path=$ENV{'PATH'} . '/sbin:/usr/sbin:/usr/local/bin:/usr/local/bin';
	$name_bin='httpd';
    }


    #  Find the httpd bin file
    #
    my $httpd_bin;
    my $wanted_cr=sub {
        # \d? allows for httpd2 to work if apache2
	($File::Find::name=~/[\\\/]\Q$name_bin\E\d?$/) && ($httpd_bin=$File::Find::name);
    };
    my @dir=grep { -d $_ } split(/:|;/, $path);
    find($wanted_cr, @dir);


    #  Return
    #
    return File::Spec->canonpath($httpd_bin);

}


sub httpd_config {


    #  Return if no Httpd_Bin, means apache binary not found
    #
    return unless $Httpd_Bin;


    #  Need to get httpd config as series of key/val pairs
    #
    my %config;
    my @httpd_config=qx($Httpd_Bin -V);


    #  Go through
    #
    foreach my $httpd_config (@httpd_config) {

	next unless ($httpd_config=~/\s*\-D\s*(.*)/);
	my ($key, $value)=split(/\=/, $1);
	$key=~s/\s+.*$//g;
	$value ||= '';
	$value=~s/^\"//;
	$value=~s/\"$//;
	$key=~/^HTTPD/ || ($key="HTTPD_${key}");
	$config{$key}=$value || 1;

    }


    #  In windows HTTPD_ROOT does not always match actual installed root path,
    #  so override if possible
    #
    my $httpd_dn=($config{'HTTPD_ROOT'}=$ServerRoot || $config{'HTTPD_ROOT'});


    #  Last resort - if does not exist use apache bin location as ref for
    #  HTTPD_ROOT
    #
    unless (-d $httpd_dn) {
	$config{'HTTPD_ROOT'}=do {
	    my $dn=File::Spec->catdir(
		(File::Spec->splitpath($Httpd_Bin))[1],
		File::Spec->updir());
            (-d $dn) && ($dn=realpath($dn));
            $dn;
        };
    }


    #  And return the config
    #
    return \%config;

}


sub dir_apache_conf {


    #  Get Apache config dir, ensure is absolute
    #
    my $apache_conf_dn=(File::Spec->splitpath(
	$Httpd_Config_hr->{'HTTPD_SERVER_CONFIG_FILE'}))[1];
    ($apache_conf_dn=~/^\//) || (
	$apache_conf_dn=File::Spec->catdir(
	    $Httpd_Config_hr->{'HTTPD_ROOT'},$apache_conf_dn));


    #  Check for ../conf.d path
    #
    foreach my $dn ('conf.d', File::Spec->catdir(File::Spec->updir(), 'conf.d')) {
      my $test_dn=File::Spec->canonpath(
	  File::Spec->catdir($apache_conf_dn, $dn));
      if (-d $test_dn) {
	$apache_conf_dn=realpath($test_dn);
	$Httpd_Config_hr->{'HTTPD_SERVER_CONFIG_SKIP'}=1;
      }
    }


    #  Return it
    #
    return $apache_conf_dn;

}


sub dir_apache_modules {


    #  Get Apache config dir, ensure is absolute
    #
    my @dn=(
	File::Spec->catdir(
	    $Httpd_Config_hr->{'HTTPD_ROOT'}, 'modules'),
	# last resorts, blech
	'/usr/lib/apache2',
	'/usr/lib/httpd/modules',
       );


    #  Check for modules
    #
    my $apache_modules_dn;
    foreach my $dn (@dn) {
	if (-d $dn) {
	    $apache_modules_dn=$dn;
	}
    }
    $apache_modules_dn ||= 'modules';


    #  Return it
    #
    return $apache_modules_dn;

}


#  Finalise and export vars
#
require Exporter;
require WebDyne::Constant;
@ISA=qw(Exporter WebDyne::Constant);
#  Local constants override globals
+__PACKAGE__->local_constant_load(\%Constant);
#  Command line options override all
GetOptions(\%Constant, map { "$_=s" } keys %Constant);
map { ${$_}=$Constant{$_} } keys %Constant;
@EXPORT=map { '$'.$_ } keys %Constant;
@EXPORT_OK=@EXPORT;
%EXPORT_TAGS=(all => [@EXPORT_OK]);
$_=\%Constant;
