package Bio::Roary::CommandLine::ProteinMuscleAlignmentFromNucleotides;

# ABSTRACT: Take in a multifasta file of nucleotides, convert to proteins and align with muscle


use Moose;
use Getopt::Long qw(GetOptionsFromArray);
use Bio::Roary::AnnotateGroups;
use Bio::Roary::External::Muscle;
use Bio::Roary::External::Revtrans;
use Bio::Roary::Output::GroupsMultifastaProtein;
use Bio::Roary::SortFasta;
extends 'Bio::Roary::CommandLine::Common';


has 'args'        => ( is => 'ro', isa => 'ArrayRef', required => 1 );
has 'script_name' => ( is => 'ro', isa => 'Str',      required => 1 );
has 'help'        => ( is => 'rw', isa => 'Bool',     default  => 0 );
has 'translation_table'  => ( is => 'rw', isa => 'Int',      default => 11 );

has 'nucleotide_fasta_files'  => ( is => 'rw', isa => 'ArrayRef' );
has '_error_message'          => ( is => 'rw', isa => 'Str' );

sub BUILD {
    my ($self) = @_;

    my ( $nucleotide_fasta_files, $help,$translation_table );

    GetOptionsFromArray(
        $self->args,
        't|translation_table=i'     => \$translation_table,
        'h|help'              => \$help,
    );

    $self->help($help) if(defined($help));
    $self->translation_table($translation_table)             if (defined($translation_table) );
    if ( @{ $self->args } == 0 ) {
        $self->_error_message("Error: You need to provide at least 1 FASTA file");
    }

    for my $filename ( @{ $self->args } ) {
        if ( !-e $filename ) {
            $self->_error_message("Error: Cant access file $filename");
            last;
        }
    }
    $self->nucleotide_fasta_files( $self->args );
}

sub run {
    my ($self) = @_;

    ( !$self->help ) or die $self->usage_text;
    if ( defined( $self->_error_message ) ) {
        print $self->_error_message . "\n";
        die $self->usage_text;
    }

    for my $fasta_file (@{$self->nucleotide_fasta_files})
    {
      
      my $sort_fasta_before = Bio::Roary::SortFasta->new(
         input_filename   => $fasta_file,
       );
      $sort_fasta_before->sort_fasta->replace_input_with_output_file;
      
      my $multifasta_protein_obj = Bio::Roary::Output::GroupsMultifastaProtein->new(
          nucleotide_fasta_file => $fasta_file,
          translation_table     => $self->translation_table
        );
      $multifasta_protein_obj->convert_nucleotide_to_protein();
      
      my $seg = Bio::Roary::External::Muscle->new(
        fasta_files => [$multifasta_protein_obj->output_filename],
        job_runner  => 'Local'
      );
      $seg->run();
      
      my $sort_fasta_after_muscle = Bio::Roary::SortFasta->new(
         input_filename   => $multifasta_protein_obj->output_filename. '.aln',
       );
      $sort_fasta_after_muscle->sort_fasta->replace_input_with_output_file;

      my $revtrans= Bio::Roary::External::Revtrans->new(
        nucleotide_filename => $fasta_file,
        protein_filename  => $multifasta_protein_obj->output_filename. '.aln',
        output_filename   => $fasta_file.'.aln',
        translation_table => $self->translation_table
      );
      $revtrans->run();
      
      my $sort_fasta_after_revtrans = Bio::Roary::SortFasta->new(
         input_filename   => $fasta_file.'.aln',
       );
      $sort_fasta_after_revtrans->sort_fasta->replace_input_with_output_file;
      
      unlink($multifasta_protein_obj->output_filename);
      unlink($multifasta_protein_obj->output_filename. '.aln');
    }
}

sub usage_text {
    my ($self) = @_;

    return <<USAGE;
    Usage: protein_muscle_alignment_from_nucleotides [options]
    Take in a multifasta file of nucleotides, convert to proteins and align with muscle
    
    # Transfer the annotation from the GFF files to the group file
    protein_muscle_alignment_from_nucleotides protein_fasta_1.faa protein_fasta_2.faa
    
    # Use a different translation table (default 11)
    protein_muscle_alignment_from_nucleotides -t 1 protein_fasta_1.faa protein_fasta_2.faa
    
    # This help message
    protein_muscle_alignment_from_nucleotides -h

USAGE
}

__PACKAGE__->meta->make_immutable;
no Moose;
1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Bio::Roary::CommandLine::ProteinMuscleAlignmentFromNucleotides - Take in a multifasta file of nucleotides, convert to proteins and align with muscle

=head1 VERSION

version 2.0.1

=head1 SYNOPSIS

Take in a multifasta file of nucleotides, convert to proteins and align with muscle, reverse translate back to nucleotides

=head1 AUTHOR

Andrew J. Page <ap13@sanger.ac.uk>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2013 by Wellcome Trust Sanger Institute.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut
