package OpenInteract2::Brick::Lookup;

use strict;
use base qw( OpenInteract2::Brick );
use OpenInteract2::Exception;

my %INLINED_SUBS = (
    'lookup-2.09.zip' => 'LOOKUP209ZIP',
);

sub get_name {
    return 'lookup';
}

sub get_resources {
    return (
        'lookup-2.09.zip' => [ 'pkg lookup-2.09.zip', 'no' ],
    );
}

sub load {
    my ( $self, $resource_name ) = @_;
    my $inline_sub_name = $INLINED_SUBS{ $resource_name };
    unless ( $inline_sub_name ) {
        OpenInteract2::Exception->throw(
            "Resource name '$resource_name' not found ",
            "in ", ref( $self ), "; cannot load content." );
    }
    return $self->$inline_sub_name();
}

OpenInteract2::Brick->register_factory_type( get_name() => __PACKAGE__ );

=pod

=head1 NAME

OpenInteract2::Brick::Lookup - Base-64 encoded OI2 package 'lookup-2.09.zip' shipped with distribution

=head1 SYNOPSIS

  oi2_manage create_website --website_dir=/path/to/site

=head1 DESCRIPTION

Are you sure you even need to be reading this? If you are just looking
to install a package just follow the instructions from the SYNOPSIS.

Still here? This class holds the Base64-encoded versions of package
file "lookup-2.09.zip" shipped with OpenInteract2. Once you decode them you
should store them as a ZIP file and then read them in with
Archive::Zip or some other utility.

A typical means to do this is:

 my $brick = OpenInteract2::Brick->new( 'lookup' );

 # there's only one resource in this brick...
 my ( $pkg_name ) = $brick->list_resources;
 my $pkg_info = $brick->load_resource( $pkg_name );
 my $pkg_file = OpenInteract2::Util->decode_base64_and_store(
     \$pkg_info->{content}
 );

 # $pkg_file now references a file on the filesystem that you can
 # manipulate as normal

These resources are associated with OpenInteract2 version 1.99_06.


=head2 Resources

You can grab resources individually using the names below and
C<load_resource()> and C<copy_resources_to()>, or you can copy all the
resources at once using C<copy_all_resources_to()> -- see
L<OpenInteract2::Brick> for details.

=over 4


=item B<lookup-2.09.zip>


=back

=head1 COPYRIGHT

Copyright (c) 2005 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS


Chris Winters E<lt>chris@cwinters.comE<gt>


=cut


sub LOOKUP209ZIP {
    return <<'SOMELONGSTRING';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SOMELONGSTRING
}

