/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { HttpStatusCode, TalerMerchantApi } from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  LocalNotificationBannerBulma,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { useState } from "preact/hooks";
import {
  FormErrors,
  FormProvider,
} from "../../../../components/form/FormProvider.js";
import { Input } from "../../../../components/form/Input.js";
import { useSessionContext } from "../../../../context/session.js";
import { undefinedIfEmpty } from "../../../../utils/table.js";

const TALER_SCREEN_ID = 37;

type Entity = TalerMerchantApi.CategoryCreateRequest;

interface Props {
  onCreated: () => void;
  onBack?: () => void;
}

export function CreatePage({ onCreated, onBack }: Props): VNode {
  const { i18n } = useTranslationContext();

  const [state, setState] = useState<Partial<Entity>>({ name_i18n: {} });

  const errors = undefinedIfEmpty<FormErrors<Entity>>({
    name: !state.name
      ? i18n.str`Required`
      : !/[a-zA-Z0-9]*/.test(state.name)
        ? i18n.str`Invalid. Please use only letters and numbers.`
        : undefined,
  });

  const hasErrors = errors !== undefined;
  const { state: session, lib } = useSessionContext();
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();

  const data = !!errors
    ? undefined
    : (state as TalerMerchantApi.CategoryCreateRequest);
  const create = safeFunctionHandler(
    lib.instance.addCategory.bind(lib.instance),
    !session.token || !data ? undefined : [session.token, data],
  );
  create.onSuccess = onCreated;
  create.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found`;
    }
  };

  return (
    <div>
      <LocalNotificationBannerBulma notification={notification} />
      <section class="section is-main-section">
        <div class="columns">
          <div class="column" />
          <div class="column is-four-fifths">
            <FormProvider
              object={state}
              valueHandler={setState}
              errors={errors}
            >
              <Input<Entity>
                name="name"
                label={i18n.str`Name`}
                tooltip={i18n.str`Category name`}
              />{" "}
              <div class="buttons is-right mt-5">
                {onBack && (
                  <button class="button" type="button" onClick={onBack}>
                    <i18n.Translate>Cancel</i18n.Translate>
                  </button>
                )}
                <ButtonBetterBulma
                  data-tooltip={
                    hasErrors
                      ? i18n.str`Please complete the marked fields`
                      : i18n.str`Confirm operation`
                  }
                  type="submit"
                  onClick={create}
                >
                  <i18n.Translate>Confirm</i18n.Translate>
                </ButtonBetterBulma>
              </div>
            </FormProvider>
          </div>
          <div class="column" />
        </div>
      </section>
    </div>
  );
}
