/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 3, or (at your option) any
   later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
   License for more details.

   You should have received a copy of the GNU General Public License
   along with TALER; see the file COPYING.  If not, see
   <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_lookup_order_charity.c
 * @brief Implementation for retrieving Donau charity_id and corresponding private key
 *        by Donau URL.
 * @author Bohdan Potuzhnyi
 * @author Vlada Svirsh
 */

#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "donau/donau_service.h"
#include "pg_lookup_order_charity.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TMH_PG_lookup_order_charity (
  void *cls,
  const char *instance_id,
  const char *donau_url,
  uint64_t *charity_id,
  struct DONAU_CharityPrivateKeyP *charity_priv,
  struct TALER_Amount *charity_max_per_year,
  struct TALER_Amount *charity_receipts_to_date,
  json_t **donau_keys_json,
  uint64_t *donau_instance_serial)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_string (donau_url),
    GNUNET_PQ_query_param_end
  };

  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_uint64 ("charity_id",
                                  charity_id),
    GNUNET_PQ_result_spec_auto_from_type ("merchant_priv",
                                          charity_priv),
    TALER_PQ_result_spec_json ("keys_json",
                               donau_keys_json),
    TALER_PQ_result_spec_amount_with_currency ("charity_max_per_year",
                                               charity_max_per_year),
    TALER_PQ_result_spec_amount_with_currency ("charity_receipts_to_date",
                                               charity_receipts_to_date),
    GNUNET_PQ_result_spec_uint64 ("donau_instances_serial",
                                  donau_instance_serial),
    GNUNET_PQ_result_spec_end
  };

  check_connection (pg);
  PREPARE (pg,
           "lookup_donau_charity",
           "SELECT"
           "  di.donau_instances_serial"
           " ,di.charity_id"
           " ,k.merchant_priv"
           " ,dk.keys_json"
           " ,di.charity_max_per_year"
           " ,di.charity_receipts_to_date"
           " FROM merchant_donau_instances di"
           " JOIN merchant_donau_keys dk"
           "   ON dk.donau_url = di.donau_url"
           " JOIN merchant_instances mi"
           "   ON mi.merchant_serial = di.merchant_instance_serial"
           " JOIN merchant_keys k"
           "   ON k.merchant_serial = mi.merchant_serial"
           " WHERE mi.merchant_id = $1"
           "   AND di.donau_url = $2;");

  return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                   "lookup_donau_charity",
                                                   params,
                                                   rs);
}