/*
 This file is part of GNU Taler
 (C) 2022-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { AbsoluteTime, KycRule } from "@gnu-taler/taler-util";
import { Time, useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { useServerMeasures } from "../hooks/server-info.js";
import { RulesInfo } from "./RulesInfo.js";

const TALER_SCREEN_ID = 120;

export function ShowDecisionLimitInfo({
  rules,
  since,
  until,
  startOpen,
  justification,
  fixed,
  measure,
}: {
  since: AbsoluteTime;
  until: AbsoluteTime;
  justification?: string;
  rules: KycRule[];
  startOpen?: boolean;
  fixed?: boolean;
  measure: string;
}): VNode {
  const { i18n } = useTranslationContext();
  const [opened, setOpened] = useState(startOpen ?? false);

  function Header() {
    return (
      <div
        data-fixed={!!fixed}
        class="p-4 relative bg-gray-200 flex justify-between data-[fixed=false]:cursor-pointer"
        onClick={() => {
          if (!fixed) {
            setOpened((o) => !o);
          }
        }}
      >
        <div class="flex min-w-0 gap-x-4">
          <div class="flex rounded-md  shadow-sm border-0 ring-1 ring-inset ring-gray-300 focus:ring-2 focus:ring-inset focus:ring-indigo-600">
            <div class="pointer-events-none bg-gray-300 inset-y-0 flex items-center px-3">
              <i18n.Translate>Since</i18n.Translate>
            </div>
            <div class="p-2  bg-gray-50 text-right rounded-md rounded-l-none data-[left=true]:text-left w-full py-1.5 pl-3 text-gray-900  placeholder:text-gray-50  sm:text-sm sm:leading-6">
              <Time format="dd/MM/yyyy HH:mm:ss" timestamp={since} />
            </div>
          </div>
        </div>
        <div class="flex shrink-0 items-center gap-x-4">
          <div class="flex rounded-md  shadow-sm border-0 ring-1 ring-inset ring-gray-300 focus:ring-2 focus:ring-inset focus:ring-indigo-600">
            <div class="pointer-events-none bg-gray-300 inset-y-0 flex items-center px-3">
              {AbsoluteTime.isExpired(until) ? (
                <i18n.Translate>Expired</i18n.Translate>
              ) : (
                <i18n.Translate>Expires</i18n.Translate>
              )}
            </div>
            <div class="p-2  bg-gray-50 text-right rounded-md rounded-l-none data-[left=true]:text-left w-full py-1.5 pl-3 text-gray-900  placeholder:text-gray-50  sm:text-sm sm:leading-6">
              <Time format="dd/MM/yyyy HH:mm:ss" timestamp={until} />
            </div>
          </div>
          {AbsoluteTime.isNever(until) ? undefined : (
            <div class="flex rounded-md  shadow-sm border-0 ring-1 ring-inset ring-gray-300 focus:ring-2 focus:ring-inset focus:ring-indigo-600">
              <div class="pointer-events-none p-2 bg-gray-300 inset-y-0 flex items-center ">
                <i18n.Translate>Successor measure</i18n.Translate>
              </div>
              <div class="p-2 bg-gray-50 rounded-md rounded-l-none data-[left=true]:text-left  text-gray-900  placeholder:text-gray-50  sm:text-sm sm:leading-6">
                {measure}
              </div>
            </div>
          )}
          {fixed ? (
            <Fragment />
          ) : (
            <div class="rounded-full bg-gray-50 p-2">
              <svg
                xmlns="http://www.w3.org/2000/svg"
                fill="none"
                viewBox="0 0 24 24"
                stroke-width="1.5"
                stroke="currentColor"
                class="size-6 w-6 h-6"
              >
                {opened ? (
                  <path
                    stroke-linecap="round"
                    stroke-linejoin="round"
                    d="m19.5 8.25-7.5 7.5-7.5-7.5"
                  />
                ) : (
                  <path
                    stroke-linecap="round"
                    stroke-linejoin="round"
                    d="m4.5 15.75 7.5-7.5 7.5 7.5"
                  />
                )}
              </svg>
            </div>
          )}
        </div>
      </div>
    );
  }

  if (!opened) {
    return (
      <div class="overflow-hidden border border-gray-800  rounded-xl">
        <Header />
      </div>
    );
  }

  return (
    <div class="overflow-hidden border border-gray-800 rounded-xl">
      <Header />
      <div class="p-4 grid gap-y-4">
        {!justification ? undefined : (
          <div class="">
            <label
              for="comment"
              class="block text-sm font-medium leading-6 text-gray-900"
            >
              <i18n.Translate>AML officer justification</i18n.Translate>
            </label>
            <div class="mt-2">
              <textarea
                rows={2}
                readOnly
                name="comment"
                id="comment"
                class="block w-full rounded-md border-0 py-1.5 text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 placeholder:text-gray-400 focus:ring-2 focus:ring-inset focus:ring-indigo-600 sm:text-sm sm:leading-6"
              >
                {justification}
              </textarea>
            </div>
          </div>
        )}

        <RulesInfo rules={rules} />
      </div>
    </div>
  );
}
