// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

namespace gcdriver {
using namespace ost;
using namespace std;

Session **iomap = NULL;

static Keydata::Define driver[] = {
	{"stack", "0"},
	{"events", "16"},
	{"priority", "1"},
	{"protocol", "isdn"},
	{"span", "e1"},
	{"spans", "1"},
        {"boardtype", "springware"},
        {"numberplan", "isdn"},
        {"numbertype", "enbloc"},
        {"l1protocol", "alaw"},
	{"analog", "na_an_io"},
	{"boards", "1"},		// analog card count
	{"accept", "240"},		// call accept timer stuff
	{"hangup", "120"},
        {NULL, NULL}};

Driver::Span Driver::span[] = {
        {"P_ISDN", "isdn", "t1", 23},
        {"P_ISDN", "isdn", "e1", 30},
        {NULL, NULL, NULL, 0}};

#ifdef	WIN32
#define	KEYS	"/bayonne/dialogic"
#else
#define	KEYS	"/bayonne/driver/dialogic"
#endif

Driver Driver::dialogic;

#define IdPos 0x9e
#define PosLenMax 64
#define NasLenMax 20
#define FromPcsToServer 0x77
#define FromServerToPcs 0x76

#define TYPE_INFO               0x83       

#define NAS_DIALED              0x83
#define NAS_CALLER              0x89
#define CONTEXT_DCB             0x85
#define CONTEXT_ASCII           0x95
#define ROUTING                 0x86
#define ORG_DEP                 0x8a
#define ORG_NUMBER              0x8b
#define ORG_OPERATOR            0x8c
#define ORG_COUNTRY             0x9a
#define MARKING                 0x88
#define LEN_MARKING             0x00

static int extractNAS(unsigned char *pt, int len, char *out)
{
        char nas[10 + 1];
        int temp, num;

	pt += 3;
        num = (unsigned int) *pt;
        temp = num/16;
        nas[1] = 0x30 + temp;
        temp = num - temp*16;
        nas[0] = 0x30 + temp;

        pt++;   /* 2 */
        num = (unsigned int) *pt;
        temp = num/16;
        nas[3] = 0x30 + temp;
        temp = num - temp*16;
        nas[2] = 0x30 + temp;

        pt++;   /* 3 */
        num = (unsigned int) *pt;
        temp = num/16;
        nas[5] = 0x30 + temp;
        temp = num - temp*16;
        nas[4] = 0x30 + temp;

        pt++;   /* 4 */
        num = (unsigned int) *pt;
        temp = num/16;
        nas[7] = 0x30 + temp;
        temp = num - temp*16;
        nas[6] = 0x30 + temp;

        pt++;   /* 5 */
        num = (unsigned int) *pt;
        temp = num/16;
        temp = num - temp*16;
        nas[8] = 0x30 + temp;
        nas[9] = 0;
        strcpy(out, nas);
        return(9);
}

static int extractBuffer(unsigned char *pt, char *out, int len)
{
        char buf[64 + 1];
        int i;

        for (i=0; i < len; i++) {
                pt++;
                buf[i] = *pt;
        }
        buf[len] = 0;
        memcpy(out, buf, len+1);
        return(0);
}

static int extractInfo(void *in, char *dialed, char *caller, char *context, char *route, int len)
{
        unsigned char *pt;
        unsigned int l, num;
	unsigned int ldialed, lcaller, lcontextAscii, lcontextDcb, lroute, lmarking;
        int temp;

	strcpy(dialed, "unknown");
	strcpy(caller, "unknown");
	strcpy(context, "none");
	strcpy(route, "none");
        pt = (unsigned char *)in;

        if (*pt != IdPos) 
	{
                return(0);
        }

	pt += 2;
	l = (unsigned int) *pt;
	pt += 3;
	if(*pt != TYPE_INFO)
		return 0;

	pt++;
	l = (unsigned int) *pt;
	pt++;
	if(*pt == NAS_DIALED)
	{
		pt++;
		ldialed = (unsigned int)*pt;
		if(ldialed == 7)
		{
			extractNAS(pt, 7, dialed);	
			if(l > 9)
			{
				++pt;
				if(*pt == NAS_CALLER)
				{
					++pt;
					lcaller = (unsigned int)*pt;
					if(lcaller == 7)
					{
						extractNAS(pt, 7, caller);
						pt += 8;
					}
				}
			}
			if(*pt == MARKING)
			{
				++pt;
				lmarking = (unsigned int)*pt;
				++pt;
			}

			if(*pt == CONTEXT_ASCII)
			{
				++pt;
				lcontextAscii = (unsigned int)*pt;
				++pt;

				extractBuffer(pt, context, lcontextAscii - 1);
				pt += lcontextAscii;
			}

			if(*pt == CONTEXT_DCB)
			{
				++pt;
				lcontextDcb = (unsigned int)*pt;
				extractBuffer(pt, context, lcontextDcb);
				pt += lcontextDcb;
			}

			if(*pt == ROUTING)
			{
				++pt;
				lroute = (unsigned int)*pt;
				extractBuffer(pt, route, lroute);
			}
			return 9;
		}
	}
	return 0;
}

static int extractSignaling(void *in, void *out, int len)
{
        int lpos = -1;
        unsigned int nie, lie;
        int n;
        unsigned char *pt;
        int i = 0;
        int foundPos = 0;

        pt = (unsigned char *)in;

        while (i < len) 
	{
                if (*pt == IdPos ) 
		{
			lpos = (unsigned int) *(pt + 2);
                        if ((i + lpos + 3) > len) 
				return -1;
			else
			{
				memcpy(out, pt, lpos + 3);
				return lpos + 3;
			}
		}
		else
		{
			++pt;
			lie = (unsigned int) *pt;
			if(i + lie <= len)
			{
				pt = pt + lie + 1;
				i = i + lie + 1;
			}
		}
	}
	return lpos;
}

Driver::Driver() : BayonneDriver(driver, KEYS, "dialogic", true),
Thread(atoi(getLast("priority")) + 1, atoi(getLast("stack")) * 1024) {
#ifdef	WIN32
	const char *env = Process::getEnv("DRIVERCONFIG");

	if(env)
		loadFile(env, "dialogic");
#else
	load("~bayonne/dialogic");
#endif

	devmap = new device_t[ts_count];
	iomap = new Session *[ts_count * 7 + 16];
	memset(devmap, 0, sizeof(device_t) * ts_count);
	memset(iomap, 0, sizeof(Session *) * (ts_count * 7 + 16));

        tpt[0].tp_type = IO_CONT;
        tpt[0].tp_termno = DX_LCOFF;
        tpt[0].tp_length = 3;
        tpt[0].tp_flags = TF_LCOFF | TF_10MS;
        tpt[1].tp_type = IO_EOT;
        tpt[1].tp_termno = DX_MAXSIL;
        tpt[1].tp_length = 0;
        tpt[1].tp_flags = TF_MAXSIL;

        dm3tpt[0].tp_type = IO_EOT;
        dm3tpt[0].tp_termno = DX_MAXSIL;
        dm3tpt[0].tp_length = 0;

        uio.u_seek = uio_seek;
        uio.u_read = uio_read;
        uio.u_write = uio_write;

        if(dx_setuio(uio))
		slog.critical("dialogic: cannot set uio");

	stopping = false;
}

DV_TPT *Driver::getPlayTPT()
{
	if(getBoardType() == DM3)
		return NULL;
	else
		return tpt;
}

DV_TPT *Driver::getRecordTPT()
{
        if(getBoardType() == DM3)
                return dm3tpt;
        else
                return tpt;
}

unsigned Driver::getSpanCount(void)
{
        const char *cp = getLast("spans");
        if(!cp)
                cp = getLast("count");

        if(cp)
                return atoi(cp);
        return 1;
}

const char *Driver::getSpanProtocol(void)
{
        const char *pro = getLast("protocol");
        const char *sp = getLast("span");

        Span *s = &span[0];

        while(s->gcname)
        {
                if(!stricmp(pro, s->gcname) || !stricmp(pro, s->protocol))
                        if(!stricmp(sp, s->span))
                                return s->gcname;
                ++s;
        }
        if(!strnicmp(pro, "P_", 2))
                pro += 2;
        return pro;
}

unsigned Driver::getSpanPorts(void)
{
        const char *pro = getLast("protocol");
        const char *sp = getLast("span");

        Span *s = &span[0];

        while(s->gcname)
        {
                if(!stricmp(pro, s->gcname) || !stricmp(pro, s->protocol))
                        if(!stricmp(sp, s->span))
                                return s->ports;
                ++s;
        }

        if(!stricmp(sp, "e1"))
                return 30;
        return 24;
}

boardtype_t Driver::getBoardType(void)
{
        const char *boardtype = getLast("boardtype");

        if(!stricmp(boardtype, "springware"))
                return SPRINGWARE;
        else if (!stricmp(boardtype, "dm3"))
                return DM3;
        else
                return SPRINGWARE;
}

void Driver::startDriver(void)
{
	GC_CCLIB_STATUSALL cclibstatus;
	GC_INFO gcinfo;
	GC_PARM gcparm;
	BayonneSpan *s;
	int queue, mode, i, gc_err, cclibid;
	long cclib_err;
	const char *cp = getLast("queue");
	unsigned scount = getSpanCount();
	unsigned pcount = getSpanPorts();
	unsigned span = 0, port;
	unsigned card = 0, vox = 0;
	unsigned bcount = 0, board = 0;
	timeslot_t ts = ts_used, tsc;
	boardtype_t boardtype = getBoardType();
	char *msg;
	Session *session;

	timeslot = ts_used;

	if(cp)
	{
		queue = atoi(cp);
		sr_setparm(SRL_DEVICE, SR_QSIZEID, &queue);
		slog.debug("dialogic: queue depth set %d", queue);
	}

#ifndef	SIGNAL
	slog.debug("dialogic: setting polled mode");
	mode = SR_POLLMODE;
	if(sr_setparm(SRL_DEVICE, SR_MODEID, &mode) < 0)
	{
		slog.critical("dialogic: unable to set polled mode");
		return;
	}	
#endif

	if(gc_Start(NULL) != GC_SUCCESS)
	{
		gc_ErrorInfo(&gcinfo);
		slog.critical("dialogic: failed; name=%s, error=%s",
			gcinfo.ccLibName, gcinfo.ccMsg);
		return;
	}

	if(gc_CCLibStatusEx("GC_ALL_LIB", &cclibstatus) != GC_SUCCESS)
	{
		slog.critical("dialogic: unable to get status");
		return;
	}

	for(i = 0; i < GC_TOTAL_CCLIBS; ++i)
	{
		cp = cclibstatus.cclib_state[i].name;
		switch(cclibstatus.cclib_state[i].state)
		{
		case GC_CCLIB_CONFIGURED:
			slog.debug("dialogic: %s: configured", cp);
			break;
		case GC_CCLIB_AVAILABLE:
			slog.debug("dialogic: %s: available", cp);
			break;
		case GC_CCLIB_FAILED:
			slog.debug("dialogic: %s: failed", cp);
			break;
		default:
			slog.debug("dialogic: %s: unknown", cp);
		}
	}	

	Thread::start();

	while(span < scount && ts < ts_count)
	{
	  s = new BayonneSpan(this, pcount);
	  for(port = 0; port < pcount; ++port)
	  {
		if(!port)
		{
			++card;
			vox = 0;
		}
		if(vox == 4)
		{
			++card;
			vox = 1;
		}
		else
			++vox;
		
		if(ts >= ts_count)
		{
			slog.warn("dialogic: exhausted timeslots");
			break;								
		}

		devmap[ts].iface = IF_SPAN;
		devmap[ts].tsdev = devmap[ts].chdev = devmap[ts].faxdev = -1;
		devmap[ts].btype = boardtype;
		sprintf(devmap[ts].voicedev, "dxxxB%dC%d", card, vox);

		cp = getLast("protocol");
		if(cp && !stricmp(cp, "isdn"))
			devmap[ts].iface = IF_ISDN;
		
		if(boardtype == DM3)
			sprintf(devmap[ts].devname, ":N_dtiB%dT%d:%s:V_%s",
                		span, port + 1, getSpanProtocol(), devmap[ts].voicedev);
		else
                        sprintf(devmap[ts].devname, ":N_dtiB%dT%d:%s",
                                span, port + 1, getSpanProtocol());

		// if span card port fails, we skip to next span;
		// hmm, maybe we have analog cards?

		if(gc_OpenEx(&devmap[ts].linedev, devmap[ts].devname, 
		   EV_SYNC, (void *)&devmap[ts]) != GC_SUCCESS)
		{
			if(!span && !port)
			{
				span = scount;
				break;
			}
			slog.notice("dialogic: skipping remainder of span; span=%d, port=%d", span, port);
			break;
		}

		if(boardtype == DM3)
                {
			gc_GetNetworkH(devmap[ts].linedev, &devmap[ts].tsdev);
			gc_GetVoiceH(devmap[ts].linedev, &devmap[ts].chdev);
                }			
		else
		{
			gc_GetNetworkH(devmap[ts].linedev, &devmap[ts].tsdev);
			devmap[ts].chdev = dx_open(devmap[ts].voicedev, 0);
			if(nr_scroute(devmap[ts].chdev, SC_VOX, devmap[ts].tsdev, SC_DTI, SC_FULLDUP) == -1)
				slog.error("dialogic: couldnt connect voice resource for port %d on span %d", port, span);
		}

		if(devmap[ts].chdev < 0)
			continue;

		gcparm.longvalue = 100;
		if (gc_SetParm (devmap[ts].linedev, RECEIVE_INFO_BUF, gcparm) != GC_SUCCESS)
		{
                        gc_ErrorValue(&gc_err, &cclibid, &cclib_err);
		        gc_ResultMsg(cclibid, cclib_err, &msg);
			slog.error("dialogic: setparm failed; span=%d, port=%d", span, port);
		}

		session = new Session(s, ts, &devmap[ts]);
		session->start();
		Thread::yield();
		++ts;

          }	
	  ++span;
	}		

	cp = getLast("accept");
	if(cp)
		pickup_timer = atoi(cp);

	cp = getLast("hangup");
	if(cp)
		hangup_timer = atoi(cp);

	cp = getLast("boards");
	if(!cp)
		cp = getLast("cards");
	if(!cp)
		cp = getLast("count");
	if(cp)
		bcount = atoi(cp);

	cp = getLast("analog");
	if(!strnicmp(cp, "P_", 2))
		cp += 2;

	if(!strnicmp(cp, "pdk_", 4))
		cp += 4; 

	while(board < bcount && ts < ts_count)
	{
	  vox = 1;
	  while(vox <= 4 && ts < ts_count)
	  {
                devmap[ts].tsdev = devmap[ts].chdev = -1;
		devmap[ts].btype = boardtype;
		devmap[ts].iface = IF_PSTN;	
		sprintf(devmap[ts].voicedev, "dxxxB%dC%d", board + 1, vox);
		sprintf(devmap[ts].devname, ":P_pdk_%s:V_%s", cp, devmap[ts].voicedev);
		if(gc_OpenEx(&devmap[ts].linedev, devmap[ts].devname,
			EV_SYNC, (void *)&devmap[ts]) != GC_SUCCESS)
		{
			if(ts == ts_used)
			{
				gc_ErrorInfo(&gcinfo);
				slog.error("dialogic: %s, lib=%s", 
					gcinfo.gcMsg, gcinfo.ccLibName);
			}
			board = bcount;
			vox = 5;
			break;
		}
		gc_GetVoiceH(devmap[ts].linedev, &devmap[ts].chdev);

		session = new Session(NULL, ts, &devmap[ts]);
		session->start();
		Thread::yield();
		++ts;
		++vox;
	  }
 	  ++board;
        }		

	if(ts > timeslot)
	{
		count = ts - timeslot;
		BayonneDriver::startDriver();
	}
}

void Driver::stopDriver(void)
{
	if(running)
	{
		stopping = true;
		BayonneDriver::stopDriver();
	}
	terminate();
}

void Driver::run(void)
{
	METAEVENT metaevent;
	int evdev, evtype;
	LINEDEV linedev;
	Session *session;
	Event event;
	DX_CST *cst;
        DV_DIGIT *digbuf;
	IE_BLK ie_blk;
	unsigned diglen, digcnt;
        unsigned long tmask;
	char buf[65];
	char caller[21];
	char context[21];
	char dialed[21];
	char route[21];
	int len;
	
	setCancel(cancelImmediate);
	oink.post();

	for(;;)
	{
		if(sr_waitevt(-1) == -1)
		{
			Thread::yield();
			continue;
		}

                if(gc_GetMetaEvent(&metaevent) != GC_SUCCESS)
                        continue;

		if(stopping)
			continue;

		evtype = metaevent.evttype;
		evdev = metaevent.evtdev;
		linedev = metaevent.linedev;

		slog.debug("dialogic: event %04x", evtype);
		
		if(metaevent.flags & GCME_GC_EVENT)
			session = iomap[linedev];	
		else
			session = iomap[evdev];	

		if(!session)
		{
			slog.error("dialogic: no device for event %04x", evtype);
			continue;
		}

		if(session->crn != -1)
			gc_GetCallState(session->crn, &session->callstate);

		memset(&event, 0, sizeof(event));
                switch(evtype)
                {
                case GCEV_FATALERROR:
                case GCEV_TASKFAIL:
                        event.id = CALL_FAILURE;
                        session->postEvent(&event);
                        break;
                case GCEV_BLOCKED:
                        slog(Slog::levelDebug) << session->logname << ": blocked" << endl;
                        session->blocked = true;
                        event.id = DEVICE_BLOCKED;
                        session->postEvent(&event);
                        break;
                case GCEV_UNBLOCKED:
                        slog(Slog::levelDebug) << session->logname << ": unblocked" << endl;
                        session->blocked = false;
                        event.id = DEVICE_UNBLOCKED;
                        session->postEvent(&event);
                        break;
                case GCEV_RESETLINEDEV:
                        session->crn = -1;
                        event.id = CALL_RESTART;
                        session->postEvent(&event);
                        break;
                case GCEV_RESTARTFAIL:
                        event.id = RESTART_FAILED;
                        session->postEvent(&event);
                        break;
                case GCEV_DISCONNECTED:
                        event.id = STOP_DISCONNECT;
                        session->postEvent(&event);
                        break;
		case GCEV_CALLINFO:
			if(gc_GetSigInfo(linedev, (char *)&ie_blk, U_IES, &metaevent) == GC_SUCCESS)
			{
				if(len = extractSignaling(ie_blk.data, buf, ie_blk.length) > 0)
				{
					extractInfo(buf, dialed, caller, context, route, len);
					session->setConst("session.caller", caller);
					session->setConst("session.dialed", dialed);
					session->setConst("session.context", context);
					session->setConst("session.route", route);
				}
			}	
			break;
                case GCEV_OFFERED:
			if(gc_GetSigInfo(linedev, (char *)&ie_blk, U_IES, &metaevent) == GC_SUCCESS)
 			{
				if(len = extractSignaling(ie_blk.data, buf, ie_blk.length) > 0)
				{
                        		extractInfo(buf, dialed, caller, context, route, len);  
                        		session->setConst("session.caller", caller);
                        		session->setConst("session.dialed", dialed);
                        		session->setConst("session.context", context);  
                        		session->setConst("session.route", route);
				}
			}

                        session->crn = metaevent.crn;
                        event.id = CALL_OFFERED;
                        session->postEvent(&event);
                        break;
                case GCEV_ACCEPT:
                        event.id = CALL_ACCEPTED;
                        session->postEvent(&event);
                        break;
                case GCEV_ANSWERED:
                        event.id = LINE_OFF_HOOK;
                        session->queEvent(&event);
                        break;
                case GCEV_CONNECTED:
                        event.id = CALL_CONNECTED;
                        session->queEvent(&event);
                        break;
                case GCEV_DROPCALL:
                        event.id = CALL_DISCONNECT;
                        session->queEvent(&event);
                        break;
                case GCEV_RELEASECALL:
                        session->crn = -1;
                        event.id = CALL_RELEASED;
                        session->queEvent(&event);
                        break;
                case GCEV_RELEASECALL_FAIL:
                        session->crn = -1;
                        event.id = RELEASE_FAILED;
                        session->queEvent(&event);
                        break;
                case TDX_CALLP:
			event.id = DIAL_CONNECT;
                        switch(ATDX_CPTERM(session->devnode->chdev))
                        {
                        case CR_BUSY:
                                switch(ATDX_CRTNID(session->devnode->chdev))
                                {
                                case TID_BUSY1:
					event.id = DIAL_BUSY;
                                        break;
                                case TID_BUSY2:
					event.id = DIAL_BUSY;
                                        break;
                                default:
					event.id = DIAL_FAILED;
                                        break;
                                }
                                break;
                        case CR_CEPT:
                                event.id = DIAL_CONNECT;
                                break;
                        case CR_CNCT:
				event.cpa.err = NULL;
                                switch(ATDX_CONNTYPE(session->devnode->chdev))
                                {
                                case CON_CAD:
                                        event.cpa.msg = "cadence";
                                        break;
                                case CON_LPC:
                                        event.cpa.msg = "loop";
                                        break;
                                case CON_PVD:
					event.id = DIAL_CONNECT;
                                        break;
                                case CON_PAMD:
					event.id = DIAL_PAM;
                                        break;
                                default:
					event.id = DIAL_FAILED;
                                        break;
                                }
                                break;
                        case CR_FAXTONE:
				event.id = DIAL_FAX;
                                break;
                        case CR_NOANS:
				event.id = DIAL_TIMEOUT;
                                break;
                        case CR_NODIALTONE:
				event.id = DIAL_FAILED;
                                break;
                        case CR_NORB:
				event.id = DIAL_FAILED;
                                break;
                        case CR_STOPD:
				event.id = DIAL_FAILED;
                                break;
                        default:	
				event.id = DIAL_FAILED;
                                break;
                        }
			session->queEvent(&event);
                        break;
                case TDX_CST:
                        cst = (DX_CST *)metaevent.evtdatap;
                        switch(cst->cst_event)
                        {
                        case DE_RINGS:                                
				if(cst->cst_data & 0x0001)
                                {
                                        event.id = LINE_CALLER_ID;
                                        session->queEvent(&event);
                                }
                                event.id = RING_ON;
                                session->queEvent(&event);
                                break;
                        case DE_RNGOFF:
                                event.id = TIMER_EXPIRED;
                                session->queEvent(&event);
                                break;
                        case DE_LCOF:
                        case DE_WINK:
                                event.id = LINE_WINK;
                                session->queEvent(&event);
                                break;
                        case DE_DIGITS:
                                event.id = DTMF_KEYUP;
                                event.dtmf.duration = 40;
                                event.dtmf.digit = getDigit(cst->cst_data);
                                session->queEvent(&event);
                                break;
                        }
			break;
                case TDX_GETDIG:
			diglen = strlen(digbuf->dg_value);
			digcnt = 0;
                        while(digcnt < diglen)
                        {
                                event.id = DTMF_KEYUP;
                                event.dtmf.duration = 40;
        			event.dtmf.digit = getDigit(digbuf->dg_value[digcnt++]);
                                session->queEvent(&event);
                        }
                case TDX_PLAY:
                case TDX_RECORD:
                case TDX_DIAL:
                case TDX_PLAYTONE:
                case TDX_ERROR:
                        tmask = ATDX_TERMMSK(session->devnode->chdev);
                        if(tmask & TM_LCOFF)
                        {
                                event.id = LINE_WINK;
                                session->queEvent(&event);
                        }
        		if((tmask & TM_IDDTIME) || (tmask & TM_MAXTIME) || (tmask & TM_ERROR))
                        {
				event.id = TIMER_EXPIRED;
                                session->queEvent(&event);
                        }
                        if(tmask & TM_TONE)
                        {
				event.tone.name = "dialtone";
				event.tone.exit = false;
                                event.id = TONE_START;
                                session->queEvent(&event);
                        }
                        if((tmask & TM_MAXSIL) || (tmask & TM_MAXNOSIL))
                        {
                                event.id = AUDIO_IDLE;
                                session->queEvent(&event);
                        }
                        switch(evtype)
                        {
                        case TDX_PLAYTONE:
                        case TDX_DIAL:
                        case TDX_CALLP:
                                event.id = TONE_IDLE;
                                break;
                        case TDX_PLAY:
                        case TDX_RECORD:
                                event.id = AUDIO_IDLE;
                        }
			if(event.id)
				session->queEvent(&event);
			break;
		case CCEV_SETCHANSTATE:
			break;
		case SR_TMOUTEVT:
                        event.id = TIMER_EXPIRED;
                        session->queEvent(&event);
			break;
		default:
			slog.error("gc: unkown event %04x", evtype);
		}
	}
}

extern "C" {

// "null" operation from our point of view
long uio_seek(int fd, long offset, int wence)
{
	return offset;
}

int uio_read(int fd, char *buffer, unsigned len)
{
	int rtn;
	int offset = 0;
	bool linear = false;
	BayonneSession *session = Bayonne::getSession((Bayonne::timeslot_t)(fd));

	if(!session)
		return -1;

        switch(session->audio.getEncoding())
        {
        case Audio::pcm16Mono:
		linear = true;
        }

	if(linear)
        {
		AudioCodec *codec = AudioCodec::getCodec(Audio::mulawAudio);
        	Audio::Linear lbuffer = new Audio::Sample[len];
                rtn = session->audio.getBuffer((Audio::Encoded)lbuffer, len * 2) / 2;
		if(rtn > 0)
			codec->encode(lbuffer, buffer, rtn / 2);
                delete[] lbuffer;
                return rtn;
	}
        else
        	return session->audio.getBuffer((Audio::Encoded)buffer, len);
}

int uio_write(int fd, char *buffer, unsigned len)
{
	int rtn;
	int offset = 0;
	bool linear = false;
	BayonneSession *session = Bayonne::getSession((Bayonne::timeslot_t)(fd));

	if(!session)
		return -1;

        switch(session->audio.getEncoding())
        {
        case Audio::pcm16Mono:
		linear = true;
        }

	if(linear)
        {
		AudioCodec *codec = AudioCodec::getCodec(Audio::mulawAudio);
        	Audio::Linear lbuffer = new Audio::Sample[len];
		codec->decode(lbuffer, buffer, len);
		rtn = session->audio.putBuffer((Audio::Encoded)lbuffer, len * 2);
		if(rtn > 0)
			rtn /= 2;
                delete[] lbuffer;
                return rtn;
	}
        else
        	return session->audio.putBuffer((Audio::Encoded)buffer, len);
}

} // extern

} // namespace
