
/*

________________________________________________________________

        scatter
        $Id: scatter.c,v 1.25 1997/01/09 16:43:14 svein Exp $
        Copyright 1990, Blab, UiO
        Image processing lab, Department of Informatics
        University of Oslo
        E-mail: blab@ifi.uio.no
________________________________________________________________
  
  Permission to use, copy, modify and distribute this software and its
  documentation for any purpose and without fee is hereby granted, 
  provided that this copyright notice appear in all copies and that 
  both that copyright notice and this permission notice appear in supporting
  documentation and that the name of B-lab, Department of Informatics or
  University of Oslo not be used in advertising or publicity pertaining 
  to distribution of the software without specific, written prior permission.

  B-LAB DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL B-LAB
  BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
  OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
  CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

*/

static char *Id = "$Id: scatter.c,v 1.25 1997/01/09 16:43:14 svein Exp $, Blab, UiO";



#include <xite/includes.h>
#include <xite/biff.h>
#include <stdlib.h>
#include <xite/scatter.h>
#include XITE_STDIO_H
#include <xite/blab.h>
#include <xite/message.h>
#include <xite/readarg.h>

#ifndef MIN
# define MIN(a,b) (((a)<(b)) ? (a) : (b))
#endif

static int scatterOverflow;



/*F:scatter*

________________________________________________________________

		scatter
________________________________________________________________

Name:		scatter - make a scatterplot

Syntax:         | #include <xite/scatter.h>
		|
                | int scatter( IBAND input1, IBAND input2,
                |    IBAND output );
Description:	Creates a scatterplot (a two dimensional
                histogram) of the two input bands. Every
		scatter update that would increase the
		scatter value beyond 255 increases the
		static int variable scatterOverflow in stead.

Return value:	| 0 => ok
                | 1 => Bad pixel type in input1
                | 2 => Bad pixel type in input2
                | 3 => Bad pixel type in output
		| 4 => Output has xsize less than 256
		| 5 => Output has ysize less than 256

Author:		Tor L|nnestad, BLAB, ifi, UiO

Id:             $Id: scatter.c,v 1.25 1997/01/09 16:43:14 svein Exp $
________________________________________________________________

*/



#ifndef FUNCPROTO
int scatter(input1, input2, output)
IBAND input1, input2, output;
#else /* FUNCPROTO */
int scatter(IBAND input1, IBAND input2, IBAND output)
#endif /* FUNCPROTO */
{
   int x, y, val1, val2, xsize, ysize;

   scatterOverflow = 0;
   if (Ipixtyp(input1) NE Iu_byte_typ)
     return(Error(1, "scatter: Bad pixel type in input1\n"));
   if (Ipixtyp(input2) NE Iu_byte_typ)
     return(Error(2, "scatter: Bad pixel type in input2\n"));
   if (Ipixtyp(output) NE Iu_byte_typ)
     return(Error(3, "scatter: Bad pixel type in output\n"));
   if (Ixsize(output) LT 256)
     return(Error(4, "scatter: Output has xsize less than 256\n"));
   if (Iysize(output) LT 256)
     return(Error(5, "scatter: Output has ysize less than 256\n"));

   xsize = MIN(Ixsize(input1),Ixsize(input2));
   ysize = MIN(Iysize(input1),Iysize(input2));

   for (y=1; y LE 256; y++)
     for (x=1; x LE 256; x++)
       output[y][x] = 0;

   FOR (y=1; y LE ysize; y++)
     FOR (x=1; x LE xsize; x++)
       val1 = input1[y][x]+1;
       val2 = input2[y][x]+1;
       if (output[val1][val2] LT 255)
         ++ output[val1][val2];
       else ++ scatterOverflow;
     ENDFOR;
   ENDFOR;
   return 0;
}  /*  scatter  */





/*P:scatter*

________________________________________________________________

		scatter
________________________________________________________________

Name:		scatter - make a scatterplot

Syntax:		scatter <input1> <input2> <output>

Description:	Creates a scatterplot (a two dimensional
                histogram) of band 1 of the two input images.
		The scatterplot is returned as an unsigned byte
		image.

Restrictions:	Accepts only UNS_BYTE pixels. Every
		scatter update that would increase the
		scatter value beyond 255 increases an
		overflow counter, which is eventually
		written to stderr.

Author:		Tor Lnnestad, BLAB, Ifi, UiO

Examples:	| scatter image1.img image2.img scatter.img
		| scatter image.img:1 image.img:2 scatter.img

Id:             $Id: scatter.c,v 1.25 1997/01/09 16:43:14 svein Exp $
________________________________________________________________

*/

#ifdef MAIN

#ifndef FUNCPROTO
int main(argc,argv)
int argc;
char *argv[];
#else /* FUNCPROTO */
int main(int argc, char **argv)
#endif /* FUNCPROTO */
{
   IMAGE i1, i2, i3;
   char* title;
   int stat;

   InitMessage(&argc, argv, xite_app_std_usage_text(
    "Usage: %s <oldfile1> <oldfile2> <newfile>\n"));
   Iset_message(TRUE);
   Iset_abort(TRUE);
   title = read_switch(&argc, argv, "-t", 1, "SCATTERPLOT");
   title = read_switch(&argc, argv, "-title", 1, title);

   if (argc == 1) Usage(1, (char*)0);
   if (argc != 4) Usage(2, "Bad number of arguments\n");

   i1 = Iread_image(argv[1]);
   i2 = Iread_image(argv[2]);
   i3 = Imake_image(1, title, Iu_byte_typ, 256, 256);
   Iset_start(i3[1], 0, 0);
   if ((stat = scatter(i1[1], i2[1], i3[1]))) exit(stat);
   if (scatterOverflow)
     Warning(6, "Scatter overflows: %d\n",scatterOverflow);
   Ihistory(i3, argv[0], "");
   Iwrite_image(i3, argv[3]);
   return(0);
} /* main */

#endif
