#ifndef lint
static       char    rcsid[] = "$Header: arc.c,v 1.2 90/12/12 15:30:21 zhang Exp $";
#endif

/*
 * $Log:	arc.c,v $
 * 
 * Revision 1.1  90/06/17  03:34:51  zhang
 * Initial revision
 * 
 * Revision 1.2  90/12/12  15:30:21  zhang
 * Add backend for RADIANCE output
 */

#include "defs.h"

/*
 * how to define faces by an arc ?
 *
 * there're two cases
 * 	1. a polygon defined the arc and a line between the start point
 * 	   and the end point (the center point is not used)
 * 	2. a fan-shape polygon defined by the arc and two lines
 *	   between the center and the start/end points
 *      3. only the side faces of an arc having thickness
 *	   
 * case 3 is chosen as it is way which AutoCAD handles. originally case 2
 * was chosen as it is more resonable :-)
 */

typedef	struct	arc	{
	FLOAT	center[3];		/* center of arc */
	FLOAT	radius;			/* radius of arc */
	FLOAT	startangle;		/* start angle of arc */
	FLOAT	endangle;		/* end angle of arc */
} ARC;

/*
 * parse an ARC
 */

ENTITY	*ArcDxfParse(layerlist)
LAYER	**layerlist;
{
	ENTITY	*entity;
	ARC	*arc;
	INT	centerset = 0;
	INT	radiusset = 0;
	INT	startangleset = 0;
	INT	endangleset = 0;
	INT	defaultflag = 0;

	entity = Malloc(ENTITY, 1);
	arc = Malloc(ARC, 1);
	entity->type = ENTITY_ARC;
	entity->data = (VOID *) arc;

	entity->layer = LayerDxfParse(layerlist);

	do {
		GetNextGroup();
		switch(Group->code) {
		case 10:
			/*
			 * center point
			 * 10, 20, 30
			 */

			if (centerset != 0)
				DXFERR("duplicated center for an ARC %s", "\n");

			centerset = 1;
			CoordDxfParse(0, arc->center);
			break;

		case 40:
			/*
			 * radius
			 */

			if (radiusset != 0)
				DXFERR("duplicated radius for an ARC %s", "\n");

			radiusset = 1;
			arc->radius = Group->fltnum;
			break;

		case 50:
			/*
			 * start angle
			 */

			if (startangleset != 0)
				DXFERR("duplicated start angle for an ARC %s", "\n");

			startangleset = 1;
			arc->startangle = DEG_TO_RAD(Group->fltnum);
			break;


		case 51:
			/*
			 * end angle
			 */

			if (endangleset != 0)
				DXFERR("duplicated end angle for an ARC %s", "\n");
			endangleset = 1;
			arc->endangle = DEG_TO_RAD(Group->fltnum);
			break;

		default:
			if (OptionsDxfParse(&entity->options) == 0)
				defaultflag = 1;
			break;
		}
	} while (defaultflag == 0);

	/*
	 * check if center point, radius, start angle
	 * and end angle are all defined
	 */

	if (centerset == 0)
		DXFERR("undefined center for an ARC %s", "\n");

	if (radiusset == 0)
		DXFERR("undefined radius for an ARC %s", "\n");

	if (startangleset == 0)
		DXFERR("undefined start angle for an ARC %s", "\n");

	if (endangleset == 0)
		DXFERR("undefined end angle for an ARC %s", "\n");

	return(entity);
}

/*
 * calculate tesselation number for an arc
 */

INT	ArcTesselationNumber(radius, deltaangle)
FLOAT	radius;
FLOAT	deltaangle;
{
	INT	n;

	if (deltaangle > (2.0 * M_PI))
		return(CircleTesselationNumber(radius));

	n = (INT) (0.5 +
		(FLOAT) CircleTesselationNumber(radius) * deltaangle / (2.0 * M_PI));

	return(max(1, n));
}

/*
 * output a (partial) disk defined by the arc into DEF/DEF/DXF file
 */

VOID	ArcDiskOutput(entity, arc)
ENTITY	*entity;
ARC	*arc;
{
	INT	i;
	INT	n;
	FLOAT	deltaangle;
	FLOAT	*extrusion;
	MATRIX	matrix[2];
	FLOAT	delta[3];
	INT	ntesselation;
	FLOAT	p[3];
	FLOAT	polygon[MaxCircleTesselation][3];

	extrusion = OptionsExtrusion(entity->options);

	MatrixExtrusion(matrix, extrusion);

	n = ArcTesselationNumber(arc->radius, fabs(arc->endangle - arc->startangle));
	deltaangle = (arc->endangle - arc->startangle) / (FLOAT) n;

	for (i = 0; i <= n; i++) {
		p[0] = arc->radius * cos(arc->startangle + deltaangle * (FLOAT) i);
		p[1] = arc->radius * sin(arc->startangle + deltaangle * (FLOAT) i);
		p[2] = 0.0;
		XformPoint(matrix[0], p, polygon[i]);
		VecAdd(polygon[i], arc->center, polygon[i]);
	}

	VecCopy(arc->center, polygon[n + 1]);

	if (OutFileType == FILE_NFF || OutFileType == FILE_RAD) {
		OutputPolygon(n + 2, polygon, entity);
		return;
	}

	/*
	 * tesselation triangles of the arc
	 */

	VecSub(polygon[0], polygon[1], delta);
	ntesselation = (INT) (arc->radius / VecLength(delta) + 0.5);
	if (ntesselation <= 0)
		ntesselation = 1;

	for (i = 0; i < n; i++)
		CircleTriangleTesselation(ntesselation, arc->center,
			polygon[i], polygon[i + 1], entity);
}

/*
 * output faces defined by an ARC into DXF/DEF/NFF/RAD file
 */

VOID	ArcDefOutput(entity)
ENTITY	*entity;
{
	ARC	*arc;
	FLOAT	thickness;
	FLOAT	*extrusion;
	FLOAT	delta[3];
	FLOAT	center[3];
	INT	i;
	INT	n;
	FLOAT	deltaangle;
	FLOAT	width;
	FLOAT	height;
	INT	heighttesselation;
	MATRIX	matrix[2];
	FLOAT	p[3];
	FLOAT	polygon[2][MaxCircleTesselation][3];
	FLOAT	tetragon[4][3];

	/* for case 2
	 *
	 * FLOAT	a, b, c, d;
	 */

	arc = (ARC *) entity->data;

	if (arc->radius < TOE)
		return;

	if (fabs(arc->endangle - arc->startangle) < TOE)
		return;

	if (OptionsThickness(entity->options, &thickness) == 0) {
		/*
		 * for case 2
		 * NameOutput(enitty->layer);
		 * ArcDiskDefOutput(entity, arc);
		 */
		return;
	}

	NameOutput(entity->layer);

	extrusion = OptionsExtrusion(entity->options);

	VecScale(thickness, extrusion, delta);
	VecAdd(delta, arc->center, center);

	height = VecLength(delta);

	MatrixExtrusion(matrix, extrusion);

	n = ArcTesselationNumber(arc->radius, fabs(arc->endangle - arc->startangle));
	deltaangle = (arc->endangle - arc->startangle) / (FLOAT) n;

	for (i = 0; i <= n; i++) {
		p[0] = arc->radius * cos(arc->startangle + deltaangle * (FLOAT) i);
		p[1] = arc->radius * sin(arc->startangle + deltaangle * (FLOAT) i);
		p[2] = 0.0;
		XformPoint(matrix[0], p, polygon[0][i]);
		VecAdd(polygon[0][i], arc->center, polygon[0][i]);
		VecAdd(polygon[0][i], delta, polygon[1][i]);
	}

	/*
	 * *
	 * * check the orientation of the arc against the
	 * * extrusion direction
	 * *
	 * 
	 * VecCopy(arc->center, polygon[0][n + 1]);
	 * PlaneEquation(n + 2, polygon[0], &a, &b, &c, &d);
	 * if ((a * extrusion[0] + b * extrusion[1] + c * extrusion[2]) > 0.0)
	 * 	ReversePolygon(n + 1, polygon[0]);
	 * else
	 * 	ReversePolygon(n + 1, polygon[1]);
	 *
	 *
	 *
	 * for case 2
	 *
	 * check the orientation against the reference layer name
	 * or layer reference point
	 * the orientations of the two circle should be changed
	 * simulatously
	 *
	 *
	 * if (CheckOrientation(n, polygon[0], entity->layer) !=
	 *     CheckOrientation(n, polygon[1], entity->layer)) {
	 * 	fprintf(stderr, "%s: the two (arc) disks of a cylinder in layer %s have the same orientation\n%s: after checking reference\n", 
	 * 			ProgName, entity->layer->name, ProgName);
	 * 	exit(-1);
	 * }
	 *
	 * *
	 * * if there is no reference point in the layer and the layer name
	 * * has an "_I" suffix, the orientation of the arc disks have to be
	 * * reversed
	 * *
	 *
	 * if (entity->layer->point == NULL)
	 * 	if (CheckLayerName(entity->layer) != 0) {
	 * 		ReversePolygon(n + 1, polygon[0]);
	 * 		ReversePolygon(n + 1, polygon[1]);
	 * 	}
	 *
	 * *
	 * * tesselation the two arcs
	 * * 
	 *
	 * for (i = 0; i < n; i++)
	 *	CircleTriangleTesselation(ntesselation, arc->center,
	 *		polygon[0][i], polygon[0][i + 1]);
	 *
	 * for (i = 0; i < n; i++)
	 * 	CircleTriangleTesselation(ntesselation, center,
	 * 		polygon[1][i], polygon[1][i + 1]);
	 * 
	 *
	 * for case 2
	 */

	/*
	 * tesselation side faces
	 * note: the direction in width is not tesselated
	 */

	VecSub(polygon[0][0], polygon[0][1], delta);
	width = VecLength(delta);
	heighttesselation = (INT) (height / width + 0.5);
	if (heighttesselation <= 0)
		heighttesselation = 1;

	for (i = 0; i < n; i++)
		if (OutFileType == FILE_NFF || OutFileType == FILE_RAD) {
			VecCopy(polygon[0][i + 1], tetragon[0]);
			VecCopy(polygon[0][i    ], tetragon[1]);
			VecCopy(polygon[1][i    ], tetragon[2]);
			VecCopy(polygon[1][i + 1], tetragon[3]);
			OutputPolygon(4, tetragon, entity);
		} else
			CircleRectangleTesselation(1, heighttesselation,
	/*
	 * for case 2
	 *		polygon[0][i + 1], polygon[0][i],
	 *		polygon[1][n - i], polygon[1][n - i - 1],
	 */
	 		polygon[0][i + 1], polygon[0][i],
	 		polygon[1][i], polygon[1][i + 1],
			entity);

	/*
	 * for case 2
	 * CircleRectangleTesselation(
	 * 	max(1, (INT) (arc->radius / width + 0.5)), heighttesselation,
	 * 	polygon[0][0], arc->center, center, polygon[1][n]);
	 *
	 *
	 * CircleRectangleTesselation(
	 * 	max(1, (INT) (arc->radius / width + 0.5)), heighttesselation,
	 * 	polygon[0][n], arc->center, center, polygon[1][0]);
 	 */
}
