/* Copyright (c) 1994 Gregory P. Ward.  All rights reserved.
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without written
 * agreement is hereby granted, provided that the above copyright
 * notice and the following two paragraphs appear in all copies of
 * this software.  
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE
 * AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER
 * IS ON AN "AS IS" BASIS, AND THE AUTHOR HAS NO OBLIGATION TO PROVIDE
 * MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

/* ----------------------------- MNI Header -----------------------------------
@NAME       : interface.c
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Stuff relating to the interactive user interface in the
              GL MPEG player.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/6/23?, Greg Ward.
@MODIFIED   : 
---------------------------------------------------------------------------- */

#include <gl/gl.h>
#include <gl/device.h>
#include "gl_mpeg.h"


/* ----------------------------- MNI Header -----------------------------------
@NAME       : InitializeQueues
@INPUT      : (none)
@OUTPUT     : (none)
@RETURNS    : (void)
@DESCRIPTION: Calls qdevice() for all the various devices we wish to 
              monitor in HandleEvents().
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/6/23?, Greg Ward
@MODIFIED   : Incessantly.
---------------------------------------------------------------------------- */
void InitializeQueues ()
{
   qdevice (ESCKEY);		/* these are all "Quit" signals */
   qdevice (QKEY);
   qdevice (WINSHUT);
   qdevice (WINQUIT);

   qdevice (SPACEKEY);		/* pause/play */
   qdevice (MIDDLEMOUSE);

   qdevice (FKEY);		/* go forwards */
   qdevice (BKEY);		/* go backwards */
   qdevice (RKEY);		/* go back and forth */

   qdevice (HKEY);		/* bring up help window */

   qdevice (CKEY);		/* continuous play */
   qdevice (OKEY);		/* once-only play */

   qdevice (ZKEY);		/* zoom up */
   qdevice (EQUALKEY);		/* ditto ('cause it has a + too!) */
   qdevice (PADPLUSKEY);	/* ditto again */
   qdevice (XKEY);		/* zoom down */
   qdevice (MINUSKEY);		/* ditto */
   qdevice (PADMINUS);		/* ditto */

   qdevice (DKEY);		/* double-buffered */
   qdevice (SKEY);		/* single-buffered */

   qdevice (RETKEY);		/* single step (if paused) */
   qdevice (BACKSPACEKEY);	/* change direction */
   qdevice (F1KEY);		/* toggle fullscreen mode */

   qdevice (LEFTARROWKEY);	/* step back one frame */
   qdevice (LEFTMOUSE);
   qdevice (RIGHTARROWKEY);	/* step forward one frame */
   qdevice (RIGHTMOUSE);

   qdevice (HOMEKEY);		/* rewind to frame 0 */
   qdevice (ENDKEY);		/* fast-forward to last frame */

   qdevice (UPARROWKEY);	/* speed movie up (decrease frame delay) */
   qdevice (DOWNARROWKEY);	/* slow movie down */

   qdevice (WINFREEZE);
   qdevice (WINTHAW);
}


/* ----------------------------- MNI Header -----------------------------------
@NAME       : RedrawWindow
@INPUT      : Movie->Window - needs Width and Height, to check if window
                 has been resized
@OUTPUT     : Movie->Window - sets Width, Height, zoom factors, and image
                 location if window has been resized
@RETURNS    : 
@DESCRIPTION: Redraw the main movie window: if size has changed, image 
              is rescaled to fit in window.  If movie is paused, the
	      current frame is explicitly redrawn; otherwise, we assume
	      things will get redrawn soon enough.
@METHOD     : 
@GLOBALS    : 
@CALLS      : CalcZoomFromWindow
              CalcImageLocation
	      DisplayFrame
	      GL stuff
@CREATED    : 94/6/25?, Greg Ward (from code formerly in HandleEvents()).
@MODIFIED   : 
---------------------------------------------------------------------------- */
void RedrawWindow (MovieState *Movie)
{
   long  WinX, WinY;	/* current size of the window */
   
   dprintf ("REDRAW\n");
   getsize (&WinX, &WinY);
   dprintf ("Old size: %d, %d; New size: %d, %d\n",
	    Movie->Window.Width, Movie->Window.Height,
	    WinX, WinY);

   if (WinX != Movie->Window.Width || WinY != Movie->Window.Height)
   {
      reshapeviewport ();
      dprintf ("Size change!\n");
      Movie->Window.Width = WinX;
      Movie->Window.Height = WinY;
      CalcZoomFromWindow (&Movie->Window, &Movie->Image, TRUE);
      CalcImageLocation (&Movie->Window, &Movie->Image);
      rectzoom (Movie->Window.ZoomX, Movie->Window.ZoomY);
      gconfig ();
      dprintf ("New zoom: %g, %g\n", 
	       Movie->Window.ZoomX, Movie->Window.ZoomY);
      dprintf ("New image location: %d, %d\n",
	       Movie->Window.ImageX, Movie->Window.ImageY);
   }
   
   clear ();
   if (Movie->Options.DoubleBuff)
   {
      swapbuffers();	
      clear();
   }
   if (Movie->Status.Paused)
   {
      DisplayFrame (Movie);
   }
}     /* RedrawWindow () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : DrawIcon
@INPUT      : Movie->Image (for image size), CurrentImage (for the actual
                 image data to be rescaled)
@OUTPUT     : (none)
@RETURNS    : (void)
@DESCRIPTION: Resamples the current frame down to icon size, and writes it
              in the current window (presumably the icon, though this isn't
	      checked here).
@METHOD     : 
@GLOBALS    : 
@CALLS      : GL stuff
@CREATED    : 94/7/8, Greg Ward (blatantly ripped off from Gabe Leger's 
                      postf)
@MODIFIED   : 
---------------------------------------------------------------------------- */
void DrawIcon (MovieState *Movie)
{
   long	 icon_width, icon_height;
   long  image_width, image_height;
   long  x, x_step, x_skip;
   long  y, y_step, y_skip;
   int   i;
   unsigned long *Icon, *Image;

   getsize (&icon_width, &icon_height);
   image_width = Movie->Image.Width;
   image_height = Movie->Image.Height;
   
   x_step = image_width / icon_width;
   y_step = image_height / icon_height;
   
   x_skip = (image_width % icon_width) / 2;
   y_skip = (image_height % icon_height) / 2;
   
   if (x_step == 0)
   {
      x_step = 1;
      x_skip = 0;
   }
   if (y_step == 0)
   {
      y_step = 1;
      y_skip = 0;
   }

   Image = (unsigned long *) Movie->CurrentImage;
   Icon = (unsigned long *) malloc (icon_width * icon_height * sizeof (long));

   i = 0;
   for (y = 0; y < icon_height; y++)
      for (x = 0; x < icon_width; x++)
	 Icon [i++] = *(Image + image_width * (y_step * y + y_skip) + x_step * x + x_skip);

   rectzoom (1.0, 1.0);
   lrectwrite (0, 0, icon_width-1, icon_height-1, Icon);
   rectzoom (Movie->Window.ZoomX, Movie->Window.ZoomY);
   
}     /* DrawIcon () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : SetPause
@INPUT      : Movie->Status.Forward (if Pause==TRUE)
              Pause - whether to pause or unpause the movie
@OUTPUT     : Movie->Status.Forward (if Pause==FALSE)
              Movie->Status.Paused (always)
@RETURNS    : (void)
@DESCRIPTION: Sets the pause state of a movie.  The current direction of play
              is saved in a static variable if Pause is TRUE, and restored
	      from the static if Pause is FALSE.  (For this reason, SetPause
	      should not be called with Pause==FALSE before it is called
	      with Pause==TRUE!)
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/7/18, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void SetPause (MovieState *Movie, Boolean Pause)
{
   static  Boolean PrevDir;	/* store state of Movie->Status.Forward */
   
   if (Pause)
      PrevDir = Movie->Status.Forward;
   else
     Movie->Status.Forward = PrevDir;

   Movie->Status.Paused = Pause;
}



/* ----------------------------- MNI Header -----------------------------------
@NAME       : RezoomWindow
@INPUT      : ZoomX, ZoomY - the new desired zoom factors
@OUTPUT     : Movie->Window.{ZoomX,ZoomY} are set to corresponding input args 
                            {Width,Height} recalculated from those
                            {ImageX,ImageY} also recalculated
@RETURNS    : (void)
@DESCRIPTION: Resizes a movie window and recalculates the position of the 
              image within the window using the desired zoom factors.
@METHOD     : 
@GLOBALS    : 
@CALLS      : WindowMode
@CREATED    : 94/7/20, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void RezoomWindow (MovieState *Movie, float ZoomX, float ZoomY)
{
   long   WinX, WinY;
   
   getorigin (&WinX, &WinY);
   Movie->Window.ZoomX = ZoomX;
   Movie->Window.ZoomY = ZoomY;
   Movie->Window.Height = Movie->Window.Width = 0;
   WindowMode (Movie, -1, -1);
   rectzoom (Movie->Window.ZoomX, Movie->Window.ZoomY); 
   winposition (WinX, WinX+Movie->Window.Width-1,
		WinY, WinY+Movie->Window.Height-1);
   reshapeviewport ();
}



/* ----------------------------- MNI Header -----------------------------------
@NAME       : HandleEvents
@INPUT      : Movie - used primarily for option and status flags, and the
                      image data of the current frame
@OUTPUT     : *Done - set to TRUE if one of the quit events is received
                      (eg. Q or ESC keys, WINSHUT or WINQUIT events)
              Movie - various flags may be adjusted, as well as the current
                      frame number (and of course the image data)
@RETURNS    : (void)
@DESCRIPTION: Tests the GL event queue, and processes events until the
              queue is empty.  If the movie is paused, HandleEvents()
	      keeps control until it is unpaused.
@METHOD     : 
@GLOBALS    : 
@CALLS      : RedrawWindow
              DrawIcon
	      AdvanceFrame, LoadFrame, DisplayFrame
	      FullScreenMode, WindowMode, OpenWindow
@CREATED    : 94/6/23, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void HandleEvents (Boolean *Done, MovieState *Movie)
{
   short   value;
   long    device;
   Boolean Iconized = FALSE;
   Boolean PrevPause;		/* in case of iconization */
   static  long CurrentWin = -1;

   if (CurrentWin == -1) CurrentWin = winget ();

   while (qtest () || Movie->Status.Paused)
   {
      UpdateStatus (Movie);
      device = qread (&value);
      if (device == INPUTCHANGE) 
      {
	 CurrentWin = value;
	 dprintf ("Now in window %d\n", CurrentWin);
	 continue;
      }

      /* Ignore key and mouse button events if they are button releases */

      if ((ISKEYBD (device) || ISMOUSEBUT (device)) && (value == 0))
      {
	 continue;
      }

      /* Ignore button events if movie window is iconized and we're
       * currently in it.
       */

      if ((ISKEYBD (device) || ISMOUSEBUT (device)) &&
	  (Iconized && CurrentWin == Movie->Window.Handle))
      {
	 continue;
      }
	 
      dprintf ("Event received: device %ld, value %d\n", device, (int) value);
/*
      dprintf ("mode: %s    direction: %s    current frame: %d\n", 
	       (Movie->Status.Paused) ? "pause" : "play",
	       (Movie->Status.Forward) ? "forwards" : "backwards",
	       Movie->CurFrame);
*/
      switch (device)
      {
	 case REDRAW:
	 {
	    if (value == Movie->Window.Handle)
	    {
	       RedrawWindow (Movie);
	    }
	    else
	    {
	       WriteHelpWindow (Movie);
	    }
	    break;
	 }

	 case REDRAWICONIC:
	 case WINFREEZE:
	 {
	    if (value != Movie->Window.Handle) break;
	    PrevPause = Movie->Status.Paused;
	    Movie->Status.Paused = TRUE;
	    Iconized = TRUE;
	    DrawIcon (Movie);
	    break;
	 }

	 case WINTHAW:
	 {
	    if (value != Movie->Window.Handle) break;
	    Movie->Status.Paused = PrevPause;
	    Iconized = FALSE;
	    break;
	 }

	 case MIDDLEMOUSE:
	 case SPACEKEY:
	 {
	    if (CurrentWin != Movie->Window.Handle) break;
	    SetPause (Movie, !Movie->Status.Paused);
	    dprintf ("Switching to %s mode\n",
		     (Movie->Status.Paused) ? "pause" : "play");
	    break;
	 }

	 case RETKEY:
	 {
	    if (!Movie->Status.Paused || 
		Movie->Status.Decoding ||
		BufferType == NO_BUFFER ||
		CurrentWin != Movie->Window.Handle)
	    {
	       break;
	    }
	
	    AdvanceFrame (Movie);
	    Movie->CurrentImage =
	       LoadFrame (Movie->CurFrame, &Movie->Image);
	    DisplayFrame (Movie);
    
	    break;
	 }

	 case BACKSPACEKEY:
	 {
	    if (CurrentWin != Movie->Window.Handle) break;
	    if (!Movie->Status.Decoding && BufferType != NO_BUFFER)
	    {
	       if (Movie->Status.Paused)
	       {
		  StyleEnum  SaveStyle;

		  SaveStyle = Movie->PlayStyle;
		  if (Movie->PlayStyle != ROCK)
		  {
		     Movie->PlayStyle = 
			(Movie->PlayStyle == FORWARD) ? BACKWARD : FORWARD;
		  }
		  Movie->Status.Forward = !Movie->Status.Forward;

		  AdvanceFrame (Movie);
		  Movie->CurrentImage =
		     LoadFrame (Movie->CurFrame, &Movie->Image);
		  DisplayFrame (Movie);
		  Movie->Status.Forward = !Movie->Status.Forward;
		  Movie->PlayStyle = SaveStyle;
	       }
	       else
	       {
		  Movie->Status.Forward = !Movie->Status.Forward;
	       }
	    }
	    break;
	 }

	 case FKEY:
	 {
	    if (Movie->Status.Decoding || BufferType == NO_BUFFER) break;
	    Movie->PlayStyle = FORWARD;
	    Movie->Status.Forward = TRUE;
	    Movie->Status.Paused = FALSE;
	    break;
	 }
	 case BKEY:
	 {
	    if (Movie->Status.Decoding || BufferType == NO_BUFFER) break;
	    Movie->PlayStyle = BACKWARD;
	    Movie->Status.Forward = FALSE;
	    Movie->Status.Paused = FALSE;
	    break;
	 }
	 case RKEY:
	 {
	    if (Movie->Status.Decoding || BufferType == NO_BUFFER) break;
	    Movie->PlayStyle = ROCK;
	    Movie->Status.Paused = FALSE;
	    break;
	 }

	 case VIRGULEKEY:
	 case HKEY:
	 {
	    if (HelpWindowID == -1)
	    {
	       OpenHelpWindow (Movie);
	    }
	    else
	    {
	       winset (HelpWindowID);
	       winpop ();
	       winset (Movie->Window.Handle);
	    }
	    break;
	 }

	 case CKEY:
	 {
	    Movie->Options.Continuous = TRUE;
	    Movie->Status.Paused = FALSE;
	    break;
	 }
	 case OKEY:
	 {
	    Movie->Options.Continuous = FALSE;
	    break;
	 }

	 case ZKEY:
	 case EQUALKEY:
	 case PADPLUSKEY:
	 {
	    if (Movie->Options.FullScreen) break;
	    RezoomWindow (Movie, Movie->Window.ZoomX+1, Movie->Window.ZoomY+1);
	    break;
	 }
	 case XKEY:
	 case MINUSKEY:
	 case PADMINUS:
	 {
	    if (Movie->Options.FullScreen || 
		Movie->Window.ZoomX == 1 || Movie->Window.ZoomY == 1)
	    {
	       break;
	    }
	    RezoomWindow (Movie, Movie->Window.ZoomX-1, Movie->Window.ZoomY-1);
	    break;
	 }

	 case DKEY:
	 {
	    if (!Movie->Options.DoubleBuff)
	    {
	       Movie->Options.DoubleBuff = TRUE;
	       doublebuffer ();
	       gconfig ();
	       clear ();
	       if (Movie->Status.Paused)
	       {
		  DisplayFrame (Movie);
	       }
	       clear ();
	    }
	    break;
	 }
	 case SKEY:
	 {
	    if (Movie->Options.DoubleBuff)
	    {
	       Movie->Options.DoubleBuff = FALSE;
	       singlebuffer ();
	       gconfig ();
	       clear ();
	       if (Movie->Status.Paused)
	       {
		  DisplayFrame (Movie);
	       }
	    }
	    break;
	 }

	 case LEFTMOUSE:
	 case LEFTARROWKEY:
	 {
	    if (Movie->Status.Decoding || BufferType == NO_BUFFER) break;
	    Movie->Status.Forward = FALSE;
	    if (Movie->Status.Paused && Movie->CurFrame > 0)
	    {	
	       Movie->CurFrame--;
	       Movie->CurrentImage =
		  LoadFrame (Movie->CurFrame, &Movie->Image);
	       DisplayFrame (Movie);
	    }
	    break;
	 }

 	 case RIGHTMOUSE:
	 case RIGHTARROWKEY:
	 {
	    if (Movie->Status.Decoding || BufferType == NO_BUFFER) break;
	    Movie->Status.Forward = TRUE;
	    if (Movie->Status.Paused && Movie->CurFrame < Movie->TotFrames-1)
	    {
	       Movie->CurFrame++;
               Movie->CurrentImage =
	          LoadFrame (Movie->CurFrame, &Movie->Image);
	       DisplayFrame (Movie);
	    }
	    break;
	 }

	 case HOMEKEY:
	 {
/*
	    if (Movie->Status.Decoding && BufferType == NO_BUFFER) 
	    {
	       RewindMPEG (Movie->InputStream, &Movie->Image);
	       GetMPEGFrame (Movie->CurrentImage);
	       FlipImage (&Movie->Image, Movie->CurrentImage);
	       DisplayFrame (Movie);
	       Movie->CurFrame = 0;
	       Movie->TotFrames = 0;
	       Movie->Status.Paused = TRUE;
	       break;
	    }
*/
	    if (Movie->Status.Decoding) break;
	    Movie->CurFrame = 0;
	    Movie->CurrentImage =
	       LoadFrame (Movie->CurFrame, &Movie->Image);
	    DisplayFrame (Movie);
	    Movie->Status.Paused = TRUE;
	    break;
	 }
    
	 case ENDKEY:
	 {
	    if (Movie->Status.Decoding) break;
	    Movie->CurFrame = Movie->TotFrames-1;
	    Movie->CurrentImage =
	       LoadFrame (Movie->CurFrame, &Movie->Image);
	    DisplayFrame (Movie);
	    Movie->Status.Paused = TRUE;
	    break;
	 }

	 case UPARROWKEY:
	 {	
	    if (Movie->Status.Paused) break;
	    Movie->FrameDelay -= 3;
	    if (Movie->FrameDelay < 0) 
	       Movie->FrameDelay = 0;
	    break;
	 }

	 case DOWNARROWKEY:
	 {	
	    if (Movie->Status.Paused) break;
	    Movie->FrameDelay += 3;
	    break;
	 }

	 case F1KEY:
	 {
	    static long PrevX, PrevY;
	    static long PrevWidth, PrevHeight;

	    Movie->Options.FullScreen = !Movie->Options.FullScreen;
	    dprintf ("FullScreen: %d\n", Movie->Options.FullScreen);

	    if (Movie->Options.FullScreen)
	    {
	       getorigin (&PrevX, &PrevY);
	       getsize (&PrevWidth, &PrevHeight);
	       winclose (Movie->Window.Handle);
	       FullScreenMode (Movie);
	       OpenWindow (Movie);
	       cursoff();
	    }
	    else
	    {
	       winclose (Movie->Window.Handle);
	       Movie->Window.Width = PrevWidth;
	       Movie->Window.Height = PrevHeight;
	       WindowMode (Movie, PrevX, PrevY);
	       OpenWindow (Movie);
	       curson();
	    }
	    break;
	 }

	 case ESCKEY:
	 case QKEY:
	 {
	    /* Just put the current window (as determined by focus changes)
	     * in value, because value will contain the window that received
	     * the WINQUIT or WINSHUT event below, and we want ESC or Q
	     * to be handled the same as WINQUIT or WINSHUT.
	     */
	    value = CurrentWin;
	 }
	 case WINQUIT:
	 case WINSHUT:
	 {
	    if (value == Movie->Window.Handle)
	    {
	       qprintf ("Bye bye!\n");
	       *Done = TRUE;
	       return;
	    }
	    else
	    {
	       CloseHelpWindow ();
	    }
	    break;
	 }
      }     /* switch on device */
   }     /* while qtest () */

   UpdateStatus (Movie);
}
