/* Copyright (c) 1994 Gregory P. Ward.  All rights reserved.
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose, without fee, and without written
 * agreement is hereby granted, provided that the above copyright
 * notice and the following two paragraphs appear in all copies of
 * this software.  
 *
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE
 * AUTHOR HAS BEEN ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER
 * IS ON AN "AS IS" BASIS, AND THE AUTHOR HAS NO OBLIGATION TO PROVIDE
 * MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 *
 */

/* ----------------------------- MNI Header -----------------------------------
@NAME       : help.c
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Functions and variables and stuff related to the gl_mpeg
              help/status window.
@METHOD     : 
@GLOBALS    : BufferNames    (from gl_mpeg.c)
              HelpWindowID   (used by interface.c)
              HelpWin        (the rest are used locally)
	      HelpText
	      StatusFormats
	      StatusText
@CALLS      : 
@CREATED    : 94/7/11, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */

#include <gl/gl.h>
#include "gl_mpeg.h"



/* Global variables the outside world might be interested in: */

int  HelpWindowID = -1;


/* Locally used types */

typedef struct
{
   int   Handle;		/* the window handle */
   long  Width, Height;		/* dimensions of the whole window */
   long  HorizOffset;
   long  CharHeight;
   short HelpX, HelpY;		/* starting point for help text section */
   short StatusX, StatusY;	/* starting point for status text section */
} HelpWinInfo;


/* Global variables of local interest: */

HelpWinInfo  HelpWin = { -1, 0, 0, 0, 0, 0, 0, 0, 0 };

char *HelpText [] = 
{
   " Space     Pause / unpause",
   "*Enter     When paused: single step in current direction",
   "*Backspace When paused: single step in other direction",
   "           When playing: temporarily change current direction of play",
   "*Right     (arrow or mouse button)",
   "           When paused: single step forwards",
   "           When playing: temporarily change current direction to forwards",
   "*Left      (arrow or mouse button)",
   "           When paused: single step backwards",
   "           When playing: temporarily change current direction to backwards",
   "*Home      Rewind to beginning of movie and pause",
   "*End       Fast-forward to end of movie and pause",
   " F         Change play style to forwards only",
   "*B         Change play style to backwards only",
   "*R         Change play style to rock back and forth",
   "           (N.B. plays continuously; ignores once-only mode)",
   " O         Play once-only mode: pause after each playback",
   "           (has no effect if in `rock' mode)",
   " C         Play movie continuously (no automatic pausing)",
   " D         Turn on double-buffering",
   " S         Turn off double-buffering",
   " Z or +    Make movie bigger",
   " X or -    Make movie smaller",
   "*Up        Speed movie up",
   "*Down      Slow movie down",
   " F1        Toggle between full screen and windowed play",
   " ESC, Q    in movie window: quit program",
   "           in help window: close help",
   "    * = option only applicable for buffered playback"
};


char *StatusFormats [] = 
{
   "User-settable options:          Movie status:",
   "  %-28s"                     "    %s",
   "  %-28s"                     "    %s",
   "  Play style: %-16s"         "    direction: %s",
   "  %-28s"                     "    current frame: %d/%d",
   "",
   "General trivia:",
   "  MPEG file: %s",
   "  MPEG dimensions: %dx%d",
   "  Window dimensions: %dx%d (zoom factor: %g)"
};


#define MAX_HELP_LINE_LENGTH 100
#define NUM_HELP_LINES sizeof(HelpText)/sizeof(char *)
#define NUM_STATUS_LINES sizeof(StatusFormats)/sizeof(char *)
  
char *StatusText [NUM_STATUS_LINES];


/* Function prototypes: */
void MakeStatusText (MovieState *Movie);
void PrintTextList (int WinID, 
		    short CharHeight,
		    short X, short Y,
		    char *Text[], int NumItems);
int MaxWidth (int CurMax, char *Text[], int NumItems);
void ShowHelp (HelpWinInfo *Win);
void ShowStatus (HelpWinInfo *Win);
void OpenHelpWindow (MovieState *Movie);
void CloseHelpWindow ();
void WriteHelpWindow (MovieState *Movie);
void UpdateStatus (MovieState *Movie);





/* ----------------------------- MNI Header -----------------------------------
@NAME       : MakeStatusText
@INPUT      : Movie - various flags and settings used to generate status text
@OUTPUT     : StatusText[] (global) - elements set according to the format
                 strings in StatusFormats[] and the settings in Movie
@RETURNS    : (void)
@DESCRIPTION: Uses the flags and settings of Movie to "fill in the blanks"
              of the format strings in StatusFormats to create a bunch
              of status text that can then be printed in whatever way 
	      desired.  Note that the implicit dependencies between
	      the format strings in StatusFormats[] and the fields of
	      MovieState are made explict here, and nowhere else.
@METHOD     : 
@GLOBALS    : StatusFormats (input)
              StatusText (output)
@CALLS      : 
@CREATED    : 94/7/11, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void MakeStatusText (MovieState *Movie)
{
   static 
   Boolean      TextAlloced = FALSE;
   char         tmp [100];
   OptionFlags *Opts = &Movie->Options;
   StatusFlags *Stat = &Movie->Status;

   if (!TextAlloced)
   {
      int   i;

      for (i = 0; i < NUM_STATUS_LINES; i++)
      {
	 StatusText [i] = (char *) malloc (MAX_HELP_LINE_LENGTH);
      }
      TextAlloced = TRUE;
   }

   sprintf (StatusText[0], StatusFormats[0]);
   sprintf (StatusText[1], StatusFormats[1], 
	    Opts->Continuous ? "continuous play" : "once-only play",
	    Stat->Paused     ? "paused"     : "playing");
   sprintf (StatusText[2], StatusFormats[2],
	    Opts->DoubleBuff ? "double-buffered" : "single-buffered",
	    Stat->Decoding   ? "decoding movie" : "playing from buffer");
   sprintf (StatusText[3], StatusFormats[3],
	    StyleNames[Movie->PlayStyle],
	    Stat->Forward    ? "forward" : "backward");	    

   if (BufferType ==  NO_BUFFER)
   {
      strcpy (tmp, "no frame buffering"); 
   }
   else
   {
      sprintf (tmp, "frames buffered to %s", BufferNames[BufferType]);
   }
   sprintf (StatusText[4], StatusFormats[4],
	    tmp, Movie->CurFrame+1, Movie->TotFrames);

   sprintf (StatusText[5], StatusFormats[5]);
   sprintf (StatusText[6], StatusFormats[6]);
   sprintf (StatusText[7], StatusFormats[7],
	    Movie->Filename ? Movie->Filename : "(stdin)");
   sprintf (StatusText[8], StatusFormats[8],
	    Movie->Image.Width, Movie->Image.Height);
   sprintf (StatusText[9], StatusFormats[9],
	    Movie->Window.Width, Movie->Window.Height,
	    Movie->Window.ZoomX);
	    
}     /* MakeStatusText () */



/* ----------------------------- MNI Header -----------------------------------
@NAME       : PrintTextList
@INPUT      : WinID - window to print in (if -1, output goes to stdout
                      and CharHeight, X, and Y are ignored)
              CharHeight - height of characters in the current font
	      X, Y - coordinates to start printing at (each line will 
	             start at X, but Y will be decremented by CharHeight
		     between each line)
              Text[] - array of pointers to strings, each of which will
	               be printed on its own line
              NumItems - number of elements in Text[]
@OUTPUT     : (none)
@RETURNS    : (void)
@DESCRIPTION: Print a list of strings, one per line, to a GL window.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/7/11, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void PrintTextList (int WinID, 
		    short CharHeight,
		    short X, short Y,
		    char *Text[], int NumItems)
{
   long  PrevWin;
   int   i;

   if (WinID != -1)
   {
      PrevWin = winget ();
      winset (WinID);
   }
   
   for (i = 0; i < NumItems; i++)
   {
      if (WinID != -1)
      {
	 cmov2s (X, Y);
	 dprintf ("Printing at (%d,%d): >%s<\n", (int) X, (int) Y, Text[i]);
	 charstr (Text[i]);
	 Y -= (short) CharHeight;
      }
      else
      {
	 puts (Text[i]);
      }
   }

   if (WinID != -1)
   {
      winset (PrevWin);
   }
}


/* ----------------------------- MNI Header -----------------------------------
@NAME       : MaxWidth
@INPUT      : CurMax - the current "length to beat" -- use this if you're
                       using MaxWidth() to find the maximum length across
		       several lists of strings
	      Text[] - list of strings
	      NumItems - number of strings in Text[]
@OUTPUT     : (none)
@RETURNS    : Length of the longest string in Text[]
@DESCRIPTION: Finds the length of the longest string from a list of strings.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/7/12, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
int MaxWidth (int CurMax, char *Text[], int NumItems)
{
   int   i, width;

   for (i = 0; i < NumItems; i++)
   {
      width = strwidth (Text [i]);

      if (width > CurMax)
      {
	 CurMax = width;
      }
   }
   return (CurMax);
}
      


/* ----------------------------- MNI Header -----------------------------------
@NAME       : ShowHelp
@INPUT      : Win - all the trivia about the help window (handle, font
                    size, where to print, etc.)
@OUTPUT     : 
@RETURNS    : (void)
@DESCRIPTION: Prints the text in HelpText to the current help window (or
              to stdout if there is no help window)
@METHOD     : 
@GLOBALS    : HelpText
@CALLS      : 
@CREATED    : 94/7/12, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void ShowHelp (HelpWinInfo *Win)
{
   PrintTextList (Win->Handle, (short) Win->CharHeight, 
		  Win->HelpX, Win->HelpY, 
		  HelpText, NUM_HELP_LINES);
}


/* ----------------------------- MNI Header -----------------------------------
@NAME       : ShowStatus
@INPUT      : Win - all the trivia about the help window (handle, font
                    size, where to print, etc.)
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Prints the text in StatusText to the current help window (or
              to stdout if there is no help window).  Note that you
	      must call MakeStatusText() before calling ShowStatus() 
	      to ensure that the status text exists and is up-to-date.
@METHOD     : 
@GLOBALS    : StatusText
@CALLS      : 
@CREATED    : 94/7/12, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void ShowStatus (HelpWinInfo *Win)
{
   PrintTextList (Win->Handle, (short) Win->CharHeight, 
		  Win->StatusX, Win->StatusY, 
		  StatusText, NUM_STATUS_LINES);
}


/********************************************************************\
 *                                                                  *
 *     Functions called by the outside world be here...             *
 *                                                                  *
\********************************************************************/


/* ----------------------------- MNI Header -----------------------------------
@NAME       : OpenHelpWindow
@INPUT      : Movie - various flags and settings used to generate the
                      status text
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: Sets up the global HelpWin struct, opens a help window,
              and writes the help and status text to it.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/7/11, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void OpenHelpWindow (MovieState *Movie)
{
   dprintf ("Creating status text\n");
   MakeStatusText (Movie);

   HelpWin.Width = 0;
   HelpWin.Width = MaxWidth (0, HelpText, NUM_HELP_LINES);
   HelpWin.Width = MaxWidth (HelpWin.Width, StatusText, NUM_STATUS_LINES);
   HelpWin.CharHeight = getheight ();

   HelpWin.HorizOffset = strwidth ("M");
   HelpWin.Width += HelpWin.HorizOffset*2;
   HelpWin.Height = HelpWin.CharHeight * (NUM_HELP_LINES + NUM_STATUS_LINES)
      + HelpWin.CharHeight;

   prefsize (HelpWin.Width, HelpWin.Height);
   HelpWindowID = HelpWin.Handle = winopen ("MPEG Player Help");
   clear ();
   color (WHITE);

   WriteHelpWindow (Movie);

}     /* OpenHelpWindow () */


/* ----------------------------- MNI Header -----------------------------------
@NAME       : CloseHelpWindow
@INPUT      : Movie->Window - just needs the handle so it can reset current win
@OUTPUT     : (none)
@RETURNS    : 
@DESCRIPTION: Closes the help window, and sets HelpWindowID to -1 so that
              we know the help window is gone.
@METHOD     : 
@GLOBALS    : HelpWin
@CALLS      : 
@CREATED    : 94/7/12, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void CloseHelpWindow ()
{
   winclose (HelpWin.Handle);
   HelpWindowID = HelpWin.Handle = -1;
}


/* ----------------------------- MNI Header -----------------------------------
@NAME       : WriteHelpWindow
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: 
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/7/12, Greg Ward: from code in OpenHelpWindow()
@MODIFIED   : 
---------------------------------------------------------------------------- */
void WriteHelpWindow (MovieState *Movie)
{
   long   WinX, WinY;		/* needed to go from abs->win coords */
   long   PrevWin;

   if (HelpWin.Handle == -1)
   {
      return;
   }

   dprintf ("Filling in help window\n");
   winset (HelpWin.Handle);
   color (BLACK);
   clear ();
   color (WHITE);
   HelpWin.HelpX = HelpWin.HorizOffset;
   HelpWin.HelpY = HelpWin.Height - HelpWin.CharHeight;
   ShowHelp (&HelpWin);

   getcpos (&HelpWin.StatusX, &HelpWin.StatusY);
   getorigin (&WinX, &WinY);
   HelpWin.StatusX -= WinX;
   HelpWin.StatusY -= WinY;
   dprintf ("Current character position (%d,%d)\n",
	    (int) HelpWin.StatusX, (int) HelpWin.StatusY);

   HelpWin.StatusX = HelpWin.HorizOffset;
   HelpWin.StatusY -= (short) HelpWin.CharHeight/2;
   move2s (0, HelpWin.StatusY);
   draw2s ((Icoord) HelpWin.Width-1, HelpWin.StatusY);
   HelpWin.StatusY -= (short) HelpWin.CharHeight;
   ShowStatus (&HelpWin);

   winset (Movie->Window.Handle);
}     /* WriteHelpWindow () */


/* ----------------------------- MNI Header -----------------------------------
@NAME       : UpdateStatus
@INPUT      : 
@OUTPUT     : 
@RETURNS    : 
@DESCRIPTION: 
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 94/7/12, Greg Ward
@MODIFIED   : 
---------------------------------------------------------------------------- */
void UpdateStatus (MovieState *Movie)
{
   int    i;

   static char SaveStatusText [NUM_STATUS_LINES] [MAX_HELP_LINE_LENGTH];
   static Boolean TextSaved = FALSE;

   if (HelpWin.Handle == -1)
   {
      return;
   }

   dprintf ("Updating status text\n");
   winset (HelpWin.Handle);
   MakeStatusText (Movie);

   /* 
    * If we haven't ever been called before, then fill in the whole
    * status window and save all the status text.
    */

   if (!TextSaved)
   {
      ShowStatus (&HelpWin);
      for (i = 0; i < NUM_STATUS_LINES; i++)
      {
	 strcpy (SaveStatusText [i], StatusText [i]);
      }
      TextSaved = TRUE;
   }

   /* 
    * Otherwise, we do things intelligently (i.e. only change/save those
    * lines that have actually changed since the last call).
    */
   else
   {
      for (i = 0; i < NUM_STATUS_LINES; i++)
      {
	 if (strcmp (SaveStatusText [i], StatusText [i]) != 0)
	 {
	    short   Y = HelpWin.StatusY - HelpWin.CharHeight*i;

	    color (BLACK);
	    cmov2s (HelpWin.StatusX, Y);
	    charstr (SaveStatusText [i]);
	    color (WHITE);
	    cmov2s (HelpWin.StatusX, Y);
	    charstr (StatusText [i]);
	    strcpy (SaveStatusText [i], StatusText [i]);
	 }
      }
   }

   winset (Movie->Window.Handle);
}     /* UpdateStatus () */
