/*
  File:	wffConvXY.c
  Authors: K.R. Sloan
  Last Modified: 1 October 1990
  Purpose: convolves a wff image with an arbitrary MxN kernel,
           plus a constant (DC) value;

           The Kernel is passed as a ONE-DIMENSIONAL array, and
           treated as:

             k00 k10 ... k(M-1)0 k01 k11 ... k(M-1)1 k02 ... ... K(M-1)(N-1)

            which is laid out as:

             k00     k10     ...  k(M-1)0
             K01     k11     ...  k(M-1)1
              .       .      ...     .
              .       .      ...     .
             k0(N-1) k1(N-1) ...  k(M-1)(N-1)

           The image is treated as if it were embedded in a 
           field of half-intensity values.  This avoids
           one type of edge effect, but introduces another. 

           Kernel (and DC) values are scaled integers
           (positive or negative.)  The scale factor is 2048.  So,
            2048 == 1.0, 1024 == 0.5, -512 == -0.25, etc.
           
           It is strongly recommended that the individual magnitudes not
           be too large, and that the sum of the values in the template
           (plus the DC value) lie in the range [0,2048] (i.e., [0.0,1.0])
           
           It is also strongly recommended that the individual filters
           have an ODD size.  If the filter width is EVEN, then your image
           will be shifted, as well as filtered!

 */

#include <stdio.h>
#include <wff.h>
#include <sys/file.h>
#include <sys/errno.h>

#define ShiftFactor 11
#define ScaleFactor 2048

static char RoutineName[] = "wffConvXY";

int
wffConvXY(fdIn, fdOut, DC, Kernel, hn, vn)
 FILE *fdIn, *fdOut;
 int DC;
 int *Kernel; 
 int hn, vn;
 {
  FrameBufferType *FBin, *FBout;
  int Bottom, Left, Top, Right, BandsPerPixel, passed;
  char WhatBands[10];
  int BitsPerBand;
  char Name[NameLength], Value[ValueLength];
  unsigned short *t;
  unsigned short **ScanLine;
  unsigned short *NewLine;
  int MaxPixelValue, width, height;
  unsigned short Gray;
  int n;
  int x, y, sx, sy, b;
  int hBorder, vBorder;
  int ScanLinesRead;

  FBin  = (FrameBufferType *)0;
  FBout = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBin))          {                     return FAILURE;}
  if (FAILURE == PassImageIn(fdIn, FBin)){(void)CloseFB(&FBin);return FAILURE;}

  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   { (void)CloseFB(FBin); return FAILURE; }
  if (FAILURE == GetColorSystem(FBin, WhatBands, &BitsPerBand))
   { (void)CloseFB(FBin); return FAILURE; }
  if (FAILURE == GetDescriptor(FBin,"BitsPerBand",Value))
   { (void)CloseFB(FBin); return FAILURE; }
  BitsPerBand = atoi(Value);
  BandsPerPixel = strlen(WhatBands);

  if (FAILURE == OpenFB(&FBout))         {(void)CloseFB(&FBin);return FAILURE;}

  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  passed = 0;
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    SetDescriptor(FBout, Name, Value);
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE; }

  /* Finally, pass the pixels */

  width = Right-Left+1;
  height = Top-Bottom+1;
  MaxPixelValue = (1 << BitsPerBand) - 1;
  Gray = MaxPixelValue >> 1;

  /* 
    if hn,vn are even, shift the image  down and to the right
    this is a completely arbitrary choice.
   */
  hBorder = (hn-1) >> 1; vBorder = (vn-1) >> 1;

  /*
    allocate scanlines and fill input lines with gray
   */
  NewLine = (unsigned short *) calloc(width*BandsPerPixel,
                                      sizeof (unsigned short));
  if ((unsigned short *)0 == NewLine)
   {
    fprintf(stderr,"%s: cannot allocate Scanline[%d]\n",
                   RoutineName,sy);
    (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE;
   }

  ScanLine = (unsigned short **) calloc(vn,
                                        sizeof (unsigned short *));
  if ((unsigned short **)0 == ScanLine)
   {
    fprintf(stderr,"%s: cannot allocate Scanline\n",
                   RoutineName);
    free(NewLine);
    (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE;
   }

  for(sy=0;sy<vn;sy++)
   {
    ScanLine[sy] = (unsigned short *) calloc((width+hn)*BandsPerPixel,
                                             sizeof (unsigned short));
    if ((unsigned short *)0 == ScanLine[sy])
     {
      fprintf(stderr,"%s: cannot allocate Scanline[%d]\n",
                     RoutineName,sy);
      free(NewLine); free(ScanLine);
      { int fy; for (fy = 0; fy < sy; fy++) free(ScanLine[fy]); }
      (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE;
     }
    t = ScanLine[sy];
    for(sx=0;sx<width+hn;sx++)
     for (b=0;b<BandsPerPixel;b++) *t++ = Gray;
   }

  /* preload the pipeline */
  for (sy=vBorder+1,         ScanLinesRead = 0;
       sy<vn;
       sy++,                ScanLinesRead++)
   {
    t = ScanLine[sy];
    for(sx=0;sx<hBorder;sx++)
     for (b=0;b<BandsPerPixel;b++) *t++ = Gray;

    if (FAILURE == NextNPixelsIn(FBin,width,t))
     {
      fprintf(stderr,"%s: failed reading Scanline %d\n",
                     RoutineName, ScanLinesRead);
      free(NewLine); free(ScanLine);
      { int fy; for (fy = 0; fy < vn; fy++) free(ScanLine[fy]); }
      (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE;
     }
    for(sx += width, t += width*BandsPerPixel;
        sx < (width+hn);
        sx++)
     for (b=0;b<BandsPerPixel;b++) *t++ = Gray;
   }

  /* step through destination, one scanline at a time */
  for (y = Bottom; y <= Top; y++)
   {
    /* move forward one line */
    t = ScanLine[0];
    for(sy=1;sy<vn;sy++) ScanLine[sy-1] = ScanLine[sy];
    ScanLine[vn-1] = t;

    if (ScanLinesRead < height)
     {
      for(sx=0;sx<hBorder;sx++)
       for (b=0;b<BandsPerPixel;b++) *t++ = Gray;

      if (FAILURE == NextNPixelsIn(FBin,width,t))
       {
        fprintf(stderr,"%s: failed reading Scanline %d\n",
                       RoutineName, ScanLinesRead);
        free(NewLine); free(ScanLine);
        { int fy; for (fy = 0; fy < vn; fy++) free(ScanLine[fy]); }
        (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE;
       }
      ScanLinesRead++;
      for(sx += width, t += width*BandsPerPixel;
          sx < (width+hn);
          sx++)
       for (b=0;b<BandsPerPixel;b++) *t++ = Gray;
     }
    else
     { /* last lines - pad with gray */
      for(sx=0;sx<(width+hn);sx++)
       for (b=0;b<BandsPerPixel;b++) *t++ = Gray;
     }

    /* convolve */
    {
     register int value;
     register int *k;
     register unsigned short *s, *out;
     register int xStart;

     out = NewLine;
     for (x = Left; x <= Right; x++)
      for ( b = 0,           xStart =(x-Left)*BandsPerPixel;
            b<BandsPerPixel;
            b++,             xStart++)
       {
        value = DC*MaxPixelValue;
        k = Kernel;
        for (sy = 0; sy < vn; sy++)
         {
          s = &ScanLine[sy][xStart];
          for (sx = 0; sx < hn; sx++)
           {
            value += *k * *s;
            k++; s += BandsPerPixel;
           }
         }
        value >>= ShiftFactor;
        if      (value < 0)             value = 0;
        else if (value > MaxPixelValue) value = MaxPixelValue;
        *out++ = (unsigned short) value;
       }
    }
    if (FAILURE == NextNPixelsOut(FBout, width, NewLine))
     {
      free(NewLine); free(ScanLine);
      { int fy; for (fy = 0; fy < vn; fy++) free(ScanLine[fy]); }
      (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE;
     }
   }

  { int fy; for (fy = 0; fy < vn; fy++) free(ScanLine[fy]); }
  (void)CloseFB(&FBin);
  return (CloseFB(&FBout));

 }
