/*
  File: wff2UNC.c
  Author: K.R. Sloan
  Last Modified: 22 April 1991
  Purpose: convert .wff file to UNC
 */

#include <stdio.h>
#include <sys/time.h>
#include "wff.h"
#include "image.h"

int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"usage is:\n\t%s [-h][-v][-f UNC] <wff\n", RoutineName);
  exit(-1);
 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s);
  exit(1);
 }

static void Pass(fdIn, UNCFileName)
 FILE *fdIn;
 char *UNCFileName;
 {
  FrameBufferType *FrameBuffer;
  int Left, Bottom, Top, Right, BitsPerBand;
  int BandsPerPixel;
  char WhatBands[10], Name[NameLength], Value[ValueLength];
  double AspectRatio;
  int width, height;
  unsigned short *WFFScanline;
  int WFFScanlineLength;
  IMAGE *UNCimage;
  int UNCformat;
  int UNCprotection;
  int UNCdimc;
  int UNCdimv[nDIMV];
  unsigned char *UNCScanline;
  int UNCScanlineLength;
  int Corners[2][2];
  int Coarseness[2];
  int x,y;

  FrameBuffer = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FrameBuffer))
   FatalError("OpenFB failed");
  if (FAILURE == ReadImage(fdIn, FrameBuffer))
   FatalError("ReadPassImage failed");
  if (FAILURE == GetBounds(FrameBuffer, &Bottom, &Left, &Top, &Right))
   FatalError("GetBounds failed");
  if (FAILURE == GetColorSystem(FrameBuffer, WhatBands, &BitsPerBand))
   FatalError("GetColorSystem failed");

  if      (0 == strcmp(WhatBands,"I"))
   { BandsPerPixel = 1; UNCformat = GREY;}
  else if (0 == strcmp(WhatBands,"RGBA"))
    {BandsPerPixel = 4; UNCformat = COLORPACKED;}
  else if (0 == strcmp(WhatBands,"RGB"))
   { BandsPerPixel = 3; UNCformat = COLORPACKED;}
  else FatalError("Sorry, we only do I, RGBA, and RGB format");

  width = Right - Left + 1;
  height = Top - Bottom + 1;

  if (VERBOSE)
   fprintf(stderr,"%s: BandsPerPixel = %d; BitsPerBand = %d\n",
     RoutineName, BandsPerPixel, BitsPerBand);

  WFFScanlineLength = width*BandsPerPixel*(sizeof (unsigned short));
  WFFScanline = (unsigned short *) malloc(WFFScanlineLength);
  if ((unsigned short *)0 == WFFScanline)
   FatalError("no memory for WFFScanline");

  switch (UNCformat)
   {
    case GREY:         UNCScanlineLength = width*2; break;
    case COLORPACKED:  UNCScanlineLength = width*4; break;
    default: FatalError("call Key operator");
   }

  UNCScanline = (unsigned char *) malloc(UNCScanlineLength);
  if ((unsigned char *)0 == UNCScanline)
   FatalError("no memory for UNCScanline");

  UNCdimc = 2;
  UNCdimv[0] = height;
  UNCdimv[1] = width;
  UNCprotection = 0666;
  if (VERBOSE) fprintf(stderr,"%s: Creating UNC image file %s\n",
                        RoutineName, UNCFileName);
  UNCimage = imcreat(UNCFileName, UNCprotection,
                      UNCformat,  UNCdimc,   UNCdimv);
  if ((IMAGE *)0 == UNCimage) FatalError("error creating UNC file");

  for (y=Top;y>=Bottom;y--)
   {
    unsigned char *UNC;
    unsigned short *WFF;
 
    if (FAILURE == GetBlock(FrameBuffer,y,Left,y,Right,WFFScanline))
     FatalError("GetBlock failed");

    UNC = UNCScanline; WFF = WFFScanline;
    if(GREY == UNCformat)    
     for(x=Left;x<=Right;x++)
      {
       int p;

       p = *WFF++;
       *UNC++ = (unsigned char)((p>>8) & 0x00ff);
       *UNC++ = (unsigned char)(  p    & 0x00ff);
      } 
    else if ((COLORPACKED == UNCformat) && (BitsPerBand <=  8))
     for(x=Left;x<=Right;x++)
      {
       *UNC++ = (unsigned char)*WFF++;
       *UNC++ = (unsigned char)*WFF++;
       *UNC++ = (unsigned char)*WFF++;
       if (4 == BandsPerPixel)
        *UNC++ = (unsigned char)*WFF++;
       else
        *UNC++ = (unsigned char)255;
      }
    else if(COLORPACKED == UNCformat)    
     for(x=Left;x<=Right;x++)
      {
       *UNC++ = (unsigned char)(*WFF++ >> (BitsPerBand-8));
       *UNC++ = (unsigned char)(*WFF++ >> (BitsPerBand-8));
       *UNC++ = (unsigned char)(*WFF++ >> (BitsPerBand-8));
       if (4 == BandsPerPixel)
        *UNC++ = (unsigned char)(*WFF++ >> (BitsPerBand-8));
       else
        *UNC++ = (unsigned char)255;
      }
    else FatalError("This can't happen");

    Corners[0][0] = height-1-y; Corners[0][1] = height-1-y;
    Corners[1][0] = 0;          Corners[1][1] = width-1;
    Coarseness[0] = 1; Coarseness[1] = 1;
    if (VERBOSE) fprintf(stderr,"%s: writing UNCScanline %d\n",
                         RoutineName, Corners[0][0]);
    if (INVALID == imputpix(UNCimage, Corners, Coarseness, UNCScanline))
     {
      FatalError("writing UNCScanline");
     }
   }

  if (INVALID == imclose(UNCimage)) FatalError("closing UNC image file");
  CloseFB(&FrameBuffer);
  free(WFFScanline); free(UNCScanline);
 }

int main(argc, argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  char *UNCFileName;

  UNCFileName = "wff.UNC";
  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit(-1);}
    switch (argv[ArgsParsed++][1])
     {
      case 'f': if (ArgsParsed >= argc)  { usage(); exit(-1);}
                UNCFileName = argv[ArgsParsed++];
                break;
      case 'v': VERBOSE = -1; break;
      default:
      case 'h':  usage(); exit(-1);
     }
   }

  Pass(stdin, UNCFileName);
  
  exit(0);
 }
