/*
   File: Crop.c
   Author: K.R. Sloan  
   Last Modified: 28 March 1991
   Purpose: Pass a WFF format image file from stdin to stdout
            Making it fit in a window of size Width x Height
            Add a border of grey where needed, and clip where needed
 */
#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <wff.h>

int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,
          "Usage:\n\t%s [-h][-v][-x Width][-y Height]\n",
          RoutineName);

 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s); exit(-1);
 }

static int IsInWindow (xIn, yIn, LeftIn, BottomIn, RightIn, TopIn)
 int xIn, yIn, LeftIn, BottomIn, RightIn, TopIn;
 {
  return (   (LeftIn <= xIn) && (xIn <= RightIn)
          && (BottomIn <= yIn) && (yIn <= TopIn));
 }

static void
Pass(fdIn,fdOut,WidthOut, HeightOut)
 FILE *fdIn, *fdOut;
 int WidthOut, HeightOut;
 {
  FrameBufferType *FBin, *FBout;
  int BottomIn, LeftIn, TopIn, RightIn;
  char Name[NameLength], Value[ValueLength];
  int n;
  unsigned short *ScanLineIn;
  int passed = 0;
  int BottomOut, LeftOut, TopOut, RightOut;
  int WidthIn;
  int xOut, yOut,xIn, yIn, yScanLineIn;
  int xOffset, yOffset;
  unsigned short FillPixel[99];
  int BitsPerBand,b;
  unsigned short Gray;

  if (VERBOSE) fprintf(stderr,"Cropping to %d by %d\n",WidthOut,HeightOut);
  FBin = (FrameBufferType *) 0;
  FBout= (FrameBufferType *) 0;

  if (FAILURE == OpenFB(&FBin))           FatalError("Open FBin failed");
  if (FAILURE == PassImageIn(fdIn, FBin)) FatalError("PassImageIn failed");
  if (FAILURE == OpenFB(&FBout))          FatalError("Open FBout failed");
 
  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    if ( (0 != strcmp(Name,"Name")) && (0 != strcmp(Name,"Title")) )
     SetDescriptor(FBout, Name, Value);
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }


  /* Header operations over, now we can start the output stream */

  GetBounds (FBin, &BottomIn ,&LeftIn ,&TopIn ,&RightIn );  

  WidthIn = RightIn - LeftIn + 1;
  ScanLineIn = (unsigned short *)malloc( WidthIn
                                        *(FBin->BandsPerPixel)
                                        *(sizeof (unsigned short)));
  if ((unsigned short*)0 == ScanLineIn) FatalError("No memory for ScanLineIn");
  yScanLineIn = BottomIn-1;

  BottomOut = 0; LeftOut = 0; TopOut = HeightOut-1; RightOut = WidthOut-1;
  SetBounds (FBout,BottomOut,LeftOut,TopOut,RightOut);

  if (FAILURE == PassImageOut(fdOut, FBout))
   FatalError("PassImageOut failed");


  xOffset = (RightOut + LeftOut - RightIn - LeftIn) / 2;
  yOffset = (TopOut + BottomOut - TopIn - BottomIn) / 2;

  strcpy(Name,"BitsPerBand");
  GetDescriptor(FBin,Name,Value); 
  BitsPerBand = atoi(Value);
  Gray = ((1 << BitsPerBand) >> 1) - 1;
  if (VERBOSE) fprintf(stderr,"Gray = %d\n",Gray);
  for(b=0;b<99;b++) FillPixel[b] = Gray;
  for (yOut=BottomOut;yOut<=TopOut;yOut++)
   {
    yIn = yOut - yOffset; 

    if ((BottomIn <= yIn) && (yIn <= TopIn))
     for(; yScanLineIn < yIn; yScanLineIn++)
      if (FAILURE == NextNPixelsIn(FBin, WidthIn, ScanLineIn))
       FatalError("NextNPixelsIn failed");

    for (xOut=LeftOut;xOut<=RightOut;xOut++)
     {
      xIn = xOut - xOffset;
      if (IsInWindow (xIn, yIn, LeftIn, BottomIn, RightIn, TopIn))
       {
        if (FAILURE == NextPixelOut(FBout,&ScanLineIn[xIn*FBin->BandsPerPixel]))
         {
          (void)CloseFB(&FBin);
          (void)CloseFB(&FBout);
          FatalError("NextPixelOut failed");
         }
       }
      else
       if (FAILURE == NextPixelOut (FBout,FillPixel))
        {
         (void)CloseFB(&FBin);
         (void)CloseFB(&FBout);
          FatalError("NextPixelOut failed");
        }
     }
    wffFlush(FBout);
   }

  (void)CloseFB(&FBin);
  (void)CloseFB(&FBout);

 }

int
main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  int WidthOut = 682, HeightOut = 455;
 
  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) {usage(); exit(-1);}
    switch (argv[ArgsParsed++][1])
     {
      case 'x': if (1 > (argc-ArgsParsed)) {usage(); exit(-1);}
                WidthOut = atoi(argv[ArgsParsed++]);
                break;
      case 'y': if (1 > (argc-ArgsParsed)) {usage(); exit(-1);}
                HeightOut = atoi(argv[ArgsParsed++]);
                break;
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': usage(); exit(-1);
     }
   }
  if (VERBOSE) fprintf(stderr,"%s: cropping to %d by %d\n",
           RoutineName,WidthOut,HeightOut);
  Pass(stdin,stdout, WidthOut, HeightOut); 
  exit(0);
}

