/*
   File: Contour.c
   Author: K.R. Sloan,
   Last Modified: 11 December 1991
   Purpose:  Given an intensity image, produce an image of its contours
 */
#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <wff.h>

static int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s [-h][-v][-c c]\n",
           RoutineName);
 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s);
  exit(-1);
 }

#define ScanLineLength (wffMaxCoordinate+2)

static void Pass(fdIn, fdOut, c)
 FILE *fdIn, *fdOut;
 int c;
 {
  FrameBufferType *FBin, *FBout;
  int Bottom, Left, Top, Right, BandsPerPixel, passed;
  char WhatBands[10], newBands[10];
  int BitsPerBand, newBitsPerBand;
  char Name[NameLength], Value[ValueLength];
  int x, y, n;
  int j, k, l;
  static unsigned short A[ScanLineLength],
                        B[ScanLineLength], 
                        C[ScanLineLength],
                        NewLine[ScanLineLength];
  unsigned short *LastLine, *ThisLine, *NextLine, *p, *q, *r, *t;
  static int Threshold[100];
  int thresh;
  int value;
  int MaxPixelValue, width;
  unsigned short Gray;

  if (VERBOSE) fprintf(stderr,"Pass: c = %d\n",c);

  if (  (c < 1) || (100 < c) ) FatalError("bad value for c");

  FBin  = (FrameBufferType *)0;
  FBout = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBin))           FatalError("Opening input FB");
  if (FAILURE == PassImageIn(fdIn, FBin)) FatalError("Passing image in");

  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   FatalError("in GetBounds");
  if (FAILURE == GetColorSystem(FBin, WhatBands, &BitsPerBand))
   FatalError("in GetColorSystem");

  if (0 != strcmp("I",WhatBands)) FatalError("sorry, only I inputs");

  if (FAILURE == GetDescriptor(FBin,"BitsPerBand",Value))
   FatalError("getting BitsPerBand");
  BitsPerBand = atoi(Value);

  if (FAILURE == OpenFB(&FBout)) FatalError("Opening output FB");

  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  passed = 0;
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    SetDescriptor(FBout, Name, Value);
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  /* we produce a binary image */

  strcpy(newBands,"I"); newBitsPerBand = 1;
  if (FAILURE == SetColorSystem(FBout, newBands, newBitsPerBand))
   FatalError("in SetColorSystem");

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  /* Finally, pass the pixels */


  width = Right-Left+1;
  MaxPixelValue = (1 << BitsPerBand) - 1;
  Gray = MaxPixelValue >> 1;

  for (thresh=0;thresh<c;thresh++)
   {
    Threshold[thresh] = ((thresh+1) * MaxPixelValue) / (c + 1);
    if (VERBOSE) fprintf(stderr,"Threshold[%d] = %d\n",
                 thresh,Threshold[thresh]);
   }

  LastLine = &A[0]; ThisLine = &B[0]; NextLine = &C[0];
  /* load up the three lines line with gray */
  p = LastLine; q = ThisLine; r = NextLine;
  for(x=0;x<width;x++)
   {
     *p++ = Gray; *q++ = Gray; *r++ = Gray;
   }

  /* preload the first line */
  r = NextLine+1;
  if (FAILURE == NextNPixelsIn(FBin, width, r))
    FatalError("reading first line");

  /* step through destination, one pixel at a time */
  for (y = Bottom; y <= Top; y++)
   {
    /* move forward one line */
    t = LastLine; LastLine = ThisLine; ThisLine = NextLine; NextLine = t;

    p = LastLine+1;
    q = ThisLine+1;
    r = NextLine+1;
    if (y<Top)
     {
      if (FAILURE == NextNPixelsIn(FBin, width, r))
       FatalError("reading pixels");
     }
    else
     {  /* last line - pad with gray */
      for(x=0;x<width;x++) *r++ = Gray;
     }
    /* filter the line */
    {
     register unsigned short *pj, *pk, *pl;
     register unsigned short *qj, *qk, *ql;
     register unsigned short *rj, *rk, *rl;
     unsigned short *out;

     pj = p-1; pk = p; pl = p+1;
     qj = q-1; qk = q; ql = q+1;
     rj = r-1; rk = r; rl = r+1;
     out = NewLine+1;
     for (x = Left; x <= Right; x++)
      {
       value = 0;
       for (thresh=0;thresh<c;thresh++)
        {
         int T;

         T = Threshold[thresh];

         if (   (((*qj) <  T) && (T <= (*ql)))
             || (((*qj) >= T) && (T >  (*ql)))
             || (((*pk) <  T) && (T <= (*rk)))
             || (((*pk) >= T) && (T >  (*rk)))
            )
          {
           value = 1; break;
          }
        }
       *out++ = (unsigned short) value;
       pj += 1;  pk += 1;  pl += 1;
       qj += 1;  qk += 1;  ql += 1;
       rj += 1;  rk += 1;  rl += 1;
      }
     p = NewLine+1;
     if (FAILURE == NextNPixelsOut(FBout,width, p))
      FatalError("writing pixel");
    }
   }
  (void)CloseFB(&FBin);
  if (FAILURE == CloseFB(&FBout)) FatalError("Closing output FB");

  return;
 }


int main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  int c = 3;
 
  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) {usage(); exit(-1);}
    switch (argv[ArgsParsed++][1])
     {
      case 'c': if (ArgsParsed >= argc) {usage(); exit(-1);}
                c = atoi(argv[ArgsParsed++]);
                break; 
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': {usage(); exit(-1);}
     }
   }

  Pass(stdin,stdout,c);

  exit (0);
 }


