/*
  File: ApplyMap.c
  Authors:    James Painter
  Last Modified: 11 April 1989
  Purpose: Apply a map to a WFF file
  */
#include <stdio.h>
#include <strings.h>
#include "assert.h"
#include "wff.h"
#include "ColorMap.h"

/* Imports */
extern char *malloc(), *malloc_2d();

/* Data Types */

static char *RoutineName;
static void usage()
{
  fprintf(stderr,"Usage is\n\t%s ColorMapFileName < wff_in  > wff_out\n",
	  RoutineName);
}

static int
  Pass(fdIn,fdOut, map)
FILE *fdIn, *fdOut;
ColorMapType *map;
{
  FrameBufferType *FBin, *FBout;
  int Bottom, Left, Top, Right;
  char Name[NameLength], Value[ValueLength], WhatBands[ValueLength];
  int b, x,y,n;
  unsigned short *Pixel, *p;
  int passed = 0, width;
  int BandsPerPixel, BitsPerBand;
  
  FBin = (FrameBufferType *) 0;
  if (FAILURE == OpenFB(&FBin))           
    {                       return FAILURE; }
  if (FAILURE == PassImageIn(fdIn, FBin)) 
    { (void)CloseFB(&FBin); return FAILURE; }
  FBout = (FrameBufferType *) 0;
  if (FAILURE == OpenFB(&FBout))
    { (void)CloseFB(&FBin); return FAILURE; }
  
  
  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  for (n=0;;n++)
    {
      GetDescriptorN(FBin, n, Name, Value);
      if (Name[0] == '\0') break;
      
      if (0 == strcmp(Name,"X-PassedBy"))
	{
	  if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
	    strcpy(Value,"...");
	  strcat(Value,", "); strcat(Value,RoutineName);
	  passed = 1;
	}
      SetDescriptor(FBout, Name, Value);
    }
  
  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
    {
      strcpy(Name,"X-PassedBy");
      strcpy(Value,RoutineName);
      SetDescriptor(FBout, Name, Value);
    }

  BandsPerPixel = FBin->BandsPerPixel;
  BitsPerBand  = FBin->BitsPerBand;
  
  /* Check that the ColorMap matches the input file */
  if ( (map->BitsIn != BitsPerBand)  ||
       (map->NBands != 1 && map->NBands != BandsPerPixel) )
    {
      fprintf( stderr, "Colormap file is incompatible with input file\n" );
      (void)CloseFB(&FBin);
      (void)CloseFB(&FBout);
      return FAILURE;
    }

  GetColorSystem( FBin, WhatBands, &BitsPerBand );
  SetColorSystem( FBout, WhatBands, map->BitsOut );

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
    { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE; }
  
  /* Finally, pass the pixels */
  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
    { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE; }
  width = Right-Left+1;
  
  Pixel = (unsigned short *) 
    malloc( sizeof(unsigned short) * (unsigned)(width * BandsPerPixel) );
  if ((unsigned short *)0 == Pixel)
    {
      fprintf( stderr, "No memory for scan line buffer!\n" );
      (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE;
    }
  
  for (y=Bottom;y<=Top;y++)
    {
      if (FAILURE == NextNPixelsIn(FBin, width, Pixel))
	{  
	  fprintf(stderr,"%s: NextNPixelsIn failed\n", RoutineName);        
	  (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE; 
	}

      /* Apply the map */
      p = Pixel;
      if (map->NBands == 1) 
	{
	  for(x=0; x < width; x++) 
	    for(b=0; b<BandsPerPixel; b++) 
	      {	*p = map->map[*p][0];	p++; }
	} 
      else 
	{
	  for(x=0; x < width; x++) 
	    for(b=0; b<BandsPerPixel; b++) 
	      {	*p = map->map[*p][b];   p++; }
	}
      if (FAILURE == NextNPixelsOut(FBout, width, Pixel))
	{
	  fprintf(stderr,"%s: NextNPixelsOut failed\n", RoutineName);        
	  (void)CloseFB(&FBin); (void)CloseFB(&FBout); return FAILURE; 
	}
      wffFlush(FBout);
    }       
  
  (void)CloseFB(&FBin);
  (void)CloseFB(&FBout);
  return SUCCESS;
}



int
  main(argc,argv)
int argc;
char *argv[];
{
  int ArgsParsed = 0;
  ColorMapType *map;
  char *ColorMapFileName;
  FILE *file;

  RoutineName = argv[ArgsParsed++];

  if (ArgsParsed >= argc) { usage(); exit (-1); }

  ColorMapFileName = argv[ArgsParsed++];
  file = fopen( ColorMapFileName, "r" );
  if (file == (FILE *) 0) {
    fprintf( stderr, "Unable to open colormap file: %s\n", ColorMapFileName );
    usage();
    exit(-1);
  }

  if (ArgsParsed < argc)  { usage(); exit (-1); }

  if ( (map=ReadColorMap(file)) && 
       (Pass(stdin, stdout, map) == SUCCESS) )
    exit(0);
  else
    exit(-1);
}










