/* Window creation, deletion and examination for GNU Emacs.
   Does not include redisplay.
   Copyright (C) 1985, 1986, 1987, 1990 Free Software Foundation, Inc.

This file is part of GNU Emacs.

GNU Emacs is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GNU Emacs is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Emacs; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/*
 * $Revision: 1.32 $
 * $Source: /import/kaplan/stable/distrib/epoch-4.0p0/src/RCS/window.c,v $
 * $Date: 92/03/19 16:30:00 $
 * $Author: love $
 */
#ifndef LINT
static char rcsid[] = "$Author: love $ $Date: 92/03/19 16:30:00 $ $Source: /import/kaplan/stable/distrib/epoch-4.0p0/src/RCS/window.c,v $ $Revision: 1.32 $";
#endif

#include "config.h"
#include "lisp.h"
#include "buffer.h"
#include "window.h"
#include "commands.h"
#include "indent.h"
#include "termchar.h"

#include "dispepoch.h"		/* Epoch */
#include "button.h"		/* Epoch */
#include "screen.h"		/* Epoch */
#include "screenW.h"		/* Epoch */

extern int distinct_minibuffer;

Lisp_Object Qwindowp;

Lisp_Object Fnext_window (), Fdelete_window (), Fselect_window ();
Lisp_Object Fset_window_buffer (), Fsplit_window (), Frecenter ();

static void replace_window(), unshow_buffer();
static int save_window_save();

extern int minibuf_prompt_width;

/* This is the window which displays the minibuffer.
It is always the same window.  */

Lisp_Object minibuf_window;

/* This is the window in which the terminal's cursor should
 be left when nothing is being done with it.  This must
 always be a leaf window, and its buffer is selected by
 the top level editing loop at the end of each command.  */

Lisp_Object selected_window;

/* Non-nil means it is the window for C-M-v to scroll
   when the minibuffer is selected.  */

Lisp_Object Vminibuf_scroll_window;

/* Non-nil means it's function to call to display temp buffers.  */

Lisp_Object Vtemp_buffer_show_hook;

/* If a window gets smaller than either of these, it is removed. */

int window_min_height;		/* Now in pixels */
int window_min_width;		/* Now in pixels */

/* Nonzero implies pop_to_buffer should create windows. */

int pop_up_windows;

/* display-buffer always splits the largest window 
 if that window is more than this high */

int split_height_threshold;

/* Number of lines of continuity in scrolling by screenfuls.  */

int next_screen_context_lines;

/* Incremented for each window created.  */

static int sequence_number;

DEFUN ("windowp", Fwindowp, Swindowp, 1, 1, 0,
  "Returns t if OBJ is a window.")
  (obj)
     Lisp_Object obj;
{
  return XTYPE (obj) == Lisp_Window ? Qt : Qnil;
}

Lisp_Object
make_window ()
{
  register Lisp_Object val;
  register struct window *p;

  /* Add sizeof (Lisp_Object) here because sizeof (struct Lisp_Vector)
     includes the first element.  */
  val = Fmake_vector (
    make_number ((sizeof (struct window) - sizeof (struct Lisp_Vector)
		  + sizeof (Lisp_Object))
		 / sizeof (Lisp_Object)),
    Qnil);
  XSETTYPE (val, Lisp_Window);
  p = XWINDOW (val);
  XFASTINT (p->sequence_number) = ++sequence_number;
  XFASTINT (p->pixleft) = XFASTINT (p->pixtop)
    = XFASTINT (p->pixheight) = XFASTINT (p->pixwidth)
      = XFASTINT (p->hscroll) = 0;
  XFASTINT (p->last_point_x) = XFASTINT (p->last_point_y) = 0;
  p->start = Fmake_marker ();
  p->pointm = Fmake_marker ();
  p->size_change = Qt;
  XSET(p->lines,Lisp_Raw_Data,get_line());
  XSET(p->modeline,Lisp_Raw_Data,get_line());
  XLINE(p->modeline)->modeline = 1;
  XFASTINT (p->use_time) = 0;
  return val;
}

/* Epoch:  return 1 if we find W in window hierarchy WIN. */
find_window(w,win)
     struct window *w;
     Lisp_Object win;
{
  int i;
  for (; !NULL(win) && !EQ(win,XROOT(XWINDOW(win)->root)->minibuf_window)
       ; win = XWINDOW(win)->next )
    {
      if (w == XWINDOW(win)) return 1;
      if (!NULL(XWINDOW(win)->vchild) &&
	  (i = find_window(w,XWINDOW(win)->vchild)))
	return i;
      if (!NULL(XWINDOW(win)->hchild) &&
	  (i = find_window(w,XWINDOW(win)->hchild)))
	return i;
    }
  return 0;
}


DEFUN ("selected-window", Fselected_window, Sselected_window, 0, 0, 0,
  "Return the window that the cursor now appears in and commands apply to.")
  ()
{
  return selected_window;
}

DEFUN ("minibuffer-window", Fminibuffer_window, Sminibuffer_window, 0, 0, 0,
  "Return the window used for minibuffers.")
  ()
{
  return minibuf_window;
}

DEFUN ("pos-visible-in-window-p", Fpos_visible_in_window_p,
  Spos_visible_in_window_p, 0, 2, 0,
  "Return t if position POS is currently on the screen in WINDOW.\n\
Returns nil if that position is scrolled vertically out of view.\n\
POS defaults to point; WINDOW, to the selected window.")
  (pos, window)
     Lisp_Object pos, window;
{
  register struct window *w;
  struct Root_Block *rb;
  register int top;
  register int height;
  register int posint;
  register struct buffer *buf;
  struct position posval;

  if (NULL (pos))
    posint = point;
  else
    {
      CHECK_NUMBER_COERCE_MARKER (pos, 0);
      posint = XINT (pos);
    }

  if (NULL (window))
    window = selected_window;
  else
    CHECK_WINDOW (window, 1);
  w = XWINDOW (window);
  rb = XROOT(w->root);
  top = marker_position (w->start);

  if (posint < top)
    return Qnil;

  height = window_char_height(w) - !EQ(window,rb->minibuf_window);
  
  buf = XBUFFER (w->buffer);
  if (XFASTINT (w->last_modified) >= BUF_MODIFF (buf))
    {
      /* If screen is up to date,
	 use the info recorded about how much text fit on it. */
      if (posint < BUF_Z (buf) - XFASTINT (w->window_end_pos)
	  || (XFASTINT (w->height) < height))
	return Qt;
      return Qnil;
    }
  else
    {
      if (posint > BUF_Z (buf))
	return Qnil;
      /* If that info is not correct, calculate afresh */
      /* New interface */
      posval = *compute_motion (top, 0, 0,
				posint, height, 0,
				XINT(w->hscroll),0,
				w,w->start_button);
      return posval.vpos < height ? Qt : Qnil;
    }
}

static struct window *
decode_window (window)
     register Lisp_Object window;
{
  if (NULL (window))
    return XWINDOW (selected_window);

  CHECK_WINDOW (window, 0);
  return XWINDOW (window);
}

DEFUN ("window-buffer", Fwindow_buffer, Swindow_buffer, 0, 1, 0,
  "Return the buffer that WINDOW is displaying.")
  (window)
     Lisp_Object window;
{
  return decode_window (window)->buffer;
}

DEFUN ("window-height", Fwindow_height, Swindow_height, 0, 1, 0,
  "Return the number of lines in WINDOW (including its mode line).")
  (window)
     Lisp_Object window;
{
  return window_char_height(decode_window(window));
}

DEFUN ("window-pixheight", Fwindow_pixheight, Swindow_pixheight, 0, 1, 0,
       "Return the height of WINDOW in pixels.  Defaults to current window.")
     (window) Lisp_Object window;
{
  return decode_window(window)->pixheight;
}

DEFUN ("window-width", Fwindow_width, Swindow_width, 0, 1, 0,
  "Return the number of columns in WINDOW.")
  (window)
     Lisp_Object window;
{
  return window_char_width(decode_window(window));
}

DEFUN ("window-pixwidth", Fwindow_pixwidth, Swindow_pixwidth, 0, 1, 0,
       "Return the width of WINDOW in pixels.  Defaults to current window.")
     (window) Lisp_Object window;
{
  return decode_window(window)->pixwidth;
}

DEFUN ("window-hscroll", Fwindow_hscroll, Swindow_hscroll, 0, 1, 0,
  "Return the number of columns by which WINDOW is scrolled from left margin.")
  (window)
     Lisp_Object window;
{
  return decode_window (window)->hscroll;
}

DEFUN ("set-window-hscroll", Fset_window_hscroll, Sset_window_hscroll, 2, 2, 0,
  "Set number of columns WINDOW is scrolled from left margin to NCOL.\n\
NCOL should be zero or positive.")
  (window, ncol)
     register Lisp_Object window, ncol;
{
  register struct window *w;

  CHECK_NUMBER (ncol, 1);
  if (XINT (ncol) < 0) XFASTINT (ncol) = 0;
  if (XFASTINT (ncol) >= (1 << (SHORTBITS - 1)))
    args_out_of_range (ncol, Qnil);
  w = decode_window (window);
  if (w->hscroll != ncol)
    clip_changed = 1;		/* Prevent redisplay shortcuts */
  w->hscroll = ncol;

  return ncol;
}

DEFUN ("window-edges", Fwindow_edges,Swindow_edges,0,1,0,
       "Return a list of the edge coordinates of WINDOW.\n\
(LEFT TOP RIGHT BOTTOM), all relative to 0, 0 at top left corner of screen.\n\
RIGHT is one more than the rightmost column used by WINDOW,\n\
and BOTTOM is one more than the bottommost row used by WINDOW\n\
and its mode-line.")
  (window)
{
  register struct window *w = decode_window(window);
  register struct Root_Block *rb = XROOT(w->root);
  WS_DEF;
  register int l = XFASTINT(w->pixleft) - ws->in_border;
  register int t = XFASTINT(w->pixtop) - ws->in_border;
  struct X_font *font  = XXFONT(XSTYLE(rb->stylenorm)->font);
  
  l = l / XFWIDTH(font);
  t = t / XFHEIGHT(font);

  return Fcons (make_number(l),
		Fcons (make_number(t),
		       Fcons(window_char_width(w),
			     Fcons(window_char_height(w),
				   Qnil))));
}

DEFUN ("window-pixedges", Fwindow_pixedges, Swindow_pixedges, 0, 1, 0,
  "Return a list of the edge coordinates of WINDOW.\n\
\(LEFT TOP RIGHT BOTTOM MODE), all relative to 0, 0 at top left corner of screen.\n\
RIGHT is one more than the rightmost column used by WINDOW,\n\
BOTTOM is one more than the bottommost row used by WINDOW\n\
and its mode-line.  MODE is pixel top of modeline")
  (window)
     Lisp_Object window;
{
  register struct window *w = decode_window (window);
  register int mode = !NULL(w->modeline) ? XLINE(w->modeline)->ascent +
    XLINE(w->modeline)->descent : 0;
  
  return Fcons (w->pixleft, Fcons (w->pixtop,
           Fcons (make_number (XFASTINT (w->pixleft) + XFASTINT (w->pixwidth)),
		  Fcons (make_number (XFASTINT (w->pixtop)
				      + XFASTINT (w->pixheight)),
			 Fcons (make_number (XFASTINT(w->pixtop)
					     + XFASTINT (w->pixheight)
					     - mode),
				Qnil)))));
}

DEFUN ("window-point", Fwindow_point, Swindow_point, 0, 1, 0,
  "Return current value of point in WINDOW.\n\
For a nonselected window, this is the value point would have\n\
if that window were selected.\n\
\n\
Note that, when WINDOW is the selected window and its buffer\n\
is also currently selected, the value returned is the same as (point).\n\
It would be more strictly correct to return the `top-level' value\n\
of point, outside of any  save-excursion  forms.\n\
But that is hard to define.")
  (window)
     Lisp_Object window;
{
  register struct window *w = decode_window (window);

  if (w == XWINDOW (selected_window)
      && current_buffer == XBUFFER (w->buffer))
    return Fpoint ();
  return Fmarker_position (w->pointm);
}

DEFUN ("window-start", Fwindow_start, Swindow_start, 0, 1, 0,
  "Return position at which display currently starts in WINDOW.")
  (window)
     Lisp_Object window;
{
  return Fmarker_position (decode_window (window)->start);
}

DEFUN ("set-window-point", Fset_window_point, Sset_window_point, 2, 2, 0,
  "Make point value in WINDOW be at position POS in WINDOW's buffer.")
  (window, pos)
     Lisp_Object window, pos;
{
  register struct window *w = decode_window (window);

  CHECK_NUMBER_COERCE_MARKER (pos, 1);
  if (w == XWINDOW (selected_window))
    Fgoto_char (pos);
  else
    set_marker_restricted (w->pointm, pos, w->buffer);
  return pos;
}

DEFUN ("set-window-start", Fset_window_start, Sset_window_start, 2, 3, 0,
  "Make display in WINDOW start at position POS in WINDOW's buffer.\n\
Optional third arg NOFORCE non-nil inhibits next redisplay\n\
from overriding motion of point in order to display at this exact start.")
  (window, pos, noforce)
     Lisp_Object window, pos, noforce;
{
  register struct window *w = decode_window (window);

  CHECK_NUMBER_COERCE_MARKER (pos, 1);
  w->start_button = Qnil;
  set_marker_restricted (w->start, pos, w->buffer);
  /* this is not right, but much easier than doing what is right.  */
  w->start_at_line_beg = Qnil;
  if (NULL (noforce))
    w->force_start = Qt;
  w->update_mode_line = Qt;
  XFASTINT (w->last_modified) = 0;
  return pos;
}

DEFUN ("delete-window", Fdelete_window, Sdelete_window, 0, 1, "",
  "Remove WINDOW from the display.  Default is selected window.")
  (window)
     register Lisp_Object window;
{
  int osize;
  register Lisp_Object tem, parent;
  register struct window *p;
  register struct window *par;

  if (NULL (window))
    window = selected_window;
  else
    CHECK_WINDOW (window, 0);

  p = XWINDOW (window);
  parent = p->parent;
  if (NULL (parent))
    error ("Attempt to delete minibuffer or sole ordinary window");
  par=XWINDOW (parent);

  windows_or_buffers_changed++;

  if (EQ (window, selected_window))
    Fselect_window (Fnext_window (window, Qnil));

  tem = p->buffer;
  /* tem is null for dummy parent windows
     (which have inferiors but not any contents themselves) */
  if (!NULL (tem))
    {
      unshow_buffer (p);
      unchain_marker (p->pointm);
      unchain_marker (p->start);
      clear_display_structs(window);
    }

  tem = p->next;
  if (!NULL (tem))
    XWINDOW (tem)->prev = p->prev;

  tem = p->prev;
  if (!NULL (tem))
    XWINDOW (tem)->next = p->next;

  if (EQ (window, par->hchild))
    par->hchild = p->next;
  if (EQ (window, par->vchild))
    par->vchild = p->next;

  /* Epoch:
   * Window has been deleted.  Now update root block pointer if this window
   * was the attachment.
   */
  if (XROOT(p->root)->ewin == window)
    XROOT(p->root)->ewin = parent;
    
  /* Stretch the siblings to use all the available space */
  if (!NULL (par->vchild))
    {
      /* It's a vertical combination */
      osize = XFASTINT (par->pixheight);
      XFASTINT (par->pixheight)
	-= XFASTINT (p->pixheight);
      set_window_height (parent, osize, 1);
    }
  if (!NULL (par->hchild))
    {
      /* It's a horizontal combination */
      osize = XFASTINT (par->pixwidth);
      XFASTINT (par->pixwidth)
	-= XFASTINT (p->pixwidth);
      set_window_width (parent, osize, 1);
    }

  /* If parent now has only one child,
     put the child into the parent's place.  */

  tem = par->hchild;
  if (NULL (tem))
    tem = par->vchild;
  if (NULL (XWINDOW (tem)->next))
    replace_window (parent, tem);
  return Qnil;
}

void				/* Epoch */
destroy_window_tree(win)
	Lisp_Object win;
{
  struct Root_Block *rb = XROOT(XWINDOW(win)->root);
  struct window *w;
  for (; !NULL(win) && !EQ(win,rb->minibuf_window) ; win = XWINDOW(win)->next)
    {
      w = XWINDOW(win);
      if (!NULL(w->hchild)) destroy_window_tree(w->hchild);
      if (!NULL(w->vchild)) destroy_window_tree(w->vchild);
      if (!NULL(w->buffer))
	{
	  unshow_buffer(w);
	  unchain_marker(w->pointm);
	  unchain_marker(w->start);
	  clear_display_structs(win);
	}
    }
}

/* Put replacement into the window structure in place of old. */
static void
replace_window (old, replacement)
     Lisp_Object old, replacement;
{
  register Lisp_Object tem;
  register struct window *o = XWINDOW (old), *p = XWINDOW (replacement);

  p->pixleft = o->pixleft;
  p->pixtop = o->pixtop;
  p->pixwidth = o->pixwidth;
  p->pixheight = o->pixheight;

  p->next = tem = o->next;
  if (!NULL (tem))
    XWINDOW (tem)->prev = replacement;

  p->prev = tem = o->prev;
  if (!NULL (tem))
    XWINDOW (tem)->next = replacement;

  p->parent = tem = o->parent;
  if (!NULL (tem))
    {
      if (EQ (XWINDOW (tem)->vchild, old))
	XWINDOW (tem)->vchild = replacement;
      if (EQ (XWINDOW (tem)->hchild, old))
	XWINDOW (tem)->hchild = replacement;
    }

  /* Epoch:
   * If old window was root block attachment, fix the window pointer
   */
  if (XROOT(o->root)->ewin == old)
    XROOT(o->root)->ewin = replacement;

/*** Here, if replacement is a vertical combination
and so is its new parent, we should make replacement's
children be children of that parent instead.  ***/
}

DEFUN ("next-window", Fnext_window, Snext_window, 0, 2, 0,
  "Return next window after WINDOW in canonical ordering of windows.\n\
Optional second arg MINIBUF t means count the minibuffer window\n\
even if not active.  If MINIBUF is neither t nor nil it means\n\
not to count the minibuffer even if it is active.")
  (window, mini)
     register Lisp_Object window, mini;
{
  register Lisp_Object tem;
  struct Root_Block *rb;
  if (NULL (window))
    window = selected_window;
  else
    CHECK_WINDOW (window, 0);
  rb = XROOT(XWINDOW(window)->root);
  if (rb == mini_root) rb = XROOT(EDIT_SCREEN);
  
  do
    {
      while (tem = XWINDOW (window)->next, NULL (tem))
	if (tem = XWINDOW (window)->parent, !NULL (tem))
	  window = tem;
        else  /* window must be minibuf_window now */
	  {
	    tem = rb->ewin;
	    while (!NULL(XWINDOW(tem)->parent)) tem = XWINDOW(tem)->parent;
	    rb->ewin = tem;
	    break;
	  }
      window = tem;
      while (1)
	{
	  if (!NULL (XWINDOW (window)->hchild))
	    window = XWINDOW (window)->hchild;
	  else if (!NULL (XWINDOW (window)->vchild))
	    window = XWINDOW (window)->vchild;
	  else break;
	}
    }
  while (EQ (window, rb->minibuf_window) && !EQ (mini, Qt)
	 && (!NULL (mini) || minibuf_level == 0));
  return window;
}

DEFUN ("previous-window", Fprevious_window, Sprevious_window, 0, 1, 0,
  "Return previous window before WINDOW in canonical ordering of windows.")
  (window)
     register Lisp_Object window;
{
  struct Root_Block *rb;
  register Lisp_Object tem;
  
  if (NULL (window))
    window = selected_window;
  else
    CHECK_WINDOW (window, 0);
  rb = XROOT(XWINDOW(window)->root);  
  do  /* at least once, and until not the minibuffer */
    {
      while (tem = XWINDOW (window)->prev, NULL (tem))
	if (tem = XWINDOW (window)->parent, !NULL (tem))
	  window = tem;
        else  /* window must be the root window now */
	  {
	    tem = rb->minibuf_window;
	    break;
	  }
      window = tem;
      while (1)
	{
	  if (!NULL (XWINDOW (window)->hchild))
	    window = XWINDOW (window)->hchild;
	  else if (!NULL (XWINDOW (window)->vchild))
	    window = XWINDOW (window)->vchild;
	  else break;
	  while (tem = XWINDOW (window)->next, !NULL (tem))
	    window = tem;
	}
    }
  while (EQ (window, rb->minibuf_window) && minibuf_level == 0);
  return window;
}

DEFUN ("other-window", Fother_window, Sother_window, 1, 1, "p",
  "Select the ARG'th different window.")
  (n)
     register Lisp_Object n;
{
  register int i;
  register Lisp_Object w;

  CHECK_NUMBER (n, 0);
  w = selected_window;
  i = XINT (n);

  while (i > 0)
    {
      w = Fnext_window (w, Qnil);
      i--;
    }
  while (i < 0)
    {
      w = Fprevious_window (w);
      i++;
    }
  Fselect_window (w);
  return Qnil;
}

static Lisp_Object
window_loop (type, obj)
     int type;
     register Lisp_Object obj;
{
  register Lisp_Object w, tem, ret_w;
  Lisp_Object w1, start_w;
  register struct window *p, *q;

  w = cur_root->minibuf_window;
  ret_w = Qnil;
  while (1)
    {
      p = XWINDOW (w);
      w1 = Fnext_window (w, Qt);
      if (!EQ (w, cur_root->minibuf_window))
	switch (type)
	  {
	  case 1:
	    if (XBUFFER (p->buffer) == XBUFFER (obj))
	      return w;
	    break;

	  case 2:
	    /* t as arg means consider only full-width windows */
	    if (!NULL (obj) && XFASTINT (p->pixwidth) != WS->pixw)
	      break;
	    if (NULL (ret_w) ||
		XFASTINT (XWINDOW (ret_w)->use_time) > XFASTINT (p->use_time))
	      ret_w = w;
	    break;

	  case 3:
	    if (p != XWINDOW (obj))
	      Fdelete_window (w);
	    break;

	  case 4:
	    if (EQ (p->buffer, obj))
	      {
		if (NULL (p->parent))
		  {
		    tem = Fother_buffer (obj);
		    if (NULL (tem))
		      tem = Fget_buffer_create (build_string ("*scratch*"));
		    Fset_window_buffer (w, tem);
		    Fset_buffer (p->buffer);
		  }
		else
		  Fdelete_window (w);
	      }
	    break;

	  case 5:
	    q = XWINDOW (ret_w);
	    if (NULL (ret_w) ||
		(XFASTINT (p->pixheight) * XFASTINT (p->pixwidth))
		>
		(XFASTINT (q->pixheight) * XFASTINT (q->pixwidth)))
	      ret_w = w;
	    break;

	  case 6:
	    if (EQ (p->buffer, obj))
	      {
		tem = Fother_buffer (obj);
		if (NULL (tem))
		  tem = Fget_buffer_create (build_string ("*scratch*"));
		Fset_window_buffer (w, tem);
	      }
	    break;
	  }
      w = w1;
      if (EQ (w, cur_root->minibuf_window))
	return ret_w;
    }
}     

DEFUN ("get-lru-window", Fget_lru_window, Sget_lru_window, 0, 0, 0,
  "Return the window least recently selected or used for display.")
  ()
{
  register Lisp_Object w;
  /* First try for a window that is full-width */
  w = window_loop (2, Qt);
  if (!NULL (w) && !EQ (w, selected_window))
    return w;
  /* If none of them, try the rest */
  return window_loop (2, Qnil);
}

DEFUN ("get-largest-window", Fget_largest_window, Sget_largest_window, 0, 0, 0,
  "Return the largest window in area.")
  ()
{
  return window_loop (5, Qnil);
}

DEFUN ("get-buffer-window", Fget_buffer_window, Sget_buffer_window, 1, 1, 0,
  "Return a window currently displaying BUFFER, or nil if none.")
  (buffer)
     Lisp_Object buffer;
{
  buffer = Fget_buffer (buffer);
  if (XTYPE (buffer) == Lisp_Buffer)
    return window_loop (1, buffer);
  else return Qnil;
}

DEFUN ("delete-other-windows", Fdelete_other_windows, Sdelete_other_windows,
  0, 1, "",
  "Make WINDOW (or the selected window) fill the screen.")
  (w)
     Lisp_Object w;
{
  window_loop (3, !NULL (w) ? w : selected_window);
  return Qnil;
}

DEFUN ("delete-windows-on", Fdelete_windows_on, Sdelete_windows_on,
  1, 1, "bDelete windows on (buffer): ",
  "Delete all windows showing BUFFER.")
  (buffer)
     Lisp_Object buffer;
{
  if (!NULL (buffer))
    {
      buffer = Fget_buffer (buffer);
      CHECK_BUFFER (buffer, 0);
      window_loop (4, buffer);
    }
  return Qnil;
}

static Lisp_Object		/* Epoch */
replace_buffer_in_window(w,obj)
	Lisp_Object w;
	Lisp_Object obj;
{
  struct Root_Block *rb = XROOT(XWINDOW(w)->root);
  Lisp_Object tem;
  struct window *p;

  for ( ; !NULL(w) && !EQ(w,rb->minibuf_window) ; w = XWINDOW(w)->next )
    {
      p = XWINDOW(w);
      if (EQ (p->buffer, obj))
	{
	  tem = Fother_buffer (obj);
	  if (NULL (tem))
	    tem = Fget_buffer_create (build_string ("*scratch*"));
	  Fset_window_buffer (w, tem);
	  Fset_buffer (p->buffer);
	  p->start_button = Qnil;
	  Fset_screen_modified(XWINDOW(w)->root);
	}
      if (!NULL(p->vchild)) replace_buffer_in_window(p->vchild,obj);
      if (!NULL(p->hchild)) replace_buffer_in_window(p->hchild,obj);
    }
  return Qnil;			/* indicate not done yet */
}

DEFUN ("replace-buffer-in-windows", Freplace_buffer_in_windows,
  Sreplace_buffer_in_windows,
  1, 1, "bReplace buffer in windows: ",
  "Replace BUFFER with some other buffer in all windows showing it.")
  (buffer)
     Lisp_Object buffer;
{
  if (!NULL (buffer))
    {
      buffer = Fget_buffer (buffer);
      CHECK_BUFFER (buffer, 0);
      do_windows_of_screens(replace_buffer_in_window,buffer);      

    }
  return Qnil;
}

/* Set the height of WINDOW and all its inferiors.  */
/* Normally the window is deleted if it gets too small.
   nodelete nonzero means do not do this.
   (The caller should check later and do so if appropriate)  */

set_window_height (window, height, nodelete)
     Lisp_Object window;
     int height;
     int nodelete;
{
  register struct window *w = XWINDOW (window);
  struct Root_Block *rb = XROOT(w->root);
  register struct window *c;
  int oheight = XFASTINT (w->pixheight);
  int top, pos, lastbot, opos, lastobot;
  Lisp_Object child;
  struct buffer *b = XBUFFER(w->buffer);
  struct X_font *font = XXFONT(STYLE_FIELD(w,font,stylenorm));
  
  if (window_min_height < 2)
    window_min_height = 2;

  if (nodelete == 0 &&
      ! NULL (w->parent) &&
      height < (((w == XWINDOW(rb->minibuf_window) ? 1 : window_min_height))
		* XFHEIGHT(font)))
    {
      Fdelete_window (window);
      return;
    }

  if (!((nodelete & 2) == 2))
    {
      XFASTINT (w->last_modified) = 0;
      w->size_change = Qt;
      windows_or_buffers_changed++;
    }
  XFASTINT (w->pixheight) = height;
  w->window_end_valid = Qnil;
  if (!NULL (w->hchild))
    {
      for (child = w->hchild; !NULL (child); child = XWINDOW (child)->next)
	{
	  XWINDOW (child)->pixtop = w->pixtop;
	  set_window_height (child, height, nodelete);
	}
    }
  else if (!NULL (w->vchild))
    {
      lastbot = top = XFASTINT (w->pixtop);
      lastobot = 0;
      for (child = w->vchild; !NULL (child); child = c->next)
	{
	  c = XWINDOW (child);

	  opos = lastobot + XFASTINT (c->pixheight);

	  XFASTINT (c->pixtop) = lastbot;

	  pos = (((opos * height) << 1) + oheight) / (oheight << 1);

	  /* Avoid confusion: inhibit deletion of child if becomes too small */
	  set_window_height (child, pos + top - lastbot, 1 | nodelete);

	  /* Now advance child to next window,
	     and set lastbot if child was not just deleted.  */
	  lastbot = pos + top, lastobot = opos;
	}
      /* Now delete any children that became too small.  */
      if (nodelete == 0)
	for (child = w->vchild; !NULL (child); child = XWINDOW (child)->next)
	  {
	    set_window_height (child, XINT (XWINDOW (child)->pixheight), 0);
	  }
    }
}

/* Recursively set width of WINDOW and its inferiors. */

set_window_width (window, width, nodelete)
     Lisp_Object window;
     int width;
     int nodelete;
{
  register struct window *w = XWINDOW (window);
  register struct window *c;
  int owidth = XFASTINT (w->pixwidth);
  int left, pos, lastright, opos, lastoright;
  Lisp_Object child;
  struct buffer *b = XBUFFER(w->buffer);
  struct X_font *font = XXFONT(STYLE_FIELD(w,font,stylenorm));

  if (nodelete == 0
      && ! NULL (w->parent)
      && width < window_min_width * XFWIDTH(font))
    {
      Fdelete_window (window);
      return;
    }

  if (!((nodelete & 2) == 2))
    {
      XFASTINT (w->last_modified) = 0;
      w->size_change = Qt;
      windows_or_buffers_changed++;
    }
  XFASTINT (w->pixwidth) = width;
  if (!NULL (w->vchild))
    {
      for (child = w->vchild; !NULL (child); child = XWINDOW (child)->next)
	{
	  XWINDOW (child)->pixleft = w->pixleft;
	  set_window_width (child, width, nodelete);
	}
    }
  else if (!NULL (w->hchild))
    {
      lastright = left = XFASTINT (w->pixleft);
      lastoright = 0;
      for (child = w->hchild; !NULL (child); child = c->next)
	{
	  c = XWINDOW (child);

	  opos = lastoright + XFASTINT (c->pixwidth);

	  XFASTINT (c->pixleft) = lastright;

	  pos = (((opos * width) << 1) + owidth) / (owidth << 1);

	  /* Inhibit deletion for becoming too small */
	  set_window_width (child, pos + left - lastright, 1 | nodelete);

	  /* Now advance child to next window,
	     and set lastright if child was not just deleted.  */
	  lastright = pos + left, lastoright = opos;
	}
      /* Delete children that became too small */
      if (nodelete == 0)
	for (child = w->hchild; !NULL (child); child = XWINDOW (child)->next)
	  {
	    set_window_width (child, XINT (XWINDOW (child)->pixwidth), 0);
	  }
    }
}

static int window_select_count;

DEFUN ("set-window-buffer", Fset_window_buffer, Sset_window_buffer, 2, 2, 0,
  "Make WINDOW display BUFFER as its contents.\n\
BUFFER can be a buffer or buffer name.")
  (window, buffer)
     register Lisp_Object window, buffer;
{
  register Lisp_Object tem;
  register struct window *w = decode_window (window);

  buffer = Fget_buffer (buffer);
  CHECK_BUFFER (buffer, 1);

  if (NULL (XBUFFER (buffer)->name))
    error ("Attempt to display deleted buffer");

  tem = w->buffer;
  if (!NULL (tem))
    {
      unshow_buffer (w);

      if (XBUFFER(tem)->style != XBUFFER(buffer)->style)
	w->size_change = Qt;
    }

  w->buffer = buffer;
  Fset_marker (w->pointm,
	       make_number (BUF_PT (XBUFFER (buffer))),
	       buffer);
  w->start_button = Qnil;
  set_marker_restricted (w->start,
			 make_number (XBUFFER (buffer)->last_window_start),
			 buffer);
  w->start_at_line_beg = Qnil;
  XFASTINT (w->last_modified) = 0;
  windows_or_buffers_changed++;
  if (EQ (window, selected_window))
    Fset_buffer (buffer);

  {
    register Lisp_Object screen = w->root;
    if (NULL(Fmemq(screen, (XBUFFER(buffer)->allowed_screens))))
      (XBUFFER(buffer)->allowed_screens) =
	Fcons(screen,(XBUFFER(buffer)->allowed_screens));

    /* Epoch:
     * Call value of set-window-buffer-hook if it is bound and non-nil
     */
    tem = intern("set-window-buffer-hook");
    tem = XSYMBOL (tem)->value;	/* convert to value */
    if (! EQ (tem, Qunbound) && ! EQ (tem, Qnil))
      tem = call1 (tem,window);
  }
  return Qnil;
}

/* Record info on buffer window w is displaying
   when it is about to cease to display that buffer.  */
static void
unshow_buffer (w)
     register struct window *w;
{
  register Lisp_Object buf;
  buf = w->buffer;

  if (XBUFFER (buf) != XMARKER (w->pointm)->buffer)
    abort ();

  if (w == XWINDOW (selected_window)
      || ! EQ (buf, XWINDOW (selected_window)->buffer))
    /* Do this except when the selected window's buffer
       is being removed from some other window.  */
    XBUFFER (buf)->last_window_start = marker_position (w->start);

  /* Point in the selected window's buffer
     is actually stored in that buffer, and the window's pointm isn't used.
     So don't clobber point in that buffer.  */
  if (! EQ (buf, XWINDOW (selected_window)->buffer))
    BUF_PT (XBUFFER (buf)) =
      clip_to_bounds (BUF_BEGV(XBUFFER (buf)),
		      marker_position(w->pointm),
		      BUF_ZV(XBUFFER(buf)));
}

DEFUN ("select-window", Fselect_window, Sselect_window, 1, 1, 0,
  "Select WINDOW.  Most editing will apply to WINDOW's buffer.\n\
The main editor command loop selects the buffer of the selected window\n\
before each command.")
  (window)
     register Lisp_Object window;
{
  register struct window *w;
  register struct window *ow = XWINDOW (selected_window);

  CHECK_WINDOW (window, 0);

  w = XWINDOW (window);

  if (NULL (w->buffer))
    error ("Trying to select window with no buffer");

  XFASTINT (w->use_time) = ++window_select_count;
  if (EQ (window, selected_window))
    return window;

  Fset_marker (ow->pointm, make_number (BUF_PT (XBUFFER (ow->buffer))),
	       ow->buffer);

  selected_window = window;

  if (!inhibit_window_system)
    x_select_screen(w->root);	/* Epoch */

  cur_root->select = window;	/* Epoch */

  record_buffer (w->buffer);
  Fset_buffer (w->buffer);

  /* Go to the point recorded in the window.
     This is important when the buffer is in more
     than one window.  It also matters when
     redisplay_window has altered point after scrolling,
     because it makes the change only in the window.  */
  SET_PT (marker_position (w->pointm));
  if (point < BEGV)
    point = BEGV;
  if (point > ZV)
    point = ZV;

  windows_or_buffers_changed++;
  screen_changed++;

  return window;
}

DEFUN ("display-buffer", Fdisplay_buffer, Sdisplay_buffer, 1, 2, 0,
  "Make BUFFER appear in some window but don't select it.\n\
BUFFER can be a buffer or a buffer name.\n\
If BUFFER is shown already in some window, just uses that one,\n\
unless the window is the selected window and NOTTHISWINDOW is non-nil.\n\
Returns the window displaying BUFFER.")
  (buffer, notthiswindow)
     register Lisp_Object buffer, notthiswindow;
{
  register Lisp_Object window;
  struct Root_Block *rb;

  buffer = Fget_buffer (buffer);
  CHECK_BUFFER (buffer, 0);

  if (NULL (notthiswindow)
      && XBUFFER (XWINDOW (selected_window)->buffer) == XBUFFER (buffer))
    return selected_window;

  window = Fget_buffer_window (buffer);
  if (!NULL (window)
      && (NULL (notthiswindow) || !EQ (window, selected_window)))
    return window;

  rb = XROOT(XWINDOW(window)->root);
  if (pop_up_windows)
    {
      /* Don't try to create a window if would get an error */
      if (window_min_height < 2)
	window_min_height = 2;
      if (split_height_threshold < window_min_height << 1)
	split_height_threshold = window_min_height << 1;

      window = Fget_largest_window ();
      
      if (window_char_height (XWINDOW(window)) >= split_height_threshold
	  &&
	  XFASTINT (XWINDOW (window)->pixwidth) == WS->pixw)
	window = Fsplit_window (window, Qnil, Qnil);
      else
	{
	  window = Fget_lru_window ();
	  rb = XROOT(XWINDOW(window)->root);
	  if ((EQ (window, selected_window)
	       || (EQ (selected_window, rb->minibuf_window)
		   && EQ (window, XWINDOW(minibuf_window)->prev)))
	      && window_char_height(XWINDOW(window)) >= window_min_height << 1)
	    window = Fsplit_window (window, Qnil, Qnil);
	}
    }
  else
    window = Fget_lru_window ();

  Fset_window_buffer (window, buffer);
  return window;
}

temp_output_buffer_show (buf)
     register Lisp_Object buf;
{
  register struct buffer *old = current_buffer;
  register Lisp_Object window;
  register struct window *w;

  Fset_buffer (buf);
  XBUFFER (buf)->save_modified = MODIFF;
  BEGV = BEG;
  ZV = Z;
  SET_PT (BEG);
  clip_changed = 1;
  set_buffer_internal (old);

  if (!EQ (Vtemp_buffer_show_hook, Qnil))
    call1 (Vtemp_buffer_show_hook, buf);
  else
    {
      window = Fdisplay_buffer (buf, Qnil);
      Vminibuf_scroll_window = window;
      w = XWINDOW (window);
      XFASTINT (w->hscroll) = 0;
      set_marker_restricted (w->start, make_number (1), buf);
      set_marker_restricted (w->pointm, make_number (1), buf);
    }
}

static
make_dummy_parent (window)
     Lisp_Object window;
{
  register Lisp_Object old, new;
  register struct window *o, *p;

  old = window;
  XSETTYPE (old, Lisp_Vector);
  new = Fcopy_sequence (old);
  XSETTYPE (new, Lisp_Window);

  o = XWINDOW (old);
  p = XWINDOW (new);
  XFASTINT (p->sequence_number) = ++sequence_number;

  /* Put new into window structure in place of window */
  replace_window (window, new);

  o->next = Qnil;
  o->prev = Qnil;
  o->vchild = Qnil;
  o->hchild = Qnil;
  o->parent = new;

  p->start = Qnil;
  p->pointm = Qnil;
  p->buffer = Qnil;
  p->root = o->root;		/* Need this? */
}

DEFUN ("split-window", Fsplit_window, Ssplit_window, 0, 3, "",
  "Split WINDOW, putting SIZE lines in the first of the pair.\n\
WINDOW defaults to selected one and SIZE to half its size.\n\
If optional third arg HOR-FLAG is non-nil, split side by side\n\
and put SIZE columns in the first of the pair.")
  (window, chsize, horflag)
     Lisp_Object window, chsize, horflag;
{
  register Lisp_Object new;
  register struct window *o, *p;
  register struct Root_Block *rb;
  register int size;
  register int psize;
  struct X_font *font;
  struct buffer *b;

  if (NULL (window))
    window = selected_window;
  else
    CHECK_WINDOW (window, 0);

  o = XWINDOW (window);
  rb = XROOT(o->root);
  b = XBUFFER(o->buffer);
  font = XXFONT(STYLE_FIELD(o,font,stylenorm));  
  if (NULL (chsize))
    {
      if (!NULL (horflag))
	{
	  size = ((XFASTINT(o->pixwidth) / XFWIDTH(font)) >> 1);
	  psize = size * XFWIDTH(font);
	}
      else
	{
	  size = ((XFASTINT(o->pixheight) / XFHEIGHT(font)) >> 1);
	  psize = size * XFHEIGHT(font);
	}
    }
  else
    {
      /*
       * Argument is in lines/columns, so mult by pixel size of base
       * font of the window
       */
      CHECK_NUMBER (chsize, 1);
      size = XINT (chsize);
      psize = size * (NULL(horflag) ? XFHEIGHT(font) : XFWIDTH(font));
    }

  if (EQ (window, rb->minibuf_window))
    error ("Attempt to split minibuffer window");

  if (NULL (horflag))
    {
      if (window_min_height < 2)
	window_min_height = 2;

      if (size < window_min_height ||
	  size + window_min_height > window_char_height(o))
	args_out_of_range_3 (window, chsize, horflag);
      if (NULL (o->parent) ||
	  NULL (XWINDOW (o->parent)->vchild))
	{
	  make_dummy_parent (window);
	  new = o->parent;
	  XWINDOW (new)->vchild = window;
	}
    }
  else
    {
      if (size < window_min_width ||
	  size + window_min_width > window_char_width(o))
	args_out_of_range_3 (window, chsize, horflag);
      if (NULL (o->parent) ||
	  NULL (XWINDOW (o->parent)->hchild))
	{
	  make_dummy_parent (window);
	  new = o->parent;
	  XWINDOW (new)->hchild = window;
	}
    }

  /* Now we know that window's parent is a vertical combination
     if we are dividing vertically, or a horizontal combination
     if we are making side-by-side windows */

  windows_or_buffers_changed++;

  new = make_window ();
  p = XWINDOW (new);
  XFASTINT(o->last_modified) = 0;
  p->size_change = Qt;
/*  o->size_change = Qt; */
  p->next = o->next;
  if (!NULL (p->next))
    XWINDOW (p->next)->prev = new;
  p->prev = window;
  o->next = new;
  p->parent = o->parent;
  p->root = o->root;		/* Epoch */
  XFASTINT(p->height) = XFASTINT(o->height) = 0;
  p->window_end_valid = o->window_end_valid = Qnil;

  Fset_window_buffer (new, o->buffer);

  /* Apportion the available screen space among the two new windows */

  if (!NULL (horflag))
    {
      /* Splitting windows side by side */
      p->pixheight = o->pixheight;
      p->pixtop = o->pixtop;
      XFASTINT (p->pixwidth) = XFASTINT (o->pixwidth) - psize;
      XFASTINT (o->pixwidth) = psize;
      XFASTINT (p->pixleft) = XFASTINT (o->pixleft) + psize;
    }
  else
    {
      /* Splitting windows vertically */
      p->pixleft = o->pixleft;	  
      p->pixwidth = o->pixwidth;  
      XFASTINT (p->pixheight) = XFASTINT (o->pixheight) - psize;
      XFASTINT (o->pixheight) = psize;
      XFASTINT (p->pixtop) = XFASTINT (o->pixtop) + psize;
    }

  return new;
}

DEFUN ("enlarge-window", Fenlarge_window, Senlarge_window, 1, 2, "p",
  "Make current window ARG lines bigger.\n\
From program, optional second arg non-nil means grow sideways ARG columns.")
  (n, side)
     register Lisp_Object n, side;
{
  struct X_font *font;
  struct buffer *b = current_buffer;
  register int size;

  CHECK_NUMBER (n, 0);
  /* Silent failure if no net change in size */
  if (!n) return Qnil;
#if 0  
  font = XXFONT(STYLE_FIELD(XWINDOW(selected_window),font,stylenorm));
  size = XINT(n) * (NULL(side) ? XFHEIGHT(font) : XFWIDTH(font));
  
  change_window_height (size, !NULL (side));
#endif
  change_window_height (XINT(n),!NULL(side));
  return Qnil;
}

DEFUN ("shrink-window", Fshrink_window, Sshrink_window, 1, 2, "p",
  "Make current window ARG lines smaller.\n\
From program, optional second arg non-nil means shrink sideways ARG columns.")
  (n, side)
     register Lisp_Object n, side;
{
  struct X_font *font;
  struct buffer *b = current_buffer;
  register int size;

  CHECK_NUMBER (n, 0);
  /* Silent failure if no change in size */
  if (!n) return Qnil;
#if 0  
  font = XXFONT(STYLE_FIELD(XWINDOW(selected_window),font,stylenorm));
  size = XINT(n) * (NULL(side) ? XFHEIGHT(font) : XFWIDTH(font));

  change_window_height (-size, !NULL (side));
#endif
  change_window_height (-XINT(n), !NULL(side));
  return Qnil;
}

int
window_height (window)
     Lisp_Object window;
{
  register struct window *p = XWINDOW (window);
  return XFASTINT (p->pixheight);
}

/*
 * Return window height as chars relative to size of window's base
 * font.
 */
int
window_char_height(w)
     register struct window *w;
{
  struct X_font *font;
  struct buffer *b = XBUFFER(w->buffer);
  register int pix, height = 0,end = 0;
  int tmp;

  font = XXFONT(STYLE_FIELD(w,font,stylenorm));
#if 0  
  mode = XLINE(w->modeline) ? 
    (XLINE(w->modeline)->ascent + XLINE(w->modeline)->descent) :
      0;
#endif
  if (!NULL(w->window_end_valid))
    {
      height = !NULL(w->height) ? XFASTINT(w->height) : 0;
      end = !NULL(w->window_end_ppos) ? XFASTINT(w->window_end_ppos) -
	w->pixtop : 0;
    }
  pix = w->pixheight - end;

  tmp = height + pix / (XFHEIGHT(font));
  return tmp;
}

int
window_width (window)
     Lisp_Object window;
{
  register struct window *p = XWINDOW (window);
  return XFASTINT (p->pixwidth);
}

/*
 * Return window width as chars relative to size of window's base font.
 * If window doesn't start at left edge of screen, subtract 1 char for
 * the left window border.
 */
int
window_char_width (w)
     register struct window *w;
{
  struct X_font *font;
  struct buffer *b = XBUFFER(w->buffer);

  font = XXFONT(STYLE_FIELD(w,font,stylenorm));
  return (w->pixleft == XWSCREEN(XROOT(w->root)->win)->in_border) ?
    XFASTINT(w->pixwidth) / XFWIDTH(font)
      : XFASTINT(w->pixwidth) / XFWIDTH(font) -1;
}

#define MINSIZE(window)\
  (widthflag ? window_min_width * XFWIDTH(font) :\
   (EQ(window,rb->minibuf_window) ? 1 : window_min_height) * XFHEIGHT(font))\

#define CURBEG(w) \
  *(widthflag ? (int *) &w->pixleft : (int *) &w->pixtop)

#define CURSIZE(w) \
  *(widthflag ? (int *) &w->pixwidth : (int *) &w->pixheight)

#define CURCHARSIZE(w) \
  (widthflag ? window_char_width(w) : window_char_height(w))

#define MINCHARSIZE(window) \
  (widthflag ? window_min_width : EQ(window,rb->minibuf_window) ? 1 : window_min_height)

/* Unlike set_window_height, this function
 also changes the heights of the siblings so as to
 keep everything consistent. */

change_window_height (delta, widthflag)
     register int delta;
     int widthflag;
{
  register Lisp_Object parent;
  Lisp_Object window;
  register struct Root_Block *rb;
  register struct window *p;
  int *sizep;
  int (*sizefun) () = widthflag ? window_width : window_height;
  register int (*setsizefun) () = widthflag ? set_window_width : set_window_height;
  struct window *w = XWINDOW(selected_window);
  struct buffer *b = XBUFFER(w->buffer);
  struct X_font *font;
  int dim;
    
  /* Silently fail to modify sole window on a screen.
   */
  rb = XROOT(XWINDOW(selected_window)->root);  
  if ((EQ(selected_window,rb->minibuf_window)) ||
      (EQ(selected_window,Fnext_window(selected_window,make_number(1)))))
    return;
  font = XXFONT(STYLE_FIELD(w,font,stylenorm));
  if (window_min_height < 2)
    window_min_height = 2;

  window = selected_window;
  while (1)
    {
      p = XWINDOW (window);
      parent = p->parent;
      if (NULL (parent))
	{
	  if (widthflag)
	    error ("No other window to side of this one");
	  break;
	}
      if (widthflag ? !NULL (XWINDOW (parent)->hchild)
	  : !NULL (XWINDOW (parent)->vchild))
	break;
      window = parent;
    }

  if (p->next == minibuf_window && NULL(p->prev) && distinct_minibuffer
      && !widthflag)
    {
      /*
       * Epoch Note:  If this (bounding) window is the sole window
       * on the screen and we are running with distinct minibuffers,
       * silently fail here, as we don't want to allow this size change
       * and reposition the minibuffer incorrectly.
       */
/*      fprintf(stderr,"Trap\n"); */
      return;
    }

  sizep = &CURSIZE (p);

  dim = CURCHARSIZE(p);
  if (dim + delta < MINCHARSIZE(window))
    {
      Fdelete_window(window);
      return;
    }
  delta *= widthflag ? XFWIDTH(font) : XFHEIGHT(font);

  {
    register int maxdelta;
    register Lisp_Object tem;

    maxdelta = (!NULL (parent) ? (*sizefun) (parent) - *sizep
		: (tem = (!NULL (p->next) ? p->next : p->prev),
		   (*sizefun) (tem) - MINSIZE (tem)));

    if (delta > maxdelta)
      /* This case traps trying to make the minibuffer
	 the full screen, or make the only window aside from the
	 minibuffer the full screen.  */
      delta = maxdelta;
  }

  if (!NULL (p->next) &&
      (*sizefun) (p->next) - delta >= MINSIZE (p->next))
    {
      (*setsizefun) (p->next, (*sizefun) (p->next) - delta, 0, 0);
      (*setsizefun) (window, *sizep + delta, 0, 0);
      CURBEG (XWINDOW (p->next)) += delta;
      /* This does not change size of p->next,
	 but it propagates the new top edge to its children */
      (*setsizefun) (p->next, (*sizefun) (p->next), 0, 0);
    }
  else if (!NULL (p->prev) &&
	   (*sizefun) (p->prev) - delta >= MINSIZE (p->prev))
    {
      (*setsizefun) (p->prev, (*sizefun) (p->prev) - delta, 0, 0);
      CURBEG (p) -= delta;
      (*setsizefun) (window, *sizep + delta, 0, 0);
    }
  else
    {
      register int delta1;
      register int opht = (*sizefun) (parent);

      /* If trying to grow this window to or beyond size of the parent,
	 make delta1 so big that, on shrinking back down,
	 all the siblings end up with less than one line and are deleted.  */
      if (opht <= *sizep + delta)
	delta1 = opht * opht * 2;
      /* Otherwise, make delta1 just right so that if we add delta1
	 lines to this window and to the parent, and then shrink
	 the parent back to its original size, the new proportional
	 size of this window will increase by delta.  */
      else
	delta1 = (delta * opht * 100) / ((opht - *sizep - delta) * 100);

      /* Add delta1 lines or columns to this window, and to the parent,
	 keeping things consistent while not affecting siblings.  */
      CURSIZE (XWINDOW (parent)) = opht + delta1;
      (*setsizefun) (window, *sizep + delta1, 0, 0);

      /* Squeeze out delta1 lines or columns from our parent,
	 shriking this window and siblings proportionately.
	 This brings parent back to correct size.
	 Delta1 was calculated so this makes this window the desired size,
	 taking it all out of the siblings.  */
      (*setsizefun) (parent, opht, 0, 0);
    }

  XFASTINT (p->last_modified) = 0;
}
#undef MINSIZE
#undef CURBEG
#undef CURSIZE


/* Scroll window WINDOW (a Lisp object) by N lines.
   If NOERROR is 0, signal an error if that can't be done.
   If NOERROR is nonzero, return Qnil if successful
   and an error name otherwise.  */

static Lisp_Object
window_scroll (window, n, noerror)
     Lisp_Object window;
     int n;
     int noerror;
{
  register struct window *w = XWINDOW (window);
  struct Root_Block *rb = XROOT(w->root);
  register int opoint = point;
  register int ht, pos;
  register Lisp_Object tem;
  int lose;
  Lisp_Object bolp;

  ht = window_char_height(w) - !EQ(window,rb->minibuf_window);
  
  XFASTINT (tem) = point;
  tem = Fpos_visible_in_window_p (tem, window);

  if (NULL (tem))
    {
      Fvertical_motion (make_number (- ht / 2));
      XFASTINT (tem) = point;
      w->start_button = get_start_button(point,w);
#if 0
      if (!NULL(w->start_button))
	fprintf(stderr,"Starting button from %d to %d for %d\n",
		marker_position(XBUTTON(w->start_button)->start),
		marker_position(XBUTTON(w->start_button)->end),
		pos);
#endif       
      Fset_marker (w->start, tem, w->buffer);
      w->force_start = Qt;
    }

  SET_PT (marker_position (w->start));
  lose = n < 0 && point == BEGV;
  Fvertical_motion (make_number (n));
  pos = point;
  bolp = Fbolp ();
  SET_PT (opoint);

  if (lose)
    {
      if (noerror)
	return Qbeginning_of_buffer;
      Fsignal (Qbeginning_of_buffer, Qnil);
    }

  if (pos < ZV)
    {
      /*      w->start_button = Qnil; */
      w->start_button = get_start_button(pos,w);       
#if 0
      if (!NULL(w->start_button))
	fprintf(stderr,"Starting button from %d to %d for %d\n",
		marker_position(XBUTTON(w->start_button)->start),
		marker_position(XBUTTON(w->start_button)->end),
		pos);
#endif 
      set_marker_restricted (w->start, make_number (pos), w->buffer);
      w->start_at_line_beg = bolp;
      w->update_mode_line = Qt;
      XFASTINT (w->last_modified) = 0;
      if (pos > opoint)
	SET_PT (pos);
      if (n < 0)
	{
	  SET_PT (pos);
	  tem = Fvertical_motion (make_number (ht));
	  if (point > opoint || XFASTINT (tem) < ht)
	    SET_PT (opoint);
	  else
	    Fvertical_motion (make_number (-1));
	}
      return Qnil;
    }
  else
    {
      if (!noerror)
	Fsignal (Qend_of_buffer, Qnil);
      return Qend_of_buffer;
    }
}

scroll_command (n, direction)
     register Lisp_Object n;
     int direction;
{
  register int defalt
    = direction * (window_char_height (XWINDOW(selected_window)) - 1
		   - next_screen_context_lines);

  if (NULL (n))
    window_scroll (selected_window, defalt, 0);
  else if (EQ (n, Qminus))
    window_scroll (selected_window, - defalt, 0);
  else
    {
      n = Fprefix_numeric_value (n);
      window_scroll (selected_window, XINT (n) * direction, 0);
    }
}

DEFUN ("scroll-up", Fscroll_up, Sscroll_up, 0, 1, "P",
  "Scroll text of current window upward ARG lines; or near full screen if no ARG.\n\
When calling from a program, supply a number as argument or nil.")
  (n)
     Lisp_Object n;
{
  scroll_command (n, 1);
  return Qnil;
}

DEFUN ("scroll-down", Fscroll_down, Sscroll_down, 0, 1, "P",
  "Scroll text of current window downward ARG lines; or near full screen if no ARG.\n\
When calling from a program, supply a number as argument or nil.")
  (n)
     Lisp_Object n;
{
  scroll_command (n, -1);
  return Qnil;
}

DEFUN ("scroll-left", Fscroll_left, Sscroll_left, 1, 1, "P",
  "Scroll selected window display ARG columns left.\n\
Default for ARG is window width minus 2.")
  (arg)
     register Lisp_Object arg;
{
  if (NULL (arg))
    XFASTINT (arg) = window_char_width(XWINDOW(selected_window)) - 2;
  else
    arg = Fprefix_numeric_value (arg);

  return Fset_window_hscroll (selected_window,
			      make_number (XINT (XWINDOW (selected_window)->hscroll)
					   + XINT (arg)));

}

DEFUN ("scroll-right", Fscroll_right, Sscroll_right, 1, 1, "P",
  "Scroll selected window display ARG columns right.\n\
Default for ARG is window width minus 2.")
  (arg)
     register Lisp_Object arg;
{
  if (NULL (arg))
    XFASTINT (arg) = window_char_width(XWINDOW(selected_window)) - 2;
  else
    arg = Fprefix_numeric_value (arg);

  return Fset_window_hscroll (selected_window,
			      make_number (XINT (XWINDOW (selected_window)->hscroll)
					   - XINT (arg)));

}

DEFUN ("scroll-other-window", Fscroll_other_window, Sscroll_other_window, 0, 1, "P",
  "Scroll text of next window upward ARG lines; or near full screen if no ARG.\n\
The next window is the one below the current one; or the one at the top\n\
if the current one is at the bottom.\n\
When calling from a program, supply a number as argument or nil.")
  (n)
     register Lisp_Object n;
{
  register Lisp_Object window;
  struct Root_Block *rb;
  struct buffer *old = current_buffer;
  register int ht;
  register int opoint = point;
  register struct window *w;
  Lisp_Object result;

  rb = XROOT(XWINDOW(selected_window)->root);
  if (EQ (selected_window, rb->minibuf_window)
      && !NULL (Vminibuf_scroll_window))
    window = Vminibuf_scroll_window;
  else
    window = Fnext_window (selected_window, Qnil);
  CHECK_WINDOW (window, 0);
  ht = window_char_height (XWINDOW(window)) - 1;

  if (EQ (window, selected_window))
    error ("There is no other window");

  w = XWINDOW (window);
  Fset_buffer (w->buffer);
  SET_PT (marker_position (w->pointm));

  if (NULL (n))
    result = window_scroll (window, ht - next_screen_context_lines, 1);
  else if (EQ (n, Qminus))
    result = window_scroll (window, next_screen_context_lines - ht, 1);
  else
    {
      if (XTYPE (n) == Lisp_Cons)
	n = Fcar (n);
      CHECK_NUMBER (n, 0);
      result = window_scroll (window, XINT (n), 1);
    }

  Fset_marker (w->pointm, make_number (point), Qnil);
  set_buffer_internal (old);
  SET_PT (opoint);
  if (!EQ (result, Qnil))
    Fsignal(result, Qnil);
  return Qnil;
}

DEFUN ("recenter", Frecenter, Srecenter, 0, 1, "P",
  "Center point in window and redisplay screen.  With ARG, put point on line ARG.\n\
The desired position of point is always relative to the current window.\n\
Just C-u as prefix means put point in the center of the screen.\n\
No arg (i.e., it is nil) erases the entire screen and then\n\
redraws with point in the center.")
  (n)
     register Lisp_Object n;
{
  register struct window *w = XWINDOW (selected_window);
  struct Root_Block *rb = XROOT(w->root);
  register int ht = window_char_height (w)
    - !EQ (selected_window, rb->minibuf_window);

  register int opoint = point;

  if (NULL (n))
    {
      extern int screen_garbaged;
      screen_garbaged++;
      XFASTINT (n) = ht / 2;
    }
  else if (XTYPE (n) == Lisp_Cons) /* Just C-u. */
    {
      XFASTINT (n) = ht / 2;
    }
  else
    {
      n = Fprefix_numeric_value (n);
      CHECK_NUMBER (n, 0);
    }

  if (XINT (n) < 0)
    XSETINT (n, XINT (n) + ht);

  XSETINT (n, - XINT (n));

  Fvertical_motion (n);
  Fset_marker (w->start, make_number (point), w->buffer);
  w->start_at_line_beg = Fbolp ();

  SET_PT (opoint);
  w->force_start = Qt;

  return Qnil;
}

DEFUN ("move-to-window-line", Fmove_to_window_line, Smove_to_window_line,
  1, 1, "P",
  "Position point relative to window.\n\
With no argument, position at text at center of window.\n\
An argument specifies screen line; zero means top of window,\n\
negative means relative to bottom of window.")
  (arg)
     register Lisp_Object arg;
{
  register struct window *w = XWINDOW (selected_window);
  struct Root_Block *rb = XROOT(w->root);
  register int height = window_char_height(w) -
    !EQ (selected_window, rb->minibuf_window);
  register int start;

  if (NULL (arg))
    XFASTINT (arg) = height / 2;
  else
    {
      arg = Fprefix_numeric_value (arg);
      if (XINT (arg) < 0)
	XSETINT (arg, XINT (arg) + height);
    }

  start = marker_position (w->start);
  if (start < BEGV || start > ZV)
    {
      Fvertical_motion (make_number (- height / 2));
      Fset_marker (w->start, make_number (point), w->buffer);
      w->start_at_line_beg = Fbolp ();
      w->force_start = Qt;
    }
  else
    SET_PT (start);

  return Fvertical_motion (arg);
}

struct save_window_data
  {
    int size_from_Lisp_Vector_struct;
    struct Lisp_Vector *next_from_Lisp_Vector_struct;
    Lisp_Object current_window;
    Lisp_Object current_buffer;
    Lisp_Object screen_base_window; /* Epoch */
    Lisp_Object minibuf_scroll_window;
    /* A vector, interpreted as a struct saved_window */
    Lisp_Object saved_windows;
  };
#define SAVE_WINDOW_DATA_SIZE 5 /* Arg to Fmake_vector */

/* This is saved as a Lisp_Vector */
struct saved_window
  {
    /* these first two must agree with struct Lisp_Vector in lisp.h */
    int size_from_Lisp_Vector_struct;
    struct Lisp_Vector *next_from_Lisp_Vector_struct;

    Lisp_Object window;
    Lisp_Object buffer, start, pointm, mark;
    Lisp_Object pixleft, pixtop, pixwidth, pixheight, hscroll;
    Lisp_Object parent, prev;
    Lisp_Object start_at_line_beg;
  };
#define SAVED_WINDOW_VECTOR_SIZE 13 /* Arg to Fmake_vector */

#define SAVED_WINDOW_N(swv,n) \
  ((struct saved_window *) (XVECTOR ((swv)->contents[(n)])))

DEFUN ("set-window-configuration",
       Fset_window_configuration, Sset_window_configuration,
       1, 1, 0,
       "Restore the configuration of Emacs' windows and buffers to\n\
the state specified by CONFIGURATION.  CONFIGURATION must be a value\n\
retrned by  current-window-configuration  -- see the documentation of that\n\
function for more information.")
     (arg)
     Lisp_Object arg;
{
  register struct window *w;
  register struct save_window_data *data;
  struct Lisp_Vector *saved_windows;
  register struct saved_window *p;
  register Lisp_Object tem;
  Lisp_Object new_current_buffer;
  int k;
  int wheight, wwidth;
  struct W_Screen *ws;
  struct Root_Block *rb;

  while (XTYPE (arg) != Lisp_Window_Configuration)
    {
      /* the function window-configuration-p isn't actually defined
	 at present --- is there a need for it? */
      arg = wrong_type_argument (intern ("window-configuration-p"), arg);
    }

  data = (struct save_window_data *) XVECTOR (arg);
  saved_windows = XVECTOR (data->saved_windows);
  /* Epoch:
   * Set rb to current edit screen.
   */
  rb = XROOT(XWINDOW(XWINDOW(minibuf_window)->prev)->root);

  windows_or_buffers_changed++;
  new_current_buffer = data->current_buffer;
  if (NULL (XBUFFER (new_current_buffer)->name))
    new_current_buffer = Qnil;

  for (k = 0; k < saved_windows->size; k++)
    {
      p = SAVED_WINDOW_N (saved_windows, k);
      w = XWINDOW (p->window);
      w->next = Qnil;

      if (!NULL (p->parent))
	w->parent = SAVED_WINDOW_N (saved_windows, XFASTINT (p->parent))->window;
      else
	w->parent = Qnil;

      /* Epoch:
       * Don't mess with pointers on minibuffer window.
       */
      if (!EQ(p->window,rb->minibuf_window))
	{
	  if (!NULL (p->prev))
	    {
	      w->prev =
		SAVED_WINDOW_N (saved_windows, XFASTINT (p->prev))->window;
	      XWINDOW (w->prev)->next = p->window;
	    }
	  else
	    {
	      w->prev = Qnil;
	      if (!NULL (w->parent))
		{
		  if (EQ (p->pixwidth, XWINDOW (w->parent)->pixwidth))
		    {
		      XWINDOW (w->parent)->vchild = p->window;
		      XWINDOW (w->parent)->hchild = Qnil;
		    }
		  else
		    {
		      XWINDOW (w->parent)->hchild = p->window;
		      XWINDOW (w->parent)->vchild = Qnil;
		    }
		}
	    }
	  /* Note:  If window has changed its position (left or top), mark it
	    * has hosed and redraw/clear the whole thing
	    */
	  if (w->pixtop != p->pixtop || w->pixleft != p->pixleft ||
	      w->pixheight != p->pixheight || w->pixwidth != p->pixwidth)
	    w->size_change = Qt;

	  w->window_end_valid = Qnil; /* Need to clear out end */
	  w->hscroll = p->hscroll;
	  w->pixleft = p->pixleft;
	  w->pixtop = p->pixtop;
	  w->pixwidth = p->pixwidth;
	  w->pixheight = p->pixheight;
/*	  XFASTINT (w->last_modified) = 0; */
	}

      /* Reinstall the saved buffer and pointers into it.  */
      if (NULL (p->buffer))
	w->buffer = p->buffer;
      else
	{
	  if (!NULL (XBUFFER (p->buffer)->name))
	    /* If saved buffer is alive, install it.  */
	    {
	      w->buffer = p->buffer;
	      w->start_at_line_beg = p->start_at_line_beg;
	      w->start_button = Qnil;
	      set_marker_restricted (w->start,
				     Fmarker_position (p->start), w->buffer);
	      set_marker_restricted (w->pointm,
				     Fmarker_position (p->pointm), w->buffer);
	      Fset_marker (XBUFFER (w->buffer)->mark,
			   Fmarker_position (p->mark), w->buffer);

	      if (!EQ (p->buffer, new_current_buffer) &&
		  XBUFFER (p->buffer) == current_buffer &&
		  rb == XROOT(w->root)) /* Epoch -don't switch screens */
		Fgoto_char (w->pointm);
	    }
	  else if (NULL (XBUFFER (w->buffer)->name))
	    /* Else if window's old buffer is dead too, get a live one.  */
	    {
	      w->buffer = Fcdr (Fcar (Vbuffer_alist));
	      /* Set window markers at start of buffer.
		 Rely on set_marker_restricted to put them
		 within the restriction.  */
	      w->start_button = Qnil;
	      set_marker_restricted (w->start, make_number (0), w->buffer);
	      set_marker_restricted (w->pointm, make_number (0), w->buffer);
	      w->start_at_line_beg = Qt;
	    }
	  else
	    /* Keeping window's old buffer; make sure the markers are real.  */
	    /* Else if window's old buffer is dead too, get a live one.  */
	    {
	      /* Set window markers at start of buffer.
		 Rely on set_marker_restricted to put them within the restriction.  */
	      if (XMARKER (w->start)->buffer == 0)
		set_marker_restricted (w->start, make_number (0), w->buffer);
	      if (XMARKER (w->pointm)->buffer == 0)
		set_marker_restricted (w->pointm,
				       make_number (BUF_PT (XBUFFER (w->buffer))),
				       w->buffer);
	      w->start_button = Qnil;
	      w->start_at_line_beg = Qt;
	    }
	}
    }

  /* Epoch:
   * Restore base window for screen.
   */
  XROOT(XWINDOW(data->screen_base_window)->root)->ewin =
    data->screen_base_window;
  /* Epoch:
   * Fix next pointer of base window.
   */
  XWINDOW(data->screen_base_window)->next =
    XROOT(XWINDOW(data->screen_base_window)->root)->minibuf_window;

  w = XWINDOW(data->screen_base_window);  
#if 0
  /* Epoch:
   * Set current screen size to size of top window, then change window
   * tree to current size
   */
  if (distinct_minibuffer)
    {
      w = XWINDOW(data->screen_base_window);
      ws = XWSCREEN(XROOT(w->root)->win);
      wheight = ws->height;	wwidth = ws->width; /* save current values */
      ws->height = w->pixheight; ws->width = w->pixwidth;
      change_screen_size(wheight,wwidth,0,XROOT(w->root));
    }
#endif
  /* Ugly: We're not saving the current screen, so it could be that we moved
   * to a different screen, or it might be that the current base screen
   * has been toasted by the restore. What we'll do is look at the screen
   * of the current base window (through minibuf->prev), and the screen of
   * the restoration base screen. If they're the same, then we're still on
   * the same screen, so select the window and roll. Otherwise, we've switched
   * and all we want to do is remember what the selected window should be on
   * this screen.
   */
  if (rb == XROOT(w->root))	/* rb is already current edit screen */
    {
      /* For same screen, we might have toasted previous root window */
      XWINDOW(minibuf_window)->prev = rb->ewin;
      rb->select = data->current_window;
      /* Go to the save selected window */
      Fselect_window (data->current_window);
      if (!NULL (new_current_buffer))
	  Fset_buffer (new_current_buffer);
      else
	  Fset_buffer (XWINDOW (selected_window)->buffer);

      windows_or_buffers_changed++;
      rb->prev_select = Qnil;
    }
  else
    {
      extern Lisp_Object Fset_screen_modified();
      
      if (selected_window == minibuf_window && !distinct_minibuffer)
	{
	  /* Epoch: we've switched screens in the midst of minibuffer
	   * operations and have a bogus selected_window
	   */
	  Fselect_window(rb->prev_select);
	}      
      rb->prev_select = Qnil;
      XROOT(w->root)->select = data->current_window;
      Fset_screen_modified(w->root);

      Fselect_window(rb->select);
      Fset_buffer(XWINDOW(selected_window)->buffer);
    }

  Vminibuf_scroll_window = data->minibuf_scroll_window;
  return (Qnil);
}


static int
count_windows (window)
     register struct window *window;
{
  register int count = 1;
  if (!NULL (window->next))
    count += count_windows (XWINDOW (window->next));
  if (!NULL (window->vchild))
    count += count_windows (XWINDOW (window->vchild));
  if (!NULL (window->hchild))
    count += count_windows (XWINDOW (window->hchild));
  return count;
}

DEFUN ("current-window-configuration",
	Fcurrent_window_configuration, Scurrent_window_configuration, 0, 0, 0,
       "Return an object representing Emacs' current window configuration,\n\
namely the number of windows, their sizes and current buffers, and for\n\
each displayed buffer, where display starts, and the positions of\n\
point and mark.  An exception is made for point in (current-buffer) --\n\
its value is -not- saved.")
  ()
{
  register Lisp_Object tem;
  register int n_windows;
  register struct save_window_data *data;
  register int i;
  struct window *w;

  w = XWINDOW(XWINDOW(minibuf_window)->prev); 
  n_windows = count_windows (w);
  data = (struct save_window_data *)
           XVECTOR (Fmake_vector (make_number (SAVE_WINDOW_DATA_SIZE),
				  Qnil));
  data->current_window = selected_window; 
  data->screen_base_window = XROOT(w->root)->ewin;
  XSET (data->current_buffer, Lisp_Buffer, current_buffer);
  data->minibuf_scroll_window = Vminibuf_scroll_window;
  tem = Fmake_vector (make_number (n_windows), Qnil);
  data->saved_windows = tem;
  for (i = 0; i < n_windows; i++)
    XVECTOR (tem)->contents[i]
      = Fmake_vector (make_number (SAVED_WINDOW_VECTOR_SIZE), Qnil);
  save_window_save (XWINDOW (minibuf_window)->prev,
		    XVECTOR (tem),
		    0, n_windows);
  XSET (tem, Lisp_Window_Configuration, data);
  return (tem);
}

static int
save_window_save (window, vector, i, maxwindow)
     Lisp_Object window;
     struct Lisp_Vector *vector;
     int i;
     int maxwindow;
{
  register struct saved_window *p;
  register struct window *w;
  register Lisp_Object tem;

  for (;!NULL (window); window = w->next)
    {
      /* If you get a crash here, you may be seeing a very weird bug.
	 When it happened to me, it seems that count_windows returned
	 a value that was too small--only two, when there were two
	 visible windows, a parent, and the minibuffer (inactive).
	 If this starts happening for you, please run under a debugger
	 with a breakpoint at the abort, so that you can at least try calling
	 count_windows again to see if it will lose again.
	 If it does, you can find the bug.  */
      if (i == maxwindow)
	abort ();

      p = SAVED_WINDOW_N (vector, i);
      w = XWINDOW (window);

      XFASTINT (w->temslot) = i++;

      p->window = window;
      p->buffer = w->buffer;
      p->hscroll = w->hscroll;
      p->pixleft = w->pixleft;
      p->pixtop = w->pixtop;
      p->pixwidth = w->pixwidth;
      p->pixheight = w->pixheight;
      if (!NULL (w->buffer))
	{
	  /* Save w's value of poit in the window configuration.
	     If w is selected window, then get the value of point
	     from the buffer; pointm is garbage in the selected window. */
	  if (EQ (window, selected_window))
	    {
	      p->pointm = Fmake_marker();
	      Fset_marker(p->pointm, BUF_PT(XBUFFER(w->buffer)),
			  w->buffer);
	    }
	  else
	    p->pointm = Fcopy_marker (w->pointm);

	  p->start = Fcopy_marker (w->start);
	  p->start_at_line_beg = w->start_at_line_beg;

	  tem = XBUFFER (w->buffer)->mark;
	  p->mark = Fcopy_marker (tem);
	}
      else
	{
	  p->pointm = Qnil;
	  p->start = Qnil;
	  p->mark = Qnil;
	  p->start_at_line_beg = Qnil;
	}

      if (NULL (w->parent))
	p->parent = Qnil;
      else
	p->parent = XWINDOW (w->parent)->temslot;

      if (NULL (w->prev))
	p->prev = Qnil;
      else
	p->prev = XWINDOW (w->prev)->temslot;

      if (!NULL (w->vchild))
	i = save_window_save (w->vchild, vector, i, maxwindow);
      if (!NULL (w->hchild))
	i = save_window_save (w->hchild, vector, i, maxwindow);
    }

  return i;
}

DEFUN ("save-window-excursion", Fsave_window_excursion, Ssave_window_excursion,
  0, UNEVALLED, 0,
  "Execute body, preserving window sizes and contents.\n\
Restores which buffer appears in which window, where display starts,\n\
as well as the current buffer.\n\
Does not restore the value of point in current buffer.")
  (args)
     Lisp_Object args;
{
  register Lisp_Object val;
  register int count = specpdl_ptr - specpdl;

  record_unwind_protect (Fset_window_configuration,
			 Fcurrent_window_configuration ());
  val = Fprogn (args);
  unbind_to (count);
  return val;
}

init_window_once ()
{
  extern Lisp_Object get_minibuffer ();
  register Lisp_Object root_window;

  root_window = make_window ();
  minibuf_window = make_window ();

  XWINDOW (root_window)->next = minibuf_window;
  XWINDOW (minibuf_window)->prev = root_window;

  /* These values 9 and 10 are arbitrary,
     just so that there is "something there."
     Correct values are put in in init_xdisp */

  Fset_window_buffer (root_window, Fcurrent_buffer ());
  Fset_window_buffer (minibuf_window, get_minibuffer (0));

  selected_window = root_window;
  /* Make sure this window seems more recently used than
     a newly-created, never-selected window. */
  XFASTINT (XWINDOW (selected_window)->use_time) = ++window_select_count;
}

syms_of_window ()
{
  Qwindowp = intern ("windowp");
  staticpro (&Qwindowp);

  /* Make sure all windows get marked */
  staticpro (&minibuf_window);

  DEFVAR_INT ("minibuffer-prompt-width", &minibuf_prompt_width,
    "Width of the prompt appearing at the start of the minibuffer window.\n\
The value is meaningless when the minibuffer is not visible.");

  DEFVAR_LISP ("temp-buffer-show-hook", &Vtemp_buffer_show_hook,
    "Non-nil means call as function to display a help buffer.\n\
Used by with-output-to-temp-buffer.");
  Vtemp_buffer_show_hook = Qnil;

  DEFVAR_LISP ("minibuffer-scroll-window", &Vminibuf_scroll_window,
    "Non-nil means it is the window that C-M-v in minibuffer should scroll.");
  Vminibuf_scroll_window = Qnil;

  DEFVAR_BOOL ("pop-up-windows", &pop_up_windows,
    "*Non-nil means display-buffer should make new windows.");
  pop_up_windows = 1;

  DEFVAR_INT ("next-screen-context-lines", &next_screen_context_lines,
    "*Number of lines of continuity when scrolling by screenfuls.");
  next_screen_context_lines = 2;

  DEFVAR_INT ("split-height-threshold", &split_height_threshold,
    "*display-buffer would prefer to split the largest window if this large.\n\
If there is only one window, it is split regardless of this value.");
  split_height_threshold = 500;

  DEFVAR_INT ("window-min-height", &window_min_height,
    "*Delete any window less than this tall (including its mode line).");
  window_min_height = 4;

  DEFVAR_INT ("window-min-width", &window_min_width,
    "*Delete any window less than this wide.");
  window_min_width = 10;

  defsubr (&Sselected_window);
  defsubr (&Sminibuffer_window);
  defsubr (&Swindowp);
  defsubr (&Spos_visible_in_window_p);
  defsubr (&Swindow_buffer);
  defsubr (&Swindow_height);
  defsubr (&Swindow_width);
  defsubr (&Swindow_pixheight);  
  defsubr (&Swindow_pixwidth);
  defsubr (&Swindow_hscroll);
  defsubr (&Sset_window_hscroll);
  defsubr (&Swindow_edges);
  defsubr (&Swindow_pixedges);
  defsubr (&Swindow_point);
  defsubr (&Swindow_start);
  defsubr (&Sset_window_point);
  defsubr (&Sset_window_start);
  defsubr (&Snext_window);
  defsubr (&Sprevious_window);
  defsubr (&Sother_window);
  defsubr (&Sget_lru_window);
  defsubr (&Sget_largest_window);
  defsubr (&Sget_buffer_window);
  defsubr (&Sdelete_other_windows);
  defsubr (&Sdelete_windows_on);
  defsubr (&Sreplace_buffer_in_windows);
  defsubr (&Sdelete_window);
  defsubr (&Sset_window_buffer);
  defsubr (&Sselect_window);
  defsubr (&Sdisplay_buffer);
  defsubr (&Ssplit_window);
  defsubr (&Senlarge_window);
  defsubr (&Sshrink_window);
  defsubr (&Sscroll_up);
  defsubr (&Sscroll_down);
  defsubr (&Sscroll_left);
  defsubr (&Sscroll_right);
  defsubr (&Sscroll_other_window);
  defsubr (&Srecenter);
  defsubr (&Smove_to_window_line);
  defsubr (&Sset_window_configuration);
  defsubr (&Scurrent_window_configuration);
  defsubr (&Ssave_window_excursion);
}

keys_of_window ()
{
  ndefkey (Vctl_x_map, '1', "delete-other-windows");
  ndefkey (Vctl_x_map, '2', "split-window");
  ndefkey (Vctl_x_map, '0', "delete-window");
  ndefkey (Vctl_x_map, 'o', "other-window");
  ndefkey (Vctl_x_map, '^', "enlarge-window");
  ndefkey (Vctl_x_map, '<', "scroll-left");
  ndefkey (Vctl_x_map, '>', "scroll-right");

  ndefkey (Vglobal_map, Ctl ('V'), "scroll-up");
  ndefkey (Vesc_map, Ctl ('V'), "scroll-other-window");
  ndefkey (Vesc_map, 'v', "scroll-down");

  ndefkey (Vglobal_map, Ctl('L'), "recenter");
  ndefkey (Vesc_map, 'r', "move-to-window-line");
}
