/*
 * This file is part of tela the Tensor Language.
 * Copyright (c) 1994 Pekka Janhunen
 */

#pragma interface

#ifndef OBJECT_H

#include "def.H"
#include "error.H"
#include "common.H"
#include "ctinfo.H"

extern int TObjectNewCounter, TObjectDelCounter;

enum Tkind {				// Object kinds (types)
	Kint,					// Integer scalar
	Kreal,					// Real scalar
	Kcomplex,				// Complex scalar
	KIntArray,				// Integer array (n-dimensional)
	KRealArray,				// Real array
	KComplexArray,			// Complex array
	KObjectArray,			// Object (pointer) array, currently not used
	Kfunction,				// User-defined function, written in Tela
	KCfunction,				// Compiled and linked C-Tela function
	KIntrinsicFunction,		// Special "functions" generating inline code: abs, min, max ..
	Kvoid,					// Empty value, when printed prints nothing
	Kundef					// Undefined value, the default for new symbols
};

#define COMPTYPE Tint
#define DEFAULT_COMPVALUE 0
#define LINEARLIST TIntLL
#  include "templ/tLL.H"
#undef LINEARLIST
#undef DEFAULT_COMPVALUE
#undef COMPTYPE

class Tprg;
class Tnode;

typedef int (*TIntrinsicCompilerPtr)
    (const Tnode*in[], const int Nargin, const Tnode*out[], const int Nargout, const int c, Tprg& prg);

#define COMPTYPE TObjectPtr
#define DEFAULT_COMPVALUE 0
#define LINEARLIST TObjectLL
#  include "templ/tLL.H"
#undef LINEARLIST
#undef DEFAULT_COMPVALUE
#undef COMPTYPE

class TDimPack {
 private:
	int len;				// len = the product  dims[0]*...*dims[Ndims-1]
	int dims[MAXRANK];		// The dimensions
	short Ndims;			// Number of dimensions, small positive integer
 public:
	// --- constructors
	TDimPack() {Ndims=0; len=0;}
	TDimPack(int d1) {dims[0]=d1; Ndims=1; len=d1;}
	TDimPack(int d1, int d2) {dims[0]=d1; dims[1]=d2; Ndims=2; len=d1*d2;}
	TDimPack(const int ds[], int n);
	TDimPack(const TDimPack& dimpack);
	// --- assignment & comparison
	TDimPack& operator=(const TDimPack& dimpack);
	TDimPack& operator=(int N) {len=N; dims[0]=N; Ndims=1; return *this;}
	int operator==(const TDimPack& dimpack) const;
	int operator!=(const TDimPack& dimpack) const {return !(operator==)(dimpack);}
	// --- indexing & inquiry functions
	int operator[](int d) const {return dims[d];}
	int length() const {return len;}
	int rank() const {return Ndims;}
	// --- I/O
	friend ostream& operator<<(ostream& o, const TDimPack& dimpack);
	friend istream& operator>>(istream& i, TDimPack& dimpack);
	// --- destructor
	~TDimPack() {}
};

class Tprg;
struct TCFunctionInfo;

class Tobject {
 private:
	union {
		struct {
			Tint i;				// scalar integer object's value
			int charflag;		// true if integer is to be displayed as a character
		} integer;
		Treal x;				// scalar real object's value
		Tprg* funcptr;			// pointer to function, if k==Kfunction
		struct {
			TCFunctionPtr ptr;			// pointer to C-function, if k==KCfunction
			TCFunctionInfo* infoptr;	// pointer back to C-functioninfo, where minin..maxout info is located
		} Cfunc;
		struct {
			TIntrinsicCompilerPtr ptr;		// pointer to intrisic 'compiler', if k==KIntrinsicFunction
			int code;						// opcode
		} intr;
		struct {
			union {
				Tint *iptr;				// pointer to Tint array, if k==KIntArray
				Treal *xptr;			// pointer to Treal array, if k==KRealArray
				Tcomplex *zptr;			// pointer to Tcomplex array, if k==KComplexArray
				TObjectPtr *opptr;		// pointer TobjectPtr array, if k==KObjectArray
			};
			int strflag;		// flag for string, meaningful if k==KIntArray
		} a;					// a as array-info
    };
	Tcomplex z;			// scalar complex object's value
	TDimPack dimpack;	// rank & dims of an array object
	Tkind k;			// tag for object's type
	int tempflag;		// flag for temporary object
	void clear();		// unallocate if an array object. After operation, object is VOID.
	void setsize(TPtrInt sz);	// make object an array of sz bytes. After operation object is undefined.
	void copydims(const Tobject& obj);
	void copykind(const Tobject& obj);
	int IntArrayAllNonzero() const;
 public:
	// --- these were first private but are now public, silvousplait!
	void copydimsIntArray(const Tobject& obj);
	void copydimsRealArray(const Tobject& obj);
	void copydimsComplexArray(const Tobject& obj);

	// --- constructors
	Tobject() {k=Kundef; global::NTobject++; tempflag=0;}
	Tobject(Tint a) {k=Kint; integer.i=a; integer.charflag=0; global::NTobject++; tempflag=0;}
	Tobject(Tchar ch) {k=Kint; integer.i=ch; integer.charflag=1; global::NTobject++; tempflag=0;}
	Tobject(Treal a) {k=Kreal; x=a; global::NTobject++; tempflag=0;}
	Tobject(Tcomplex a) {k=Kcomplex; z=a; global::NTobject++; tempflag=0;}
	Tobject(Tprg* ptr) {k=Kfunction; funcptr=ptr; global::NTobject++; tempflag=0;}
	Tobject(TCFunctionPtr ptr) {k=KCfunction; Cfunc.ptr=ptr; global::NTobject++; tempflag=0;}
	Tobject(TIntrinsicCompilerPtr ptr) {k=KIntrinsicFunction; intr.ptr=ptr; global::NTobject++; tempflag=0;}
	Tobject(Treal x, Treal y);
	Tobject(const Tint itab[], int N, int stringflag=0);
	Tobject(const Treal xtab[], int N);
	Tobject(const Tcomplex ztab[], int N);
	Tobject(const TObjectPtr otab[], int N);
	Tobject(const Tint itab[], const TDimPack& dp);
	Tobject(const Treal xtab[], const TDimPack& dp);
	Tobject(const Tcomplex ztab[], const TDimPack& dp);
	Tobject(const TObjectPtr otab[], const TDimPack& dp);
	Tobject(const Tchar *str);
	Tobject(const Tobject& obj);

	// --- assignments
	Tobject& operator=(const Tobject& obj);
	Tobject& operator=(Tint a) {clear(); k=Kint; integer.i=a; integer.charflag=0; return *this;}
	Tobject& operator=(Treal a) {clear(); k=Kreal; x=a; return *this;}
	Tobject& operator=(Tprg* ptr) {clear(); k=Kfunction; funcptr=ptr; return *this;}
	Tobject& operator=(TCFunctionPtr ptr) {clear(); k=KCfunction; Cfunc.ptr=ptr; return *this;}
	Tobject& operator=(TIntrinsicCompilerPtr ptr) {clear(); k=KIntrinsicFunction; intr.ptr=ptr; return *this;}
	Tobject& operator=(const Tcomplex& a) {clear(); k=Kcomplex; z=a; return *this;}
	void izeros(const TDimPack& dp);	// set to zeroed int array of given dims
	void rzeros(const TDimPack& dp);	// set to zeroed real array of given dims
	void czeros(const TDimPack& dp);	// set to zeroed complex array of given dims
	void ozeros(int N);					// set to nulled object vector of given length
	void ireserv(const TDimPack& dp);	// set to uninitialized int array of given dims
	void rreserv(const TDimPack& dp);	// set to uninitialized real array of given dims
	void creserv(const TDimPack& dp);	// set to uninitialized complex array of given dims
	void SetToVoid() {k=Kvoid;}			// set to void value
	void SetToUndefined() {k=Kundef;}	// set to undefined value
	void SetStringFlag() {a.strflag=1;}	// set string flag (assuming it is IntArray already)
    void SetCharFlag() {integer.charflag=1;} // assume it is Kint already
	void ClearStringFlag() {a.strflag=0;}	// unset string flag (assuming it is IntArray already)
	void ClearCharFlag() {integer.charflag=0;}	// assume it is Kint already

		// --- comparison
	int operator==(const Tobject& obj) const;
	int operator!=(const Tobject& obj) const {return ! (operator==(obj));}

	// --- inquiry functions
	Tkind kind() const {return k;}
	int length() const {return dimpack.length();}
	Tint rank() const {return dimpack.rank();}
	Tint IntValue() const {return integer.i;}
	Treal RealValue() const {return x;}
	const Tcomplex& ComplexValue() const {return z;}
	Tint& IntRef() {return integer.i;}
	Treal& RealRef() {return x;}
	Tcomplex& ComplexRef() {return z;}
	Tprg* FunctionValue() const {return funcptr;}
	TCFunctionPtr& CFunctionRef() {return Cfunc.ptr;}
	TCFunctionPtr CFunctionPtr() const {return Cfunc.ptr;}
	const TCFunctionInfo* CFunctionInfoPtr() const {return Cfunc.infoptr;}
	const TCFunctionInfo*& CFunctionInfoPtrRef() {return Cfunc.infoptr;}
	TIntrinsicCompilerPtr IntrinsicCompilerPtr() const {return intr.ptr;}
	TIntrinsicCompilerPtr& IntrinsicCompilerPtrRef() {return intr.ptr;}
	int IntrinsicCode() const {return intr.code;}
	int& IntrinsicCodeRef() {return intr.code;}
	Tint *IntPtr() const {return a.iptr;}
	Treal *RealPtr() const {return a.xptr;}
	Tcomplex *ComplexPtr() const {return a.zptr;}
	TObjectPtr *ObjectPtrPtr() const {return a.opptr;}
	int IsArray() const {return (k>=KIntArray && k<=KObjectArray);}
	int IsScalar() const {return (k<=Kcomplex);}
	int IsFunction() const {return k==Kfunction;}
	int IsCfunction() const {return k==KCfunction;}
	int IsIntrinsicFunction() const {return k==KIntrinsicFunction;}
	int IsString() const {return k==KIntArray && dimpack.rank()==1 && a.strflag;}
	int IsChar() const {return k==Kint && integer.charflag;}
	int IsNonzero() const;
	int IsTemporary() const {return tempflag;}
	void FlagTemporary() {tempflag=1;}
	void UnflagTemporary() {tempflag=0;}
	const TDimPack& dims() const {return dimpack;}

	// --- other operations
	friend ostream& operator<<(ostream& o, const Tobject& obj);
	void flatten();
	void bitwiseMoveFrom(Tobject& obj);
	void SetNewDimPack(const TDimPack& dp) {dimpack=dp;}		// DANGEROUS!!!!! Used by reshape only.

	// --- destructor
	~Tobject() {clear(); global::NTobject--;}
};

// With bitwiseMoveFrom you can 'copy' an object quickly without
// moving the data. See usage examples in templ/tbinop.C.

inline void Tobject::bitwiseMoveFrom(Tobject& obj) {
	clear();
	memcpy(this,&obj,sizeof(Tobject));
	obj.SetToVoid();
}

// Class Tshort is a wrapper class, when outputted, it prints in short form

class Tshort {
 private:
	const Tobject* ptr;
 public:
	Tshort(const Tobject& obj) {ptr = &obj;}
	friend ostream& operator<<(ostream& o, const Tshort& sh);
};

#define OBJECT_H

#endif
