#!/usr/local/bin/perl
#
# File:   rand_mon.pl
# Version: 1.0
# Author: John Lame (lame@clark.net, john.lame@gsc.gte.com)
#
# Description:
#  This file contains subroutines used for randomizing
#  the r_info.txt file distributed with Angband and 
#  its' variants.
#
# Revision History:
#  date:          name:            action:
#  032598         J. Lame          Created version 1.0

# Monster speed has a huge impact on difficulty.
# As monster depth is always preserved, the next
# best thing is to modify the experience whenever
# the speed has been altered.  The experience
# awarded for killing a monster is 
# speed_exp_mod(old_speed, new_speed, old_exp).
# The result is that an a% increase (or decrease)
# in energy (obtained from speed via the
# extract_energy[] array taken from the source),
# induces (roughly, and for small values of a)
# an ($SPEED_EXP_DEG * a)% increase (or decrease)
# in experience.   The actual formula is
# new_exp = 
#   old_exp * 
#     ((new_energy/old_energy) ** $SPEED_EXP_DEG) 
sub speed_exp_mod {
  my $old_speed = shift @_;
  my $new_speed = shift @_;
  my $old_exp = shift @_;
  my $new_exp = $old_exp;
  if ($SPEED_EXP_DEG > 0) {
    $old_speed = ($old_speed < 0 ? 0 : $old_speed);
    $old_speed = ($old_speed >199 ? 199 : $old_speed);
    $new_speed = ($new_speed < 0 ? 0 : $new_speed);
    $new_speed = ($new_speed > 199 ? 199 : $new_speed);
    my $old_energy = $extract_energy[$old_speed];
    my $new_energy = $extract_energy[$new_speed];
    $new_exp *= (($new_energy/$old_energy) ** $SPEED_EXP_DEG);
    $new_exp = int($new_exp);
  }
  return $new_exp;
}

# lint(x) returns the least integer greater than
# or equal to x
sub lint {
  my $x = shift @_;
  return ($x == int($x) ? $x : int($x+1));
}

# min(x,y) returns the minimum of x and y
sub min {
  my $x = shift @_;
  my $y = shift @_;
  return ($x < $y ? $x : $y);
}

# max(x,y) returns the maximum of x and y
sub max {
  my $x = shift @_;
  my $y = shift @_;
  return ($x > $y ? $x : $y);
}

# randint(x,y) returns a random integer between 
# int(x) and int(y) inclusive.
sub randint {
  my $x = int(shift @_);
  my $y = int(shift @_);
  return int(rand(abs($y-$x)+1)+&min($x,$y));
}

# d_ave(x,y,fm) returns ( fm ? x*y : x*((1+y)/2) )
sub d_ave {
  my $x = shift @_;
  my $y = shift @_;
  my $fm = shift @_;
  return ($fm ? $x*$y : $x*((1+$y)/2));
}  

# random_dice(x0,y0,fm,p,q) returns (x1,y1)
# somewhat randomly chosen so that
# (1-p)*d_ave(x0,y0,fm) <= d_ave(x1,y1,fm) and
# d_ave(x1,y1,fm) <= (1+q)*d_ave(x0,y0,fm).
# This isn't the best random dice generator
# by any means.  It behaves very nicely in
# certain circumstances (for instance,
# random_dice(10,10,0,.1,.1) returns one
# of 9 possible outcomes with roughly
# equal probability) and it behaves very
# badly in others (for instance, 
# random_dice(10,10,0,.11,.11) yields the
# same result every time).
sub random_dice {
  my $x0 = int(shift @_);
  my $y0 = int(shift @_);
  my $fm = shift @_;
  my $p = shift @_;
  my $q = shift @_;
  my $N0 = &d_ave($x0,$y0,$fm);

  $x0 = ($x0 < 1 ? 1 : $x0);
  $y0 = ($y0 < 1 ? 1 : $y0);
  $p = ((($p < 0) || ($p >= 1)) ? 0 : $p);
  $q = ($q < 0 ? 0 : $q);
  my $x1 = &randint(&lint($x0*(1-$p)),int($x0*(1+$q)));
  my $y1 = 0;
  if ($fm) {
    $y1 = &randint(&lint($N0*(1-$p)/$x1),int($N0*(1+$q)/$x1));
  }
  else {
    $y1 = 
      &randint
        (
         &max(1,&lint(2*$N0*(1-$p)/$x1)-1),
         int(2*$N0*(1+$q)/$x1)-1
        );
  }    
  return ($x1,$y1);
}

# (remove_duplicates @_) returns @_ sorted and with
# all identical entrys deleted.
sub remove_duplicates {
  my @sorted_array = sort @_;
  my $i;
  for ($i = 0; $i < (@sorted_array - 1); $i++) {
    if ($sorted_array[$i] == $sorted_array[$i+1]) {
      splice(@sorted_array,$i,1);
      $i--;
    }
  }
  return @sorted_array;    
}

# (strip $the_string) returns $the_string with all whitespace
# and unprintable characters removed.  
# (strip $the_string $save_space) will preserve spaces.
sub strip {
  my $the_string = shift @_;
  my $save_space = shift @_;
  if ($save_space) {
    $the_string =~ s/[^\040-\176]//g;
  }
  else {
    $the_string =~ s/[^\041-\176]//g;
  }
  return $the_string;
}

# (in_list $the_element "S" @the_list ) returns true if
# the_element is in the list as a string, false otherwise.
# (in_list $the_element "N" @the_list ) returns true if
# the_element is in the list as a number, false otherwise.
sub in_list {
    my $the_element = shift @_;
    my $the_type = shift @_;
    my @the_list = @_;
    my $the_answer = (0 == 1);
    foreach $el (@the_list) {
      if ($the_type eq "N") {
        $the_answer ||= ($el == $the_element);
      }
      else {
        $the_answer ||= ($el eq $the_element);
      }
    }
    return $the_answer;
}

# build_c(\@F, \@C) uses the frequency array @F
# to build a counting array @C.  F[0] and C[0]
# are ignored for these purposes.  For example,
# if @F = (0, 0,1,3,0 ,4, 0), then
#    @C = (0,-1,1,4,-5,8,-9).
# The sum of the elements of F, 7 in this example, 
# is returned.  This subroutine assumes that
# @F is an array of N+1 non-negative integers.  In
# this case, we call F a "frequency" array of
# length N and total T.  In the above example, N=6
# and T=7. 
sub build_c {
  my $fref = shift @_;
  my $cref = shift @_;
  my $tot = 0;
  my $i = 0;
  for ($i=1; $i < @$fref; $i++) {
    if ($$fref[$i] == 0) {
      $$cref[$i] = -$tot-1;
    }
    else {
      $tot += $$fref[$i];
      $$cref[$i] = $tot;
    }
  }
  return($tot);
}

# get_flag(\@C,$R,$fmin,$fmax) returns the unique
# n such that:
# fmin <= n <= fmax,
# R <= C[n]
# R >= C[n-1]
# C[n] > 0.
# If no such unique n exists, then the output of this
# function is ill-behaved.  If the array C is obtained
# by applying the build_c subroutine to a valid 
# frequency array F of length N and total T, and if i 
# ranges between 1 and T, then get_flag(\@C,i,1,N) will
# return the value j exactly F[j] times for each j
# between 1 and N, hence the term "frequency array".
sub get_flag {
  my $cref = shift @_;
  my $r = shift @_;
  my $fmin = shift @_;
  my $fmax = shift @_;
  my $fhat = 0;
  my $flag = 0;

  #print "Computing get_flag(C, $r,$fmin,$fmax)\n";

  if ($fmax-$fmin <= 1) {
    $flag = ($r > $$cref[$fmin] ? $fmax : $fmin);
  }
  else {
    $fhat = int(($fmin+$fmax)/2);
    if (($r > abs($$cref[$fhat])) || ($r == -$$cref[$fhat])) {
      $flag = &get_flag($cref,$r,$fhat+1,$fmax);
    }
    else {
      $flag = &get_flag($cref,$r,$fmin,$fhat);
    }
  }
  return($flag);
}

# print_frequencies is used for debugging purposes.
# Note that it outputs over 1M of text.
sub print_frequencies {
  print "B1 flags\n";
  print "###########\n";
  for ($n=0; $n <= 100; $n++) {
    print "Level $n\n";
    for ($j=1; $j < @b1flags; $j++) {
      print "  \"$b1flags[$j]\" OLD_FRQ=";
      print ${$b1freq{$b1flags[$j]}}[$n];
      print " NEW_FRQ=";
      print $b1dun[$n][$j];
      print " OLD_RNG = (";
      print ${$b1depth{$b1flags[$j]}}[0];
      print ", ";
      print ${$b1depth{$b1flags[$j]}}[1];
      print ")";
      print " NEW_RNG = (";
      print ${$b1depth{$b1flags[$j]}}[0] - $B1_D_MOD;
      print ", ";
      print ${$b1depth{$b1flags[$j]}}[1] + $B1_S_MOD;
      print ")";
      print "\n";
    }
    print "\n";
  }
  print "B2 flags\n";
  print "###########\n";
  for ($n=0; $n <= 100; $n++) {
    print "Level $n\n";
    for ($j=1; $j < @b2flags; $j++) {
      print "  \"$b2flags[$j]\" OLD_FRQ=";
      print ${$b2freq{$b2flags[$j]}}[$n];
      print " NEW_FRQ=";
      print $b2dun[$n][$j];
      print " OLD_RNG = (";
      print ${$b2depth{$b2flags[$j]}}[0];
      print ", ";
      print ${$b2depth{$b2flags[$j]}}[1];
      print ")";
      print " NEW_RNG = (";
      print ${$b2depth{$b2flags[$j]}}[0] - $B2_D_MOD;
      print ", ";
      print ${$b2depth{$b2flags[$j]}}[1] + $B2_S_MOD;
      print ")";
      print "\n";
    }
    print "\n";
  }
  print "F flags\n";
  print "###########\n";
  for ($n=0; $n <= 100; $n++) {
    print "Level $n\n";
    for ($j=1; $j < @fflags; $j++) {
      print "  \"$fflags[$j]\" OLD_FRQ=";
      print ${$ffreq{$fflags[$j]}}[$n];
      print " NEW_FRQ=";
      print $fdun[$n][$j];
      print " OLD_RNG = (";
      print ${$fdepth{$fflags[$j]}}[0];
      print ", ";
      print ${$fdepth{$fflags[$j]}}[1];
      print ")";
      print " NEW_RNG = (";
      print ${$fdepth{$fflags[$j]}}[0] - $F_D_MOD;
      print ", ";
      print ${$fdepth{$fflags[$j]}}[1] + $F_S_MOD;
      print ")";
      print "\n";
    }
    print "\n";
  }
  print "S flags\n";
  print "###########\n";
  for ($n=0; $n <= 100; $n++) {
    print "Level $n\n";
    for ($j=1; $j < @sflags; $j++) {
      print "  \"$sflags[$j]\" OLD_FRQ=";
      print ${$sfreq{$sflags[$j]}}[$n];
      print " NEW_FRQ=";
      print $sdun[$n][$j];
      print " OLD_RNG = (";
      print ${$sdepth{$sflags[$j]}}[0];
      print ", ";
      print ${$sdepth{$sflags[$j]}}[1];
      print ")";
      print " NEW_RNG = (";
      print ${$sdepth{$sflags[$j]}}[0] - $S_D_MOD;
      print ", ";
      print ${$sdepth{$sflags[$j]}}[1] + $S_S_MOD;
      print ")";
      print "\n";
    }
    print "\n";
  } 
}

return 1;
