/* frv specific support for Elf loading and relocation.
   Copyright 2004 Red Hat, Inc

   Contributed by Clark Williams <williams@redhat.com>
   
   based on obj_i386.c by Richard Henderson
   (plus patches from Alexandre Oliva <aoliva@redhat.com>)

   This file is part of the Linux modutils.

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2 of the License, or (at your
   option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <string.h>
#include <assert.h>

#include <module.h>
#include <obj.h>
#include <util.h>


/*======================================================================*/

struct frv_got_entry
{
  int offset;
  unsigned offset_done : 1;
  unsigned reloc_done : 1;
};

struct frv_file
{
  struct obj_file root;
  struct obj_section *got;
};

struct frv_symbol
{
  struct obj_symbol root;
  struct frv_got_entry gotent;
};


/*======================================================================*/

struct obj_file *
arch_new_file (void)
{
  struct frv_file *f;
  f = xmalloc(sizeof(*f));
  f->got = NULL;
  return &f->root;
}

struct obj_section *
arch_new_section (void)
{
  return xmalloc(sizeof(struct obj_section));
}

struct obj_symbol *
arch_new_symbol (void)
{
  struct frv_symbol *sym;
  sym = xmalloc(sizeof(*sym));
  memset(&sym->gotent, 0, sizeof(sym->gotent));
  return &sym->root;
}

int
arch_load_proc_section(struct obj_section *sec, int fp)
{
    /* Assume it's just a debugging section that we can safely
       ignore ...  */
    sec->contents = NULL;

    return 0;
}

enum obj_reloc
arch_apply_relocation (struct obj_file *f,
		       struct obj_section *targsec,
		       struct obj_section *symsec,
		       struct obj_symbol *sym,
		       Elf32_Rela *rel,
		       Elf32_Addr v)
{
  struct frv_file *ifile = (struct frv_file *)f;
  struct frv_symbol *isym  = (struct frv_symbol *)sym;

  Elf32_Addr *loc = (Elf32_Addr *)(targsec->contents + rel->r_offset);
  Elf32_Addr dot = targsec->header.sh_addr + rel->r_offset;
  Elf32_Addr got = ifile->got ? ifile->got->header.sh_addr : 0;

  enum obj_reloc ret = obj_reloc_ok;

  switch (ELF32_R_TYPE(rel->r_info))
    {
    case R_FRV_NONE:
      break;

    case R_FRV_32:
      {
	void *loc_unalign;
	asm ("" : "=r" (loc_unalign) : "0" (loc));
	if ((int)loc_unalign & 3)
	  memcpy (loc_unalign, &v, sizeof (*loc));
	else
	  *loc = v;
      }
      break;

    case R_FRV_LABEL16:
      v -= dot;
      if (v % 4)
	ret = obj_reloc_dangerous;
      if (v >= 0x20000UL && v < -0x20000UL)
	ret = obj_reloc_overflow;
      v >>= 2;
      *loc &= ~0xffff;
      *loc |= v & 0xffff;
      break;
		
    case R_FRV_LABEL24:
      v -= dot;
      if (v % 4)
	ret = obj_reloc_dangerous;
      if (v >= 0x2000000UL && v < -0x2000000UL)
	ret = obj_reloc_overflow;
      v >>= 2;
      *loc &= ~0x7e03ffff;
      *loc |= ((v & 0xfc0000) << 7) | (v & 0x3ffff) ;
      break;

    case R_FRV_HI16:
      v >>= 16;
      /* Fall through.  */
    case R_FRV_LO16:
      *loc &= ~0xffff;
      *loc |= v & 0xffff;
      break;
		
#if 0
    case R_FRV_PLT32:
    case R_FRV_PC32:
      *loc += v - dot;
      break;

    case R_FRV_GLOB_DAT:
    case R_FRV_JMP_SLOT:
      *loc = v;
      break;

    case R_FRV_RELATIVE:
      *loc += f->baseaddr;
      break;

    case R_FRV_GOTPC:
      assert(got != 0);
      *loc += got - dot;
      break;

    case R_FRV_GOT32:
      assert(isym != NULL);
      if (!isym->gotent.reloc_done)
	{
	  isym->gotent.reloc_done = 1;
	  *(Elf32_Addr *)(ifile->got->contents + isym->gotent.offset) = v;
	}
      *loc += isym->gotent.offset;
      break;

    case R_FRV_GOTOFF:
      assert(got != 0);
      *loc += v - got;
      break;
#endif

    default:
      ret = obj_reloc_unhandled;
      break;
    }

  return ret;
}

int
arch_create_got (struct obj_file *f)
{
  return 1;
}

int
arch_init_module (struct obj_file *f, struct module *mod)
{
  return 1;
}

int
arch_finalize_section_address(struct obj_file *f, Elf32_Addr base)
{
  int  i, n = f->header.e_shnum;

  f->baseaddr = base;
  for (i = 0; i < n; ++i)
    f->sections[i]->header.sh_addr += base;
  return 1;
}

int
arch_archdata (struct obj_file *fin, struct obj_section *sec)
{
  return 0;
}
