/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  TransactionMajorState,
  TransactionMinorState,
  TransactionType,
  WithdrawalType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import { createSimpleTestkudosEnvironmentV3 } from "../harness/environments.js";

/**
 * Test for a withdrawal that is externally confirmed.
 */
export async function runWithdrawalExternalTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange } =
    await createSimpleTestkudosEnvironmentV3(t);

  // Create a withdrawal operation

  const bankUser = await bankClient.createRandomBankUser();
  bankClient.setAuth(bankUser);
  const wop = await bankClient.createWithdrawalOperation(
    bankUser.username,
    "TESTKUDOS:10",
  );

  const talerWithdrawUri = wop.taler_withdraw_uri + "?external-confirmation=1";

  // Hand it to the wallet

  const detResp = await walletClient.call(
    WalletApiOperation.GetWithdrawalDetailsForUri,
    {
      talerWithdrawUri: talerWithdrawUri,
    },
  );

  const acceptResp = await walletClient.call(
    WalletApiOperation.AcceptBankIntegratedWithdrawal,
    {
      exchangeBaseUrl: detResp.defaultExchangeBaseUrl!!,
      talerWithdrawUri,
    },
  );

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: acceptResp.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.BankConfirmTransfer,
    },
  });

  const txDetails = await walletClient.call(
    WalletApiOperation.GetTransactionById,
    {
      transactionId: acceptResp.transactionId,
    },
  );

  // Now we check that the external-confirmation=1 flag actually did something!

  t.assertDeepEqual(txDetails.type, TransactionType.Withdrawal);
  t.assertDeepEqual(
    txDetails.withdrawalDetails.type,
    WithdrawalType.TalerBankIntegrationApi,
  );
  t.assertDeepEqual(txDetails.withdrawalDetails.externalConfirmation, true);
  t.assertDeepEqual(txDetails.withdrawalDetails.bankConfirmationUrl, undefined);

  t.logStep("confirming withdrawal operation");

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: acceptResp.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.BankConfirmTransfer,
    },
  });

  await bankClient.confirmWithdrawalOperation(bankUser.username, {
    withdrawalOperationId: wop.withdrawal_id,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});
}

runWithdrawalExternalTest.suites = ["wallet"];
