/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  encodeCrock,
  ExchangeWalletKycStatus,
  hashFullPaytoUri,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  configureCommonKyc,
  createKycTestkudosEnvironment,
  postAmlDecisionNoRules,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

export async function runKycExchangeWalletTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, exchange, amlKeypair } =
    await createKycTestkudosEnvironment(t, {
      adjustExchangeConfig(config) {
        configureCommonKyc(config);

        config.setString("KYC-RULE-R1", "operation_type", "balance");
        config.setString("KYC-RULE-R1", "enabled", "yes");
        config.setString("KYC-RULE-R1", "exposed", "yes");
        config.setString("KYC-RULE-R1", "is_and_combinator", "yes");
        config.setString("KYC-RULE-R1", "threshold", "TESTKUDOS:5");
        config.setString("KYC-RULE-R1", "timeframe", "forever");
        config.setString("KYC-RULE-R1", "next_measures", "M1");

        config.setString("KYC-MEASURE-M1", "check_name", "C1");
        config.setString("KYC-MEASURE-M1", "context", "{}");
        config.setString("KYC-MEASURE-M1", "program", "NONE");

        config.setString("KYC-CHECK-C1", "type", "INFO");
        config.setString("KYC-CHECK-C1", "description", "my check!");
        config.setString("KYC-CHECK-C1", "fallback", "FREEZE");
      },
    });

  await walletClient.call(WalletApiOperation.AddExchange, {
    exchangeBaseUrl: exchange.baseUrl,
  });

  await walletClient.call(WalletApiOperation.StartExchangeWalletKyc, {
    amount: "TESTKUDOS:20",
    exchangeBaseUrl: exchange.baseUrl,
  });

  await walletClient.call(WalletApiOperation.TestingWaitExchangeWalletKyc, {
    amount: "TESTKUDOS:20",
    exchangeBaseUrl: exchange.baseUrl,
    passed: false,
  });

  const exchangeEntry = await walletClient.call(
    WalletApiOperation.GetExchangeEntryByUrl,
    {
      exchangeBaseUrl: exchange.baseUrl,
    },
  );

  console.log(j2s(exchangeEntry));

  t.assertDeepEqual(
    exchangeEntry.walletKycStatus,
    ExchangeWalletKycStatus.Legi,
  );

  const kycReservePub = exchangeEntry.walletKycReservePub;

  t.assertTrue(!!kycReservePub);

  // FIXME: Create/user helper function for this!
  const hPayto = hashFullPaytoUri(
    `payto://taler-reserve-http/localhost:${exchange.port}/${kycReservePub}`,
  );

  await postAmlDecisionNoRules(t, {
    amlPriv: amlKeypair.priv,
    amlPub: amlKeypair.pub,
    exchangeBaseUrl: exchange.baseUrl,
    paytoHash: encodeCrock(hPayto),
  });

  await walletClient.call(WalletApiOperation.TestingWaitExchangeWalletKyc, {
    amount: "TESTKUDOS:20",
    exchangeBaseUrl: exchange.baseUrl,
    passed: true,
  });
}

runKycExchangeWalletTest.suites = ["wallet"];
