/* graphic.c
 *
 * Copyright (C)  1993  The Board of Trustees of  
 * The Leland Stanford Junior University.  All Rights Reserved.
 *
 * Authors: Scott Francis, Adam Fedor (CU, Boulder), and Paul Kunz
 *
 * This file is part of an Objective-C class library for a window system
 *
 * graphics.c,v 1.20 1994/07/10 21:43:29 pfkeb Exp
 */

#include "graphics.h"

#include <stdlib.h>
#include "stdmacros.h"
#include <math.h>


void NXSetRect(NXRect *aRect, NXCoord x, NXCoord y, NXCoord width, NXCoord height)
{
    if (!aRect) X_MALLOC(aRect, NXRect, 1);
    aRect->origin.x = x;
    aRect->origin.y = y;
    aRect->size.width = width;
    aRect->size.height = height;
}

void NXOffsetRect(NXRect *aRect, NXCoord dx, NXCoord dy)
{
    if (!aRect) return;
    aRect->origin.x += dx;
    aRect->origin.y += dy;
}

void NXInsetRect(NXRect *aRect, NXCoord dx, NXCoord dy)
{
    if (!aRect) return;
    NXOffsetRect(aRect, dx, dy);
    aRect->size.width -= (2*dx);
    aRect->size.height -= (2*dy);
}

NXCoord round(NXCoord x)
{
    if ((x%1) < .5) return ((NXCoord)floor((float)x));
    else return ((NXCoord)ceil((float)x));
}

void NXIntegralRect(NXRect *aRect)
{
    if (!aRect) return;
    aRect->origin.x = round((float)aRect->origin.x);
    aRect->origin.y = round((float)aRect->origin.y);
    aRect->size.width = round((float)aRect->size.width);
    aRect->size.height = round((float)aRect->size.height);
}

BOOL NXMouseInRect(const NXPoint *aPoint, const NXRect *aRect, BOOL flipped)
{
	return NXPointInRect(aPoint, aRect);
}

BOOL NXPointInRect(const NXPoint *aPoint, const NXRect *aRect)
{
	return ((aPoint->x >= NX_X(aRect)) &&
	        (aPoint->y >= NX_Y(aRect)) &&
		(aPoint->x <= NX_MAXX(aRect)) &&
		(aPoint->y <= NX_MAXY(aRect)));
}

BOOL NXIntersectsRect(const NXRect *aRect, const NXRect *bRect)
{
/*
*    return (((NX_X(aRect) < NX_X(bRect)) && (NX_MAXX(aRect) > NX_X(bRect)) && 
*             (NX_Y(aRect) > NX_Y(bRect)) && (NX_Y(aRect) < NX_MAXY(bRect))) ||
* 	    ((NX_Y(aRect) < NX_Y(bRect)) && (NX_MAXY(aRect) > NX_Y(bRect)) &&
* 	     (NX_X(aRect) > NX_X(bRect)) && (NX_X(aRect) < NX_MAXX(bRect))));
*/
    if ( NX_MAXX(aRect) <= NX_X(bRect)    || 
            NX_X(aRect) >= NX_MAXX(bRect) || 
         NX_MAXY(aRect) <= NX_Y(bRect)    || 
            NX_Y(aRect) >= NX_MAXY(bRect)    ) {
    	return NO;
    }
    return YES;
}
    
BOOL NXContainsRect(const NXRect *aRect, const NXRect *bRect) 
{
    if (NXEqualRect(aRect,bRect)) return YES;
    return ((NX_X(aRect) <= NX_X(bRect)) &&
    	    (NX_Y(aRect) <= NX_Y(bRect)) &&
	    (NX_MAXX(aRect) >= NX_MAXX(bRect)) &&
	    (NX_MAXY(aRect) >= NX_MAXY(bRect)));
}
BOOL NXEqualRect(const NXRect *aRect, const NXRect *bRect)
{
    return ((NX_X(aRect) == NX_X(bRect)) &&
    	    (NX_Y(aRect) == NX_Y(bRect)) &&
	    (NX_WIDTH(aRect) == NX_WIDTH(bRect)) &&
	    (NX_HEIGHT(aRect) == NX_HEIGHT(bRect)));
}

BOOL NXEmptyRect(const NXRect *aRect)
{
    return (!((NX_WIDTH(aRect) > 0) && (NX_HEIGHT(aRect) > 0)));
}

NXRect *NXUnionRect(const NXRect *aRect, NXRect *bRect)
{
    if (NXEmptyRect(aRect) && NXEmptyRect(bRect)) {
    	bRect->origin.x = 0;
	bRect->origin.y = 0;
	bRect->size.width = 0;
	bRect->size.height = 0;
    	return bRect;
    }
    else if (NXEmptyRect(aRect)) return bRect;
    else if (NXEmptyRect(bRect))
    {
	bRect->origin.x    = aRect->origin.x;
	bRect->origin.y    = aRect->origin.y;
	bRect->size.width  = aRect->size.width;
	bRect->size.height = aRect->size.height;
	return bRect;
    }
    bRect->origin.x = MIN(NX_X(aRect), NX_X(bRect));
    bRect->origin.y = MIN(NX_Y(aRect), NX_Y(bRect));
    bRect->size.width = MAX(NX_MAXX(aRect), NX_MAXX(bRect)) - bRect->origin.x;
    bRect->size.height = MAX(NX_MAXY(aRect), NX_MAXY(bRect)) - bRect->origin.y;
    return bRect;
}

NXRect *NXCopyRect(NXRect *aRect)
{
	NXRect *bRect;
	bRect = (NXRect *)malloc(sizeof(NXRect));
	NXSetRect(bRect, aRect->origin.x, aRect->origin.y, aRect->size.width,
			 aRect->size.height);
	return bRect;
}

NXRect *NXDivideRect(NXRect *aRect, NXRect *bRect, NXCoord slice, int edge)
{
    if (NXEmptyRect(aRect)) {
    	NXSetRect(bRect, 0,0,0,0);
	return bRect;
    }
    switch (edge) {
    case (NX_XMIN):
    	if (slice > aRect->size.width) 
	{
	    NXSetRect(bRect, aRect->origin.x, aRect->origin.y, 
	    	      aRect->size.width, aRect->size.height);
	    break;
	}
	NXSetRect(bRect, aRect->origin.x, aRect->origin.y, slice, 
		  aRect->size.height);
	break;
    case (NX_YMIN):
    	if (slice > aRect->size.height) return (bRect = NXCopyRect(aRect));
	NXSetRect(bRect, aRect->origin.x, aRect->origin.y, aRect->size.width,
		  slice);
	break;
    case (NX_XMAX):
    	if (slice > aRect->size.width) return (bRect = NXCopyRect(aRect));
	NXSetRect(bRect, NX_MAXX(aRect)-slice, aRect->origin.y, slice,
		  aRect->size.height);
	break;
    case (NX_YMAX):
    	if (slice > aRect->size.height) return (bRect = NXCopyRect(aRect));
	NXSetRect(bRect, aRect->origin.x, NX_MAXY(aRect)-slice, 
		  aRect->size.width, slice);
	break;
    default:
    	printf("error in divideRect\n");
	break;
    }
    return bRect;
}


/* Received from 
Adam Fedor. CU, Boulder             | Never blame on enthalpy what can be 
fedor@boulder.colorado.edu (W)      |   attributed to entropy.
*/

NXRect *NXIntersectionRect(const NXRect *aRect, NXRect *bRect)
{
    if (!NXIntersectsRect(aRect, bRect)) {
        NXSetRect(bRect, 0, 0, 0, 0);
        return bRect;
    }
    if (NX_X(aRect) <= NX_X(bRect))
        NX_WIDTH(bRect) = MIN(NX_MAXX(aRect), NX_MAXX(bRect)) - NX_X(bRect);
    else {
        NX_WIDTH(bRect) = MIN(NX_MAXX(aRect), NX_MAXX(bRect)) - NX_X(aRect);
        NX_X(bRect) = NX_X(aRect);
    }
    if (NX_Y(aRect) <= NX_Y(bRect))
        NX_HEIGHT(bRect) = MIN(NX_MAXY(aRect), NX_MAXY(bRect)) - NX_Y(bRect);
    else {
        NX_HEIGHT(bRect) = MIN(NX_MAXY(aRect), NX_MAXY(bRect)) - NX_Y(aRect);
        NX_Y(bRect) = NX_Y(aRect);
    }

    return bRect;
}

/* Color/Graphics functions */
NXColorSpace 
NXColorSpaceFromDepth (NXWindowDepth depth)
{
    return (NXColorSpace)((depth & 0xf00) >> 16);
}

int 
NXBPSFromDepth (NXWindowDepth depth)
{
    return (int)(depth & 0x00f);
}

int 
NXNumberOfColorComponents (NXColorSpace colorSpace)
{
    int number;

    switch(colorSpace) {
    case NX_CustomColorSpace:
	number = 0;
	break;
    case NX_OneIsBlackColorSpace:
    case NX_OneIsWhiteColorSpace:
	number = 1;
	break;
    case NX_RGBColorSpace:
    case NX_CMYKColorSpace:
	number = 3;
	break;
    }
    return number;
}

BOOL 
NXGetBestDepth (NXWindowDepth *depth, int numColors, int bps)
{
    if (numColors > 1 && numColors < 5) {
	if (bps > 12)
	    *depth = NX_TwentyFourBitRGBDepth;
	else
	    *depth = NX_TwelveBitRGBDepth;
    } else if (numColors == 1) {
	if (bps > 2)
	    *depth = NX_EightBitGrayDepth;
	else
	    *depth = NX_TwoBitGrayDepth;
    } else
	return NO;
    return YES;
}

/* Since GNU Objective-C archving doesn't support float, 
 * we'll use integers 
 */
void 
NXReadRect (NXTypedStream *stream, NXRect *aRect)
{
    int		array[4];
    
    objc_read_array(stream, "i", 4, array);
    aRect->origin.x    = array[0];
    aRect->origin.y    = array[1];
    aRect->size.width  = array[2];
    aRect->size.height = array[3];
    return;    
}  
   
void 
NXWriteRect (NXTypedStream *stream, const NXRect *aRect)
{
 /* change to float when GNU Objective-C archiving supports it.*/
    int		array[4];
    
    array[0] = aRect->origin.x;
    array[1] = aRect->origin.y;
    array[2] = aRect->size.width;
    array[3] = aRect->size.height;
    return objc_write_array(stream, "i", 4, array);
    return;    
}     
