/*
    tiff - Functions for dealing with tiff images

    Copyright (C) 1994, Adam Fedor

    Some of this code is derived from xtiff, by Dan Sears. See the
    copyright below.
*/

/*
 * xtiff - view a TIFF file in an X window
 *
 * Dan Sears
 * Chris Sears
 *
 * Copyright 1991 by Digital Equipment Corporation, Maynard, Massachusetts.
 *
 *                      All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of Digital not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * Notes:
 *
 * TIFFTAG_COLORMAP is often incorrectly written as ranging from
 * 0 to 255 rather than from 0 to 65535.  CheckAndCorrectColormap()
 * takes care of this.
 *
 * Only ORIENTATION_TOPLEFT is supported correctly.  This is the
 * default TIFF and X orientation.  Other orientations will be
 * displayed incorrectly.
 *
 * There is no support for or use of 3/3/2 DirectColor visuals.
 * TIFFTAG_MINSAMPLEVALUE and TIFFTAG_MAXSAMPLEVALUE are not supported.
 *
 * Only TIFFTAG_BITSPERSAMPLE values that are 1, 2, 4 or 8 are supported.
 */
#include <math.h>
#include <stdlib.h>
#include <streams/streams.h>
#include "appkit/tiff.h"
#include "appkit/stdmacros.h"

#define SCALE(x, s)     (((x) * 65535L) / (s))
#define MCHECK(m)       \
	if (!m) { fprintf(stderr, "tiff: Malloc failed.\n"); return(NULL); }

static const char *
type_for_mode(int mode)
{
    char *type = NULL;

    if (mode == NX_READONLY)
        type = "r";
    else if (mode == NX_WRITEONLY)
        type = "w";
    else if (mode == NX_READWRITE)
        type = "rw";

    return type;
}

#ifdef HAVE_LIBTIFF
/* Open a tiff file. Returns NULL if can't read tiff information */
TIFF *
NXOpenTiffFile(const char *filename, int mode)
{
    return TIFFOpen((char *)filename, (char *)type_for_mode(mode));
}

/* Open a tiff from a stream. Returns NULL if can't read tiff information.
   Note that the libtiff library has to be patched so that it uses streams
   instead of file descriptors.
*/
TIFF *
NXOpenTiffStream(NXStream *stream, int mode)
{
    return TIFFFdOpen((int)stream, NULL, (char *)type_for_mode(mode));
}

int  
NXCloseTiff(TIFF *image)
{
    TIFFClose(image);
    return 0;
}

/* Read some information about the image. Note that currently we don't
   determine numImages.
*/
NXTiffInfo *      
NXGetTiffInfo(int imageNumber, TIFF *image)
{
    NXTiffInfo *info;


    if (imageNumber >= 0 && !TIFFSetDirectory(image, imageNumber)) {
	return NULL;
    }

    NX_MALLOC(info, NXTiffInfo, 1);
    memset(info, 0, sizeof(NXTiffInfo));
    if (imageNumber >= 0)
    	info->imageNumber = imageNumber;

    TIFFGetField(image, TIFFTAG_IMAGEWIDTH, &info->width);
    TIFFGetField(image, TIFFTAG_IMAGELENGTH, &info->height);
    TIFFGetField(image, TIFFTAG_COMPRESSION, &info->compression);
    TIFFGetField(image, TIFFTAG_SUBFILETYPE, &info->subfileType);

    /*
     * If the following tags aren't present then use the TIFF defaults.
     */
    TIFFGetFieldDefaulted(image, TIFFTAG_BITSPERSAMPLE, &info->bitsPerSample);
    TIFFGetFieldDefaulted(image, TIFFTAG_SAMPLESPERPIXEL, 
	&info->samplesPerPixel);
    TIFFGetFieldDefaulted(image, TIFFTAG_PLANARCONFIG, 
	&info->planarConfig);

    /*
     * If TIFFTAG_PHOTOMETRIC is not present then assign a reasonable default.
     * The TIFF 5.0 specification doesn't give a default.
     */
    if (!TIFFGetField(image, TIFFTAG_PHOTOMETRIC,
            &info->photoInterp)) {
	char *redMap, *greenMap, *blueMap;
        if (info->samplesPerPixel != 1)
            info->photoInterp = PHOTOMETRIC_RGB;
        else if (info->bitsPerSample == 1)
            info->photoInterp = PHOTOMETRIC_MINISBLACK;
        else if (TIFFGetField(image, TIFFTAG_COLORMAP,
                redMap, greenMap, blueMap)) {
            info->photoInterp = PHOTOMETRIC_PALETTE;
        } else
            info->photoInterp = PHOTOMETRIC_MINISBLACK;
    }

    return info;

}

#define READ_SCANLINE(sample) \
	if ( TIFFReadScanline( image, buf, row, sample ) < 0 ) { \
	    fprintf(stderr, "tiff: bad data read on line %d\n", row ); \
	    error = 1; \
	    break; \
	} \
	inP = buf;

/* Read an image into a data array.  The data array is assumed to have been
   already allocated to the correct size, unless it is NULL, in which case,
   this routine allocates the data and returns a pointer to this data.

   Note that 8-bit color images are implicitly coverted to 24-bit contig
   direct color images.
*/
void *
NXReadTiff(int imageNumber, TIFF *image, NXTiffInfo *info, void *data)
{
    int     i;
    int     row, col;
    int     maxval;
    int	    size;
    int	    line;
    int	    error = 0;
    u_char *inP, *outP;
    u_char *buf;
    u_char *imageData;
    NXTiffInfo *newinfo;

    /* Make sure we're at the right image */
    if ((newinfo = NXGetTiffInfo(imageNumber, image)) == NULL)
	return NULL;

    if (info)
	memcpy(info, newinfo, sizeof(NXTiffInfo));

    imageData = (u_char *)data;
    maxval = ( 1 << newinfo->bitsPerSample ) - 1;
    line   = ceil((float)newinfo->width * newinfo->bitsPerSample / 8.0);
    size   = ceil((float)line * newinfo->height * newinfo->samplesPerPixel );
    if (!data) {
	NX_MALLOC(imageData, u_char, size);
	MCHECK(imageData);
    }
    NX_MALLOC(buf, u_char, TIFFScanlineSize(image));
    MCHECK(buf);

    outP = imageData;
    switch ( newinfo->photoInterp ) {
	case PHOTOMETRIC_MINISBLACK:
    	    for ( row = 0; row < newinfo->height; ++row ) {
		READ_SCANLINE(0)
	        for ( col = 0; col < line; col++)
		    *outP++ = *inP++;
	    }
	    break;
	case PHOTOMETRIC_MINISWHITE:
    	    for ( row = 0; row < newinfo->height; ++row ) {
		READ_SCANLINE(0)
	        for ( col = 0; col < line; col++)
		    *outP++ = maxval - *inP++;
	    }
	    break;
	case PHOTOMETRIC_PALETTE:
	    {
	    /* Need to get colormap and convert to 24-bit image */
	    NXColormap *map;
	    map = NXGetColormap(image);
	    NX_REALLOC(imageData, u_char, size*3);
	    MCHECK(imageData);

    	    for ( row = 0; row < newinfo->height; ++row ) {
		READ_SCANLINE(0)
	        for ( col = 0; col < newinfo->width; col++) {
		    *outP++ = map->red[*inP] / 256;
		    *outP++ = map->green[*inP] / 256;
		    *outP++ = map->blue[*inP] / 256;
		    inP++;
		}
	    }
	    free(map->red);
	    free(map->green);
	    free(map->blue);
	    free(map);
	    }
	    break;
	case PHOTOMETRIC_RGB:
	    if (newinfo->planarConfig == PLANARCONFIG_CONTIG) {
    	        for ( row = 0; row < newinfo->height; ++row ) {
		    READ_SCANLINE(0)
	            for ( col = 0; col < newinfo->width; col++) 
			for (i = 0; i < newinfo->samplesPerPixel; i++)
			    *outP++ = *inP++;
		}
	    } else {
		for (i = 0; i < newinfo->samplesPerPixel; i++)
    	            for ( row = 0; row < newinfo->height; ++row ) {
		    	READ_SCANLINE(i)
	            	for ( col = 0; col < newinfo->width; col++) 
			    *outP++ = *inP++;
		    }
	    }
	    break;

	default:
	    fprintf(stderr, "tiff: Can't read photometric %d\n", 
		newinfo->photoInterp);
	    break;
    }

    free(newinfo);
    NX_FREE(buf);
    if (error) {
	if (!data)
	    free(imageData);
	return NULL;
    }
    return imageData;
}

int  
NXWriteTiff(TIFF *image, NXTiffInfo *info, void *data)
{
    return 0;
}

/*------------------------------------------------------------------------*/

/*
 * Many programs get TIFF colormaps wrong.  They use 8-bit colormaps instead of
 * 16-bit colormaps.  This function is a heuristic to detect and correct this.
 */
static void
CheckAndCorrectColormap(NXColormap *map)
{
    register int i;

    for (i = 0; i < map->size; i++)
        if ((map->red[i] > 255)||(map->green[i] > 255) || (map->blue[i] > 255))
            return;

    for (i = 0; i < map->size; i++) {
        map->red[i] = SCALE(map->red[i], 255);
        map->green[i] = SCALE(map->green[i], 255);
        map->blue[i] = SCALE(map->blue[i], 255);
    }
    fprintf(stderr, "tiff: Assuming 8-bit colormap");
}

/* Gets the colormap for the image if there is one. Returns a
   NXColormap if one was found.
*/
NXColormap *
NXGetColormap(TIFF *image)
{
    int      i;
    int	     mapped;
    double  *dRed, *dGreen, *dBlue;
    NXTiffInfo *info;
    NXColormap *map;

    /* Re-read the tiff information. We pass -1 as the image number which
       means just read the current image.
    */
    info = NXGetTiffInfo(-1, image);
    NX_MALLOC(map, NXColormap, 1);
    map->size = 1 << info->bitsPerSample;

    dRed = (double *) malloc(map->size * sizeof(double));
    dGreen = (double *) malloc(map->size * sizeof(double));
    dBlue = (double *) malloc(map->size * sizeof(double));
    MCHECK(dRed); MCHECK(dGreen); MCHECK(dBlue);

    /*
     * Given TIFFTAG_PHOTOMETRIC extract or create the response curves.
     */
    mapped = 0;
    switch (info->photoInterp) {
    case PHOTOMETRIC_PALETTE:
	mapped = 1;
        if (TIFFGetField(image, TIFFTAG_COLORMAP,
                &map->red, &map->green, &map->blue)) {
            CheckAndCorrectColormap(map);
            for (i = 0; i < map->size; i++) {
                dRed[i] = (double) map->red[i];
                dGreen[i] = (double) map->green[i];
                dBlue[i] = (double) map->blue[i];
            }
        } else {
            map->red = (u_short *) malloc(map->size * sizeof(u_short));
            map->green = (u_short *) malloc(map->size * sizeof(u_short));
            map->blue = (u_short *) malloc(map->size * sizeof(u_short));
	    MCHECK(map->red); MCHECK(map->green); MCHECK(map->blue);
	    for (i = 0; i < map->size; i++)
		dRed[i] = dGreen[i] = dBlue[i]
		    = (double) SCALE(i, map->size - 1);
	}
        break;
    default:
	break;
    }

    if (!mapped) {
	free(map);
	return NULL;
    }

    free(dRed);
    free(dGreen);
    free(dBlue);
    free(info);
    return map;
}

#else /* HAVE_LIBTIFF */

TIFF *
NXOpenTiffFile(const char *filename, int mode)
{
    return NULL;
}

TIFF *
NXOpenTiffStream(NXStream *stream, int mode)
{
    return NULL;
}

int  
NXCloseTiff(TIFF *image)
{
    return 0;
}

NXTiffInfo *      
NXGetTiffInfo(int imageNumber, TIFF *image)
{
    return NULL;
}

void *
NXReadTiff(int imageNumber, TIFF *image, NXTiffInfo *info, void *data)
{
    return NULL;
}

int  
NXWriteTiff(TIFF *image, NXTiffInfo *info, void *data)
{
    return -1;
}

NXColormap *
NXGetColormap(TIFF *image)
{
    return NULL;
}

#endif /* not HAVE_LIBTIFF */
