/* Implementation of NXBrowser class
 *
 * Copyright (C)  1994  The Board of Trustees of  
 * The Leland Stanford Junior University.  All Rights Reserved.
 *
 * Authors: Mike Gravina, Fred Harris, Paul Kunz, 
 *          Imran Qureshi, and Libing Wang
 *	    Mike L. Kienenberger (Alaska)
 *
 * This file is part of an Objective-C class library for a window system
 *
 * NXBrowser.m,v 1.57 1994/06/11 03:56:21 pfkeb Exp
 */

/* Implementation notes:
 *
 * Temporarily using MList object for the vertical scrolling columns.
 * This class uses Motif List widget.   Will convert to using Matrix
 * when Matrix class can handle the job
 *
 * shadowList contains a list of matrices corresponding to the columns
 * so that OpenStep API is correct even tho MList objects are created.
 */


#include "NXBrowser.h"

#include "Box.h"
#include "NXBrowserCell.h"
#include "Matrix.h"
#include "Scroller.h"
#include "ScrollView.h"
#include "TextField.h"
#include "MList.h"
#include <coll/List.h>
#include "Text.h"
#include <objc2/hashtable.h>
#include "stdmacros.h"

#define MLIST_WIDTH_DIFF 26

@implementation NXBrowser : Control


/* Private methods not declared in interface */

- _calcColumnFrame:(NXRect *)aRect at:(int) column
{
    aRect->origin.x   = column * colWidth;
    aRect->size.width = colWidth - 4;
    if ( isTitled ) {
	aRect->origin.y    = titleHeight;
	aRect->size.height = frame.size.height - titleHeight;
    } else {
	aRect->origin.y    = 0;
	aRect->size.height = frame.size.height;
    }
    if ( hasHorizontalScroller ) {
        aRect->size.height -= (NX_SCROLLERWIDTH + 4) ;
    }
    return self;
}

- _loadColumn:(int)column
{
    ScrollView		*sview;
    Matrix		*matrix;
    MList		*mlist;
    NXBrowserCell	*proto;
    NXRect		rect = { {0,0}, {0,0} };	/* dummy for now */
    int			rows, visblecol;
    
    matrix = [shadowList objectAt:column];
    if ( !matrix ) {
	proto = [cellPrototype copy];
	matrix = [[matrixClass alloc] initFrame:&rect mode:NX_LISTMODE
				    prototype:proto numRows:0 numCols:1];
	[shadowList addObject:matrix];
    } else {
	[matrix renewRows:0 cols:1];
    }
    rows = [delegate browser:self fillMatrix:matrix inColumn:column];
    visblecol = column - columnOfViewZero;
    if ( (visblecol >= 0) && (visblecol < numVisibleCols) ) {
	sview = [columnList objectAt:visblecol];
	mlist = [sview docView];
	[mlist fillListFrom:matrix];
	[self reflectScroll:[mlist superview]];
    }
    return self;
}
- (MList *)_createMList
{
    MList	*mlist;
    NXRect	rect = { {0,0}, {0,0} };
    
    rect.size.width = colWidth - MLIST_WIDTH_DIFF;
    rect.size.height = 10;	/* will re-adjust when filled */
    mlist = [[MList alloc] initFrame:&rect];
    [mlist setTarget: self];
    [mlist setAction:@selector(doClick:)];
    [mlist setDoubleAction:@selector(doDoubleClick:)];
    return mlist;
}

- _emptyColumnsFrom:(int)column;
{
    Matrix             *matrix;
    MList              *mlist;
    ScrollView		*sview;
    int                 i, j, count;

    count = [shadowList count];
    for ( i = count -1; i >= column; i-- ) {
	matrix = [shadowList removeObjectAt:i];
	[matrix free];
	j = i - columnOfViewZero;
	if ( j < numVisibleCols ) {
	    sview = [columnList objectAt:j];
	    mlist = [sview docView];
	    [mlist empty];
	}
    }
    return self;
}

- _tileContents
{
 /* BUG: only handling titleFromPrev state */
 
    NXBrowserCell	*aCell;
    ScrollView		*sview;
    Matrix		*matrix;
    MList		*mlist;
    const char		*title;
    float		d, pos, percent;
    int			i, j, selrow;
        
    if ( isTitled ) {
	if (columnOfViewZero == 0 ) {
	    if ( colOneTitle ) {
		title = colOneTitle;
	    } else {
		title = "/";
	    }
	} else {
	    matrix = [shadowList objectAt:columnOfViewZero - 1];
	    aCell = [matrix selectedCell];
	    title = [aCell stringValue];
	}
    }
    for ( i = 0; i < numVisibleCols; i++ ) {
        j = columnOfViewZero + i;
	matrix = [shadowList objectAt:j];
	selrow = [matrix selectedRow];
	sview = [columnList objectAt:i];
	mlist = [sview docView];
	[mlist fillListFrom:matrix];
	[self reflectScroll:[mlist superview]];
	if (selrow >=0 ) {
	    [mlist selectCellAt:selrow];
	}
	if ( isTitled ) {
	    [self setTitle:title ofColumn:i];
	    aCell  = [matrix selectedCell];
	    title = [aCell stringValue];
	}
    }
    if ( horizScroller ) {
	d = [shadowList count];
	percent = numVisibleCols / d;
	percent = MIN(1.0, percent);
	d -= numVisibleCols;
	if ( columnOfViewZero ) {
	    pos = columnOfViewZero / d;
	} else {
	    pos = 0.0;
	}
	[horizScroller setFloatValue:pos :percent];
    }
    return self;
}

- _initFrame:(const NXRect *)frameRect
{
  /* invoked by both -initFrame: and -awake */
  
    [super initFrame:frameRect];
    instancename = "NXBrowser";
    colWidth = frame.size.width / numVisibleCols;
    shadowList = [[List alloc] initCount:numVisibleCols];
    [self setMatrixClass:[Matrix class]];
    [self setCellClass:[NXBrowserCell class]];
    pathSeparator = '/';
    return self;
}

/* Public methods */

- initFrame:(const NXRect *)frameRect
{
    numVisibleCols = 1;
    titleHeight = 20;
    isTitled = YES;
    [self getTitleFromPreviousColumn:YES];
    minColWidth = 100;
    [self _initFrame:frameRect];
    [self setHorizontalScrollerEnabled:YES];
    return self;
}

- free
{
    [shadowList freeObjects];
    [shadowList free];
    [columnList free]; /* objects in it will be freed as subviews */
    [titleList free];
    [horizScroller free];
    return [super free];
}

- (SEL)action 
{
    return  action;
}

- setAction:(SEL)aSelector
{
    action = aSelector;
    return self;
}

- target
{
    return target;
}

- setTarget:anObject
{
    target = anObject;
    return self;
}

- (SEL)doubleAction
{
    if (doubleAction)
        return doubleAction;
    else
        return action;
}

- setDoubleAction:(SEL)aSelector
{
    doubleAction = aSelector;
    return self;
}

- setMatrixClass:classId
{
    matrixClass = classId;
    return self;
}

- setCellClass:classId
{
    id		proto;
    
    proto = [[classId alloc] init];
    if ( ![proto isKindOf:[NXBrowserCell class]] ) {
        [proto free];
        return self;
    }
    cellClass = classId;
    if ( cellPrototype ) {
    	[cellPrototype free];
    }
    cellPrototype = proto;
    return self;
}

- cellPrototype
{
    return cellPrototype;
}

- setCellPrototype:aCell
{
    NXBrowserCell	*oldCell;
    
    if ( ![aCell isKindOf:[NXBrowserCell class]] ) {
    	return nil;
    }
    oldCell = cellPrototype;
    cellPrototype = aCell;
    return oldCell;
}

- delegate
{
    return delegate;
}

- setDelegate:aDelegate
{
    delegate = aDelegate;
    return  self;
}


// - setEnabled:(BOOL)flag;
// - setMultipleSelectionEnabled:(BOOL)flag;
// - (BOOL)isMultipleSelectionEnabled;
// - setBranchSelectionEnabled:(BOOL)flag;
// - (BOOL)isBranchSelectionEnabled;
// - setEmptySelectionEnabled:(BOOL)flag;
// - (BOOL)isEmptySelectionEnabled;

- setHorizontalScrollerEnabled:(BOOL)flag
{    
    if ( flag == hasHorizontalScroller ) {
        return self;
    }
    hasHorizontalScroller = flag;
    [self tile];
    return self;
}

- (BOOL)isHorizontalScrollerEnabled
{
    return hasHorizontalScroller;
}


// - setHorizontalScrollButtonsEnabled:(BOOL)flag;
// - (BOOL)areHorizontalScrollButtonsEnabled;
// - reuseColumns:(BOOL)flag;
// - separateColumns:(BOOL)flag;
// - (BOOL)columnsAreSeparated;
// - useScrollButtons:(BOOL)flag;
// - useScrollBars:(BOOL)flag;

- getTitleFromPreviousColumn:(BOOL)flag
{
    Matrix		*matrix;
    NXBrowserCell	*aCell;
    int                 i, j, count;

    if ( titleFromPrev == flag ) 
    	return self;
    titleFromPrev = flag;
    if (flag && [self isTitled] ) {
	if (colOneTitle ) {
	    [self setTitle:colOneTitle ofColumn:0];
	} else {
	    [self setTitle:"/" ofColumn:0];
	}
	matrix = [columnList objectAt:0];
	j = [self columnOf:matrix];
        count = [columnList count];
	for ( i = 1; i < count; i++, j++ ) {
	    matrix = [shadowList objectAt:j];
	    aCell = [matrix selectedCell];
	    [self setTitle:[aCell stringValue] ofColumn:i];
	}
    }
    return self;
}

- (BOOL)isTitled
{
    return isTitled;
}

- setTitled:(BOOL)flag
{
    if ( flag == isTitled ) 
        return self;
	
    isTitled = flag;
    if (flag) {
	if (titleHeight <= 0) {
	    titleHeight = 20;
	}
    }
    [self tile];
    return self;
}


// - (NXRect *)getTitleFrame:(NXRect *)theRect ofColumn:(int)column;

- setTitle:(const char *)aString ofColumn:(int)column
{
    [[titleList objectAt:column] setStringValue:aString];
    return self;
}

- (const char *)titleOfColumn:(int)column 
{
    const char *result;

    if (column > ([columnList count] -1) )  {
        result = NULL;
    } else {
        result = [[titleList objectAt:column] stringValue];
    }
    return result;
}

- drawTitle:(const char *)aTitle inRect:(const NXRect *)aRect
    ofColumn:(int)column
{
    [[titleList objectAt:column] setStringValue:aTitle];
    return self;
}

- clearTitleInRect:(const NXRect *)aRect ofColumn:(int)column
{
    [self drawTitle:"\0" inRect:aRect ofColumn:column];

    return self;
}

- (NXCoord)titleHeight
{
    return titleHeight;
}

- loadColumnZero
{
    [self _emptyColumnsFrom:1];
    columnOfViewZero = 0;
    [self _loadColumn:0];
    return self;
}


// - (BOOL)isLoaded;

- setPathSeparator:(unsigned short)charCode
{
   pathSeparator = charCode;
   return self;
}

// - setPath:(const char *)path;

- (char *)getPath:(char *)thePath toColumn:(int)column;
{
    Cell	*aCell;
    Matrix	*matrix;
    char	septemp[] = "/";
    int		i;

    if ( !thePath ) {
        return NULL;
    }
    septemp[0] = pathSeparator;
    strcpy( thePath, "" );
    for ( i = 0; i <= column; i++ ) {
        matrix = [shadowList objectAt:i];
	aCell = [matrix selectedCell];
	if ( aCell ) {
	    strcat( thePath, septemp );
	    strcat( thePath, [aCell stringValue] );
	}
    }
    return thePath;
}

// - displayColumn:(int)column;

- reloadColumn:(int)index
{
    [self _loadColumn:index];
    return self;
}

// - validateVisibleColumns;

- displayAllColumns
{
    return self;
}

- scrollColumnsRightBy:(int)shiftAmount
{
    int		i;
    
    i = columnOfViewZero - shiftAmount;
    columnOfViewZero = ( i<0 ) ? 0 : i;
    [self _tileContents];
    return self;
}

- scrollColumnsLeftBy:(int)shiftAmount
{
    int		i, last, cols = [shadowList count];
    
    i = columnOfViewZero + shiftAmount;
    last = columnOfViewZero + numVisibleCols - 1;
    if ( last >= cols ) {
        return [self setLastColumn:last-1];
    }
    columnOfViewZero = i;
    [self _tileContents];
    return self;
}

// - scrollColumnToVisible:(int)column;
// - scrollUpOrDown:sender;

- reflectScroll:clipView
{
    ScrollView	*scrollview;
    
    scrollview = [clipView superview];
    [scrollview reflectScroll:clipView];
    return self;
}

- scrollViaScroller:sender
{
    float	f;
    int		i, hit, count;
    
    hit = [sender hitPart];
    switch (hit) {
    case NX_KNOB :
        f = [sender floatValue];
	count = [shadowList count];
	if ( count > numVisibleCols ) {
	    i = rint( f * (count - numVisibleCols) );
	    if ( i == columnOfViewZero ) {
	        return self;
	    }
	    if ( i > columnOfViewZero ) {
	    	[self scrollColumnsLeftBy:1];
	    } else {
	    	[self scrollColumnsRightBy:1];
	    }
	} else {
	    return self;
	}
        break;
    case NX_INCLINE :
    case NX_INCPAGE :
        [self scrollColumnsLeftBy:1];
        break;
    case NX_DECLINE :
    case NX_DECPAGE :
	[self scrollColumnsRightBy:1];
        break;
    case NX_NOPART :
    default:
    	break;
    }
    return self;
}

// - updateScroller;

- setLastColumn:(int)column
{
    int		i, cols = [shadowList count];
        
    if ( column >= cols ) {
        return self;
    }
    i = column - numVisibleCols + 1;
    columnOfViewZero = (i < 0 ) ? 0 : i;
    [self _tileContents];
    return self;
}

- addColumn
{
    Matrix		*matrix;
    NXBrowserCell	*proto;
    NXRect		rect = { {0,0}, {0,0} };	/* dummy for now */
    int			cols = [shadowList count];

    proto = [cellPrototype copy];
    matrix = [[matrixClass alloc] initFrame:&rect mode:NX_LISTMODE
    			     prototype:proto numRows:0 numCols:1];
    [matrix setCellClass:[NXBrowserCell class]];
    [shadowList addObject:matrix];
    [self _loadColumn:cols];
    if ( cols >= numVisibleCols ) {
	[self setLastColumn:cols];
    }
    return self;
}

- setMinColumnWidth:(int)columnWidth
{
    minColWidth = columnWidth;
    return self;
}

// - (int)minColumnWidth;

- setMaxVisibleColumns:(int)columnCount
{
    numVisibleCols = (columnCount <= 0) ? 1 : columnCount;
    colWidth = frame.size.width / numVisibleCols;
    [self tile];
    return self;
}

// - (int)maxVisibleColumns;
// - (int)numVisibleColumns;
// - (int)firstVisibleColumn;
// - (int)lastVisibleColumn;
// - (int)lastColumn;

- (int)selectedColumn
{
   return  selectedColumn;
}

- selectedCell
{
    return [[shadowList objectAt:selectedColumn] selectedCell];
}

- getSelectedCells:(List *)aList
{
    return [[shadowList objectAt:selectedColumn] getSelectedCells:aList];
}

- (int)columnOf:matrix
{
    return [shadowList indexOf:matrix];
}

- matrixInColumn:(int) index
{
    return [shadowList objectAt:index];
}

// - getLoadedCellAtRow:(int)row inColumn:(int)col;
// - selectAll:sender;
// - (NXRect *)getFrame:(NXRect *)theRect ofColumn:(int)column;
// - (NXRect *)getFrame:(NXRect *)theRect ofInsideOfColumn:(int)column;

- tile
{
    MList	*mlist;
    ScrollView	*sview;
    TextField	*textfield;
    NXRect	rect;
    float	d, percent;
    int		i, initialcount;
 
 /* First get the right number of components */      
    if ( !columnList ) {
        columnList = [[List alloc] initCount:numVisibleCols];
    }
    if ( isTitled ) {
        if ( !titleList ) {
            titleList = [[List alloc] initCount:numVisibleCols];
	}
    } else {
        if ( titleList ) {
	    [titleList freeObjects];
	    [titleList free];
	}
    }
    if ( hasHorizontalScroller ) {
	rect.origin.x = 0;
	rect.origin.y = frame.size.height - NX_SCROLLERWIDTH;
	rect.size.width = frame.size.width;
	rect.size.height = NX_SCROLLERWIDTH;
	if ( !horizScroller ) {
	    horizScroller = [[Scroller alloc] initFrame:&rect];
	    [horizScroller setTarget:self];
	    [horizScroller setAction:@selector(scrollViaScroller:)];
	    [self addSubview:horizScroller];
	} else {
	    [horizScroller setFrame:&rect];
	}
        d = [shadowList count];
        if ( d > 0.0 ) {
	    percent = numVisibleCols / d;
	    percent = MIN(1.0, percent);
	} else {
	    percent = 1.0;
	}
        [horizScroller setFloatValue:0.0 :percent];
    }
    initialcount = [columnList count];
    i = initialcount - numVisibleCols;
    if ( i > 0 ) {
        while ( i-- ) {
	    if ( isTitled ) {
	        textfield = [titleList removeLastObject];
	        [textfield free];
	    }
	    sview = [columnList removeLastObject];
	    [sview free];
	}
    }
  /* Now do the tiling */
    for ( i = 0; i < numVisibleCols; i++ ) {
	if ( isTitled ) {
	    rect.origin.x = i * colWidth;
	    rect.origin.y = 0;
	    rect.size.width = colWidth;
	    rect.size.height = titleHeight;
	    if ( i >= initialcount ) {
	        textfield = [[TextField alloc] initFrame:&rect];
		[textfield _initAsLabel];
	        [textfield setAlignment:NX_CENTERED];
	        [titleList addObject:textfield];
	        [self addSubview:textfield];
	    } else {
		textfield = [titleList objectAt:i];
		[textfield setFrame:&rect];
	    }
	    if ( i == 0 ) {
	        if ( colOneTitle ) {
		    [textfield setStringValue:colOneTitle];
		} else {
		    [textfield setStringValue:"/" ];
		}
	    } else {
	    	[textfield setStringValue:"" ];
	    }
	}
	[self _calcColumnFrame:&rect at:i];
	if ( i >= initialcount ) {
	    sview = [[ScrollView alloc] initFrame:&rect];
	    [sview setVertScrollerRequired:YES];
	    [sview setBorderType:NX_BEZEL];
	    mlist = [self _createMList];
	    [sview setDocView:mlist];
	    [columnList addObject:sview];
	    [self addSubview:sview];
	} else {
	    sview = [columnList objectAt:i];
	    [sview setFrame:&rect];
	}
    }
    return self;
}

// - drawSelf:(const NXRect *)rects :(int)rectCount;
// - mouseDown:(NXEvent *)theEvent;
// - sizeTo:(NXCoord)width :(NXCoord)height;
// - sizeToFit;
// - acceptArrowKeys:(BOOL)acceptFlag andSendActionMessages:(BOOL)sendFlag;
// - keyDown:(NXEvent *)theEvent;
// - (BOOL)acceptsFirstResponder;

- doClick:sender
{
 /* sender will be a MList managing the List widget */

    List		*selList = nil;
    Matrix		*matrix;
    MList		*mlist;
    NXBrowserCell	*aCell;
    ScrollView		*sview;
    int                 i, count, newColumn;

  /* find the column of sender */
    for ( i = 0; i < numVisibleCols; i++ ) {
	sview = [columnList objectAt:i];
	mlist = [sview docView];
	if ( mlist == sender ) {
	    selectedColumn = columnOfViewZero + i;
	    break;
	}
    }
    matrix = [shadowList objectAt:selectedColumn];
    selList = [sender getSelectedCells:selList from:matrix];
    count = [selList count];
    for (i = 0; i < count; i++) {
	aCell = [selList objectAt:i];
	[matrix selectCell:aCell];
    }
    if (count == 1) {
	aCell = [selList lastObject];
	newColumn = selectedColumn + 1;
	if ( isTitled ) {
	    [self setTitle:[aCell stringValue] 
	          ofColumn:selectedColumn - columnOfViewZero + 1];
	}
	if ([aCell isLeaf]) {
	    [self _emptyColumnsFrom:newColumn];
	} else {
	    if (newColumn == [shadowList count]) {
	        if ( (newColumn < numVisibleCols)
		       || hasHorizontalScroller ) {
		    [self addColumn];
		}
	    } else {
		[self _emptyColumnsFrom:newColumn + 1];
		[self _loadColumn:newColumn];
	    }
	}
    }
    if (action && target) {
	[target perform:action with:self];
    }
    return self;
}

- doDoubleClick:sender
{
    if (!target)
	return self;
    if (doubleAction) {
	[target perform:doubleAction with:self];
    } else {
	[target perform:action with:self];
    }
    return self;
}

-  sendAction
{
    [target perform:action with:self];
    return self;
}

// + newFrame:(const NXRect *)frameRect;

- read:(TypedStream *)stream
{
    short	_reserved4[4];
    
    [super read:stream];    
    objc_read_types(stream, "cc", &hasHorizontalScroller, &isTitled);
    objc_read_type(stream, "s", &_reserved4[2]); /* number of columns */
    numVisibleCols = _reserved4[2];
    if ( isTitled ) {
        objc_read_types(stream, "i*", &titleHeight, &colOneTitle);
    }

    return self;
}

- awake
{
    [self _initFrame:&frame];
    if (isTitled) {
	[self setTitled:YES];
	[[titleList objectAt:0] setStringValue:colOneTitle];
    }
    [self tile];
    return self;
}

@end
