/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 1993, 1994, Elan Feingold (feingold@zko.dec.com)           *
 *                                                                           *
 *     PERMISSION TO USE, COPY, MODIFY, AND TO DISTRIBUTE THIS SOFTWARE      *
 *     AND ITS DOCUMENTATION FOR ANY PURPOSE IS HEREBY GRANTED WITHOUT       *
 *     FEE, PROVIDED THAT THE ABOVE COPYRIGHT NOTICE APPEAR IN ALL           *
 *     COPIES AND MODIFIED COPIES AND THAT BOTH THAT COPYRIGHT NOTICE AND    *
 *     THIS PERMISSION NOTICE APPEAR IN SUPPORTING DOCUMENTATION.  THERE     *
 *     IS NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR      *
 *     ANY PURPOSE.  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS       *
 *     OR IMPLIED WARRANTY.                                                  *
 *                                                                           *
 *****************************************************************************/

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>

#include "riskgame.h"
#include "types.h"
#include "network.h"
#include "debug.h"


/* Local prototypes */
void _NET_SendString(Int iSocket, String strString);
void _NET_SendLong(Int iSocket, Long lLong);
void _NET_RecvString(Int iSocket, String *pstrString);
void _NET_RecvLong(Int iSocket, Long *plLong);


/************************************************************************ 
 *  FUNCTION: NET_SendSyncMessage
 *  HISTORY: 
 *     03.03.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 *     This procedure assumes that there is no other async. I/O on the
 *    sockets.  You must remove all callbacks you added with XtAddInput,
 *    for example, before calling it.
 ************************************************************************/
void NET_SendSyncMessage(Int iSendSocket, Int iSendMessType, void *pvMessage, 
			 Int iRecvSocket, Int iReturnMessType,
			 void (*CBK_MessageReceived)(Int, void *))
{
  Int     iMessType;
  void   *pvMess;

  /* Send the message */
  NET_SendMessage(iSendSocket, iSendMessType, pvMessage);

  /* Loop, until we receive the desired message, dispatching all others
   * to the specified callback, which is assumed to be an X type callback,
   * with three pointers passed to it.
   */
  
  for (;;)
    {
      /* This will block when there is no input */
      NET_RecvMessage(iRecvSocket, &iMessType, &pvMess);
      
      /* If we received the message we were looking for, 
       * then dispatch it and return finally.
       */
      
      CBK_MessageReceived(iMessType, pvMess);
      NET_DeleteMessage(iMessType, pvMess);

      if (iMessType == iReturnMessType)
	return;
    }
}


/************************************************************************ 
 *  FUNCTION: NET_SendMessage
 *  HISTORY: 
 *     01.24.94  ESF  Created 
 *     01.28.94  ESF  Added strFrom in MSG_MESSAGEPACKET.
 *     03.04.94  ESF  Changed MSG_UPDATE mesage.
 *     03.05.94  ESF  Added MSG_ENTERSTATE for fault tolerance.
 *     03.28.94  ESF  Added MSG_DEADPLAYER & MSG_ENDOFGAME.
 *     03.28.94  ESF  Changed MSG_REQUESTCARDS to ...CARD.
 *     03.29.94  ESF  Changed MSG_UPDATECARDS to MSG_EXCHANGECARDS.
 *     03.29.94  ESF  Changed uiReply to be an Int.
 *     04.11.94  ESF  Added a player parameter to MSG_CARDPACKET.
 *     04.11.94  ESF  Added a killer paramter to MSG_DEADPLAYER.
 *     05.03.94  ESF  Added MSG_OBJ*UPDATE msgs.
 *     05.03.94  ESF  Removed MSG_REGISTERPLAYER and MSG_UPDATEARMIES.
 *     05.05.94  ESF  Added MSG_OBJ* msgs.
 *     05.12.94  ESF  Removed MSG_OBJ* msgs and added GAME messages.
 *     05.12.94  ESF  Added MSG_DEREGISTERCLIENT.
 *     05.12.94  ESF  Added MSG_DELETEMSGDST.
 *     05.13.94  ESF  Added MSG_STARTREGISTRATION.
 *     05.15.94  ESF  Added MSG_[FREE|ALLOC]PLAYER
 *     05.15.94  ESF  Removed MSG_DEADPLAYER.
 *     05.17.94  ESF  Added MSG_NETMESSAGE.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void NET_SendMessage(Int iSocket, Int iMessType, void *pvMessage)
{
  Int i;

  /* Send the message ID */
  _NET_SendLong(iSocket, (Long)iMessType);
	
  switch(iMessType)
    {
    case MSG_REGISTERCLIENT:  
      {
	MsgRegisterClient *pmsgMess = (MsgRegisterClient *)pvMessage;
	
	_NET_SendString(iSocket, pmsgMess->strClientAddress);
      }
      break;

    case MSG_EXCHANGECARDS:
      {
	MsgExchangeCards *pmsgMess = (MsgExchangeCards *)pvMessage;
	
	for(i=0; i!=3; i++)
	  _NET_SendLong(iSocket, (Long)pmsgMess->piCards[i]);
      }
      break;

    case MSG_CARDPACKET:
      {
	MsgCardPacket *pmsgMess = (MsgCardPacket *)pvMessage;

	_NET_SendLong(iSocket, (Long)pmsgMess->iPlayer);
	_NET_SendLong(iSocket, (Long)pmsgMess->cdCard);
      }
      break;
      
    case MSG_REPLYPACKET:
      {
	MsgReplyPacket *pmsgMess = (MsgReplyPacket *)pvMessage;

	_NET_SendLong(iSocket, (Long)pmsgMess->iReply);
      }
      break;

    case MSG_SENDMESSAGE:
      {
	MsgSendMessage *pmsgMess = (MsgSendMessage *)pvMessage;

	_NET_SendString(iSocket, pmsgMess->strMessage);
	_NET_SendString(iSocket, pmsgMess->strDestination);
      } 
      break;

    case MSG_MESSAGEPACKET:
      {
	MsgMessagePacket *pmsgMess = (MsgMessagePacket *)pvMessage;

	_NET_SendString(iSocket, pmsgMess->strMessage);
	_NET_SendString(iSocket, pmsgMess->strFrom);
	_NET_SendLong(iSocket, pmsgMess->iTo);
      }
      break;

    case MSG_TURNNOTIFY:
      {
	MsgTurnNotify *pmsgMess = (MsgTurnNotify *)pvMessage;
	
	_NET_SendLong(iSocket, pmsgMess->iPlayer);
	_NET_SendLong(iSocket, pmsgMess->iClient);
      }
      break;
      
    case MSG_CLIENTIDENT:
      {
	MsgClientIdent *pmsgMess = (MsgClientIdent *)pvMessage;
	_NET_SendLong(iSocket, pmsgMess->iClientID);
      }
      break;

    case MSG_REQUESTCARD:
      {
	MsgRequestCard *pmsgMess = (MsgRequestCard *)pvMessage;
	_NET_SendLong(iSocket, pmsgMess->iPlayer);
      }
      break;

    case MSG_ENTERSTATE:
      {
	MsgEnterState *pmsgMess = (MsgEnterState *)pvMessage;
	_NET_SendLong(iSocket, pmsgMess->iState);
      }
      break;

    case MSG_OBJSTRUPDATE:
      {
	MsgObjStrUpdate *pmsgMess = (MsgObjStrUpdate *)pvMessage;
	_NET_SendLong(iSocket, pmsgMess->iField);
	_NET_SendLong(iSocket, pmsgMess->iIndex1);
	_NET_SendLong(iSocket, pmsgMess->iIndex2);
	_NET_SendString(iSocket, pmsgMess->strNewValue);
      }
      break;

    case MSG_OBJINTUPDATE:
      {
	MsgObjIntUpdate *pmsgMess = (MsgObjIntUpdate *)pvMessage;
	_NET_SendLong(iSocket, pmsgMess->iField);
	_NET_SendLong(iSocket, pmsgMess->iIndex1);
	_NET_SendLong(iSocket, pmsgMess->iIndex2);
	_NET_SendLong(iSocket, pmsgMess->iNewValue);
      }
      break;

    case MSG_DELETEMSGDST:
      {
	MsgDeleteMsgDst *pmsgMess = (MsgDeleteMsgDst *)pvMessage;
	_NET_SendLong(iSocket, pmsgMess->iClient);
      }
      break;

    case MSG_FREEPLAYER:
      {
	MsgFreePlayer *pmsgMess = (MsgFreePlayer *)pvMessage;
	_NET_SendLong(iSocket, pmsgMess->iPlayer);
      }
      break;

    case MSG_NETMESSAGE:
      {
	MsgNetMessage *pmsgMess = (MsgNetMessage *)pvMessage;
	_NET_SendString(iSocket, pmsgMess->strMessage);
      }
      break;

    case MSG_EXIT:
    case MSG_STARTGAME:
    case MSG_ENDTURN:
    case MSG_ENDOFGAME:
    case MSG_GAMENEWPLAYERS:
    case MSG_GAMEMOREPLAYERS:
    case MSG_GAMEANOTHER:
    case MSG_DEREGISTERCLIENT:
    case MSG_STARTREGISTRATION:
    case MSG_ALLOCPLAYER:
      /* No parameters */
      break;

    default:
      printf("Illegal message!\n");
      
    }
}


/************************************************************************ 
 *  FUNCTION: NET_RecvMessage
 *  HISTORY: 
 *     01.24.94  ESF  Created.
 *     01.27.94  ESF  Fixed bug in MSG_MESSAGEPACKET case.
 *     01.28.94  ESF  Added strFrom in MSG_MESSAGEPACKET.
 *     03.04.94  ESF  Changed _UPDATE mesage.
 *     03.05.94  ESF  Added _ENTERSTATE for fault tolerance.
 *     03.28.94  ESF  Added MSG_DEADPLAYER & MSG_ENDOFGAME.
 *     03.28.94  ESF  Changed MSG_REQUESTCARDS to ...CARD.
 *     03.29.94  ESF  Changed MSG_UPDATECARDS to MSG_EXCHANGECARDS.
 *     03.29.94  ESF  Changed uiReply to be an Int.
 *     04.11.94  ESF  Added a player parameter to MSG_CARDPACKET.
 *     04.11.94  ESF  Added a killer paramter to MSG_DEADPLAYER.
 *     05.03.94  ESF  Added MSG_OBJ*UPDATE msgs.
 *     05.03.94  ESF  Removed MSG_REGISTERPLAYER and MSG_UPDATEARMIES.
 *     05.05.94  ESF  Added MSG_OBJ* msgs.
 *     05.12.94  ESF  Removed MSG_OBJ* msgs and added GAME messages.
 *     05.12.94  ESF  Added MSG_DEREGISTERCLIENT.
 *     05.12.94  ESF  Added MSG_DELETEMSGDST.
 *     05.13.94  ESF  Added MSG_STARTREGISTRATION.
 *     05.15.94  ESF  Added MSG_[FREE|ALLOC]PLAYER
 *     05.15.94  ESF  Removed MSG_DEADPLAYER.
 *     05.17.94  ESF  Added MSG_NETMESSAGE.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void NET_RecvMessage(Int iSocket, Int *piMessType, void **ppvMessage)
{
  Int i;

  /* Get the message ID */
  _NET_RecvLong(iSocket, (Long *)piMessType);

  switch(*piMessType)
    {
    case MSG_REGISTERCLIENT:  
      {
	MsgRegisterClient *pmsgMess = 
	  (MsgRegisterClient *)MEM_Alloc(sizeof(MsgRegisterClient));
	
	_NET_RecvString(iSocket, &pmsgMess->strClientAddress);

	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_EXCHANGECARDS:
      {
	MsgExchangeCards *pmsgMess = 
	  (MsgExchangeCards *)MEM_Alloc(sizeof(MsgExchangeCards));
	
	for(i=0; i!=3; i++)
	  _NET_RecvLong(iSocket, (Long *)&pmsgMess->piCards[i]);

	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_CARDPACKET:
      {
	MsgCardPacket *pmsgMess = 
	  (MsgCardPacket *)MEM_Alloc(sizeof(MsgCardPacket));

	_NET_RecvLong(iSocket, (Long *)&pmsgMess->iPlayer);
	_NET_RecvLong(iSocket, (Long *)&pmsgMess->cdCard);

	*ppvMessage = (void *)pmsgMess;
      }
      break;
      
    case MSG_REPLYPACKET:
      {
	MsgReplyPacket *pmsgMess = 
	  (MsgReplyPacket *)MEM_Alloc(sizeof(MsgReplyPacket));

	_NET_RecvLong(iSocket, (Long *)&pmsgMess->iReply);

	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_SENDMESSAGE:
      {
	MsgSendMessage *pmsgMess = 
	  (MsgSendMessage *)MEM_Alloc(sizeof(MsgSendMessage));

	_NET_RecvString(iSocket, &pmsgMess->strMessage);
	_NET_RecvString(iSocket, &pmsgMess->strDestination);

	*ppvMessage = (void *)pmsgMess;
      } 
      break;

    case MSG_MESSAGEPACKET:
      {
	MsgMessagePacket *pmsgMess = 
	  (MsgMessagePacket *)MEM_Alloc(sizeof(MsgMessagePacket));

	_NET_RecvString(iSocket, &pmsgMess->strMessage);
	_NET_RecvString(iSocket, &pmsgMess->strFrom);
	_NET_RecvLong(iSocket, &pmsgMess->iTo);

	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_TURNNOTIFY:
      {
	MsgTurnNotify *pmsgMess = 
	  (MsgTurnNotify *)MEM_Alloc(sizeof(MsgTurnNotify));
	
	_NET_RecvLong(iSocket, &pmsgMess->iPlayer);
	_NET_RecvLong(iSocket, &pmsgMess->iClient);
	
	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_CLIENTIDENT:
      {
	MsgClientIdent *pmsgMess = 
	  (MsgClientIdent *)MEM_Alloc(sizeof(MsgClientIdent));

	_NET_RecvLong(iSocket, &pmsgMess->iClientID);
	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_REQUESTCARD:
      {
	MsgRequestCard *pmsgMess = 
	  (MsgRequestCard *)MEM_Alloc(sizeof(MsgRequestCard));

	_NET_RecvLong(iSocket, &pmsgMess->iPlayer);
	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_ENTERSTATE:
      {
	MsgEnterState *pmsgMess = 
	  (MsgEnterState *)MEM_Alloc(sizeof(MsgEnterState));

	_NET_RecvLong(iSocket, &pmsgMess->iState);
	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_OBJSTRUPDATE:
      {
	MsgObjStrUpdate *pmsgMess = 
	  (MsgObjStrUpdate *)MEM_Alloc(sizeof(MsgObjStrUpdate));

	_NET_RecvLong(iSocket, &pmsgMess->iField);
	_NET_RecvLong(iSocket, &pmsgMess->iIndex1);
	_NET_RecvLong(iSocket, &pmsgMess->iIndex2);
	_NET_RecvString(iSocket, &pmsgMess->strNewValue);
	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_OBJINTUPDATE:
      {
	MsgObjIntUpdate *pmsgMess = 
	  (MsgObjIntUpdate *)MEM_Alloc(sizeof(MsgObjIntUpdate));

	_NET_RecvLong(iSocket, &pmsgMess->iField);
	_NET_RecvLong(iSocket, &pmsgMess->iIndex1);
	_NET_RecvLong(iSocket, &pmsgMess->iIndex2);
	_NET_RecvLong(iSocket, &pmsgMess->iNewValue);
	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_DELETEMSGDST:
      {
	MsgDeleteMsgDst *pmsgMess = 
	  (MsgDeleteMsgDst *)MEM_Alloc(sizeof(MsgDeleteMsgDst));
	
	_NET_RecvLong(iSocket, &pmsgMess->iClient);
	*ppvMessage = (void *)pmsgMess;
      }
      break;

    case MSG_FREEPLAYER:
      {
	MsgFreePlayer *pmsgMess = 
	  (MsgFreePlayer *)MEM_Alloc(sizeof(MsgFreePlayer));
	
	_NET_RecvLong(iSocket, &pmsgMess->iPlayer);
	*ppvMessage = (void *)pmsgMess;
      }
      break;      

    case MSG_NETMESSAGE:
      {
	MsgNetMessage *pmsgMess = 
	  (MsgNetMessage *)MEM_Alloc(sizeof(MsgNetMessage));
	
	_NET_RecvString(iSocket, &pmsgMess->strMessage);
	*ppvMessage = (void *)pmsgMess;
      }
      break;      

    case MSG_EXIT:
    case MSG_STARTGAME:
    case MSG_ENDTURN:
    case MSG_ENDOFGAME:
    case MSG_GAMENEWPLAYERS:
    case MSG_GAMEMOREPLAYERS:
    case MSG_GAMEANOTHER:
    case MSG_DEREGISTERCLIENT:
    case MSG_STARTREGISTRATION:
    case MSG_ALLOCPLAYER:
      *ppvMessage = NULL;
      break;

    default:
      printf("Illegal message!\n");
      
    }
}

/************************************************************************ 
 *  FUNCTION: _NET_SendString
 *  HISTORY: 
 *     01.24.94  ESF  Created 
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void _NET_SendString(Int iSocket, String strString)
{
  /* Send the length and then the string itself */
  _NET_SendLong(iSocket, (Long)(strlen(strString)+1));
  write(iSocket, (Char *)strString, strlen(strString)+1); 
}


/************************************************************************ 
 *  FUNCTION: _NET_SendLong
 *  HISTORY: 
 *     01.24.94  ESF  Created 
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void _NET_SendLong(Int iSocket, Long lLong)
{
  Long lData = htonl(lLong);
  
  write(iSocket, &lData, sizeof(Long));
}


/************************************************************************ 
 *  FUNCTION: _NET_RecvString 
 *  HISTORY: 
 *     01.24.94  ESF  Created 
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void _NET_RecvString(Int iSocket, String *pstrString)
{
  Long   lLength;
  String strTemp;

  /* Receive the length and then the byte stream */
  _NET_RecvLong(iSocket, &lLength);
  strTemp = (String)MEM_Alloc(lLength);
  read(iSocket, strTemp, lLength);

  *pstrString = strTemp;
}


/************************************************************************ 
 *  FUNCTION: _NET_RecvLong
 *  HISTORY: 
 *     01.24.94  ESF  Created 
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void _NET_RecvLong(Int iSocket, Long *plLong)
{
  read(iSocket, plLong, sizeof(Long));
  *plLong = ntohl(*plLong);
}


/************************************************************************ 
 *  FUNCTION: NET_DeleteMessage
 *  HISTORY: 
 *     06.24.94  ESF  Created 
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void NET_DeleteMessage(Int iMessType, void *pvMessage)
{
  switch(iMessType)
    {
    case MSG_REGISTERCLIENT:  
      {
	MsgRegisterClient *pmsgMess = (MsgRegisterClient *)pvMessage;
	MEM_Free(pmsgMess->strClientAddress);
	MEM_Free(pmsgMess);
      }
      break;

    case MSG_EXCHANGECARDS:
      MEM_Free(pvMessage);
      break;

    case MSG_CARDPACKET:
      MEM_Free(pvMessage);
      break;
      
    case MSG_REPLYPACKET:
      MEM_Free(pvMessage);
      break;

    case MSG_SENDMESSAGE:
      {
	MsgSendMessage *pmsgMess = (MsgSendMessage *)pvMessage;

	MEM_Free(pmsgMess->strMessage);
	MEM_Free(pmsgMess->strDestination);
	MEM_Free(pvMessage);
      } 
      break;

    case MSG_MESSAGEPACKET:
      {
	MsgMessagePacket *pmsgMess = (MsgMessagePacket *)pvMessage;

	MEM_Free(pmsgMess->strMessage);
	MEM_Free(pmsgMess->strFrom);
	MEM_Free(pvMessage);
      }
      break;

    case MSG_TURNNOTIFY:
      MEM_Free(pvMessage);
      break;
      
    case MSG_CLIENTIDENT:
      MEM_Free(pvMessage);
      break;

    case MSG_REQUESTCARD:
      MEM_Free(pvMessage);
      break;

    case MSG_ENTERSTATE:
      MEM_Free(pvMessage);
      break;

    case MSG_OBJSTRUPDATE:
      {
	MsgObjStrUpdate *pmsgMess = (MsgObjStrUpdate*)pvMessage;
	
	MEM_Free(pmsgMess->strNewValue);
	MEM_Free(pvMessage);
      }
      break;
      
    case MSG_OBJINTUPDATE:
      MEM_Free(pvMessage);
      break;

    case MSG_DELETEMSGDST:
      MEM_Free(pvMessage);
      break;

    case MSG_FREEPLAYER:
      MEM_Free(pvMessage);
      break;

    case MSG_NETMESSAGE:
      {
	MsgNetMessage *pmsgMess = (MsgNetMessage *)pvMessage;
	MEM_Free(pmsgMess->strMessage);
	MEM_Free(pvMessage);
      }
      break;

    case MSG_EXIT:
    case MSG_STARTGAME:
    case MSG_ENDTURN:
    case MSG_ENDOFGAME:
    case MSG_GAMENEWPLAYERS:
    case MSG_GAMEMOREPLAYERS:
    case MSG_GAMEANOTHER:
    case MSG_DEREGISTERCLIENT:
    case MSG_STARTREGISTRATION:
    case MSG_ALLOCPLAYER:
      /* No parameters */
      break;

    default:
      printf("Illegal message!\n");
      
    }
}
