/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 1993, 1994, Elan Feingold (feingold@zko.dec.com)           *
 *                                                                           *
 *     PERMISSION TO USE, COPY, MODIFY, AND TO DISTRIBUTE THIS SOFTWARE      *
 *     AND ITS DOCUMENTATION FOR ANY PURPOSE IS HEREBY GRANTED WITHOUT       *
 *     FEE, PROVIDED THAT THE ABOVE COPYRIGHT NOTICE APPEAR IN ALL           *
 *     COPIES AND MODIFIED COPIES AND THAT BOTH THAT COPYRIGHT NOTICE AND    *
 *     THIS PERMISSION NOTICE APPEAR IN SUPPORTING DOCUMENTATION.  THERE     *
 *     IS NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR      *
 *     ANY PURPOSE.  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS       *
 *     OR IMPLIED WARRANTY.                                                  *
 *                                                                           *
 *****************************************************************************/

#include <X11/X.h>
#include <X11/Intrinsic.h>
#include <X11/cursorfont.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/List.h>
#include <X11/Xaw/Toggle.h>

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>

#include "gui.h"
#include "utils.h"
#include "callbacks.h"
#include "riskgame.h"
#include "client.h"
#include "network.h"
#include "dice.h"
#include "cards.h"
#include "game.h"
#include "colormap.h"
#include "help.h"
#include "debug.h"

/* Game globals */
Char     strScratch[256];
Cursor   cursorWait=0, cursorPlay=0;
Int      iState, iCurrentPlayer;
Boolean  fAttacked, fGameStarted=FALSE, fPlayingRemotely=FALSE;
Boolean  fGetsCard=FALSE, fCanExchange=TRUE, fForceExchange=FALSE;
Boolean  fRegistrationEnded=FALSE;
Int      iAttackSrc=-1, iAttackDst=-1;
Int      iMoveSrc=-1, iMoveDst=-1;
Int      iActionState=ACTION_PLACE;
Int      iLastAttackSrc=-1, iLastAttackDst=-1;
Int      iReply;

String   pstrMsgDstString[MAX_PLAYERS+1];
Int      piMsgDstPlayerID[MAX_PLAYERS+1];
Int      iIndexMD=1;


/************************************************************************ 
 *  FUNCTION: CBK_XIncomingMessage
 *  HISTORY: 
 *     03.17.94  ESF  Created.
 *     06.24.94  ESF  Fixed memory leak bug.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_XIncomingMessage(XtPointer pClientData, int *iSource, XtInputId *id)
{
  Int     iMessType;
  void   *pvMess;
  
  NET_RecvMessage(*iSource, &iMessType, &pvMess);
  CBK_IncomingMessage(iMessType, pvMess);
  NET_DeleteMessage(iMessType, pvMess);
}


/************************************************************************ 
 *  FUNCTION: CBK_IncomingMessage
 *  HISTORY: 
 *     01.26.94  ESF  Created.
 *     01.27.94  ESF  Coded MSG_MESSAGEPACKET case.
 *     01.29.94  ESF  Changed MSG_MESSAGEPACKET to scroll correctly.
 *     02.05.94  ESF  Added MSG_REGISTERPLAYER handling.
 *     03.02.94  ESF  Added more army placing glue.
 *     03.04.94  ESF  Factored out code, cleaned up.
 *     03.05.94  ESF  Added color-coded player turn indicator call.
 *     03.06.94  ESF  Fixed initialization of iCurrentPlayer bug.
 *     03.16.94  ESF  Added continent bonuses.
 *     03.17.94  ESF  Factored out X code so that this could be more general.
 *     03.17.94  ESF  Fixed bug, player indicator broken for remote case.
 *     03.17.94  ESF  Added fPlayingRemotely setting.
 *     03.18.94  ESF  Completed MSG_UPDATEARMIES code.
 *     03.28.94  ESF  Added code for MSG_ENDOFGAME and MSG_DEADPLAYER.
 *     03.29.94  ESF  Added code for MSG_CARDPACKET.
 *     03.29.94  ESF  Fixed bug in handling on MSG_UPDATEARMIES.
 *     03.29.94  ESF  Added handling for MSG_REPLYPACKET.
 *     04.11.94  ESF  Added handling for MSG_CARDPACKET.
 *     05.12.94  ESF  Added handling for MSG_DELETEMSGDST.
 *     05.17.94  ESF  Added MSG_NETMESSAGE.
 *     05.19.94  ESF  Added verbose message when exit occurs.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_IncomingMessage(Int iMessType, void *pvMess)
{
  switch(iMessType)
    {
    /******************/
    case MSG_NETMESSAGE:
    /******************/
      UTIL_DisplayComment(((MsgNetMessage *)(pvMess))->strMessage);
      break;

    /********************/
    case MSG_OBJINTUPDATE:
    case MSG_OBJSTRUPDATE:
    /********************/
      /* Process the message, indicating that it's from the server */
      RISK_ProcessMessage(iMessType, pvMess, -1);
      break;

    /********************/
    case MSG_DELETEMSGDST:
    /********************/
      UTIL_DeleteMsgDst(((MsgDeleteMsgDst *)pvMess)->iClient);
      break;
      
    /*******************/
    case MSG_STARTREGISTRATION:
    /*******************/
      UTIL_DisplayComment("");
      break;

    /*******************/
    case MSG_REPLYPACKET:
    /*******************/
      iReply = ((MsgReplyPacket *)pvMess)->iReply;
      break;
      
    /*******************/
    case MSG_ENDOFGAME:
    /*******************/
      GAME_GameOverMan();
      break;

    /*******************/
    case MSG_CLIENTIDENT:
    /*******************/
      {
	MsgClientIdent *pMess = (MsgClientIdent *)pvMess;
	iThisClient = pMess->iClientID;
	break;
      }
    
    /*********************/
    case MSG_MESSAGEPACKET:
    /*********************/
      {
	MsgMessagePacket *pMess = (MsgMessagePacket *)pvMess;
	UTIL_DisplayMessage(pMess->strFrom, pMess->strMessage);
      }
      break;
      
    /******************/
    case MSG_TURNNOTIFY:
    /******************/
      { 
	MsgTurnNotify  *pTurn = (MsgTurnNotify *)pvMess;
	Int             i;

	/* If the client is being notified of a turn, 
	 * the registration must be over.
	 */
	fRegistrationEnded = TRUE;

	iCurrentPlayer = pTurn->iPlayer;

	if (pTurn->iClient == iThisClient)
	  {
	    fPlayingRemotely = FALSE;

	    /* Set the cursor to indicate play */
	    if (cursorPlay==0)
	      cursorPlay = XCreateFontCursor(hDisplay, XC_crosshair);
	    XDefineCursor(hDisplay, hWindow, cursorPlay);
	    
	    /* See if everyone has finished fortifying their territories */
	    if (!RISK_GetNumArmiesOfPlayer(iCurrentPlayer) &&
		fGameStarted == FALSE)
	      for (i=0, fGameStarted=TRUE; i!=MAX_PLAYERS; i++)
		if (RISK_GetNumArmiesOfPlayer(i) && 
		    RISK_GetClientOfPlayer(i)!=-1)
		  fGameStarted=FALSE;
	    
	    if (fGameStarted)
	      {
		/* Force the player to exchange if he or she possesses 
		 * 5 cards.
		 */
		
		if (RISK_GetNumCardsOfPlayer(iCurrentPlayer) >= 5)
		  {
		    UTIL_DisplayComment("You have more than five cards"
					" and must exchange a set.");
		    fForceExchange = TRUE;
		    CBK_ShowCards((Widget)NULL, 
				  (XtPointer)NULL, (XtPointer)NULL);
		  }
		
		/* Go into placing state, give the player as many armies as
		 * he or she deserves by dividing the total number of 
		 * countries owned divided by 3, for a minimum of three
		 * armies.
		 */

		iState = STATE_PLACE;
		UTIL_ServerEnterState(iState);

		/* ASSERT: Old number of armies must equal 0 */
		RISK_SetNumArmiesOfPlayer(iCurrentPlayer,
		   GAME_GetContinentBonus(iCurrentPlayer) +
	           MAX(RISK_GetNumCountriesOfPlayer(iCurrentPlayer)/3, 3));
	      }
	    else
	      {
		iState = STATE_FORTIFY;
		UTIL_ServerEnterState(iState);
	      }
	    UTIL_DisplayActionString(iState, iCurrentPlayer);
	  }
	else
	  {
	    fPlayingRemotely = TRUE;
	    
	    /* Set the cursor to indicate waiting */
	    if (cursorWait==0)
	      cursorWait = XCreateFontCursor(hDisplay, XC_watch);
	    XDefineCursor(hDisplay, hWindow, cursorWait);
	    
	    sprintf(strScratch, "%s is playing remotely...",
		    RISK_GetNameOfPlayer(iCurrentPlayer));
	    UTIL_DisplayComment(strScratch);
	  }

	/* Set the color of the player color-coded indicator */
	COLOR_CopyColor(COLOR_PlayerToColor(iCurrentPlayer),
			COLOR_DieToColor(2));
      }
      break;
      
    /************/
    case MSG_EXIT:
    /************/
      /* Popup telling what happened */
      UTIL_PopupDialog("Exit Notification", 
		       "Another client terminated the game!",
		       1, "Ok", NULL, NULL);

      close(iReadSocket);
      close(iWriteSocket);
      UTIL_ExitProgram(0);
      break;
    
    default:
      sprintf(strScratch, "CALLBACKS: Unhandled message (%d)\n", iMessType);
      UTIL_PopupDialog("Fatal Error", strScratch, 1, "Ok", NULL, NULL);
      NET_SendMessage(iWriteSocket, MSG_EXIT, NULL);
      UTIL_ExitProgram(-1);
    }
}


/************************************************************************ 
 *  FUNCTION: CBK_RefreshMap
 *  HISTORY: 
 *     01.27.94  ESF  Created.
 *     01.28.94  ESF  Changed to work with pixmap.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_RefreshMap(Widget w, XtPointer pData, XtPointer pCalldata)
{
  XExposeEvent *pExpose = (XExposeEvent *)pCalldata;

  XCopyArea(hDisplay, pixMapImage, hWindow, hGC, 
	    pExpose->x, pExpose->y, pExpose->width, pExpose->height, 
	    pExpose->x, pExpose->y);
}


/************************************************************************ 
 *  FUNCTION: CBK_MapClick
 *  HISTORY: 
 *     01.27.94  ESF  Created 
 *     02.03.94  ESF  Don't allocate colors in order, use mapping.
 *     03.03.94  ESF  Added some game glue.
 *     03.04.94  ESF  Fixed the calculation of fGameStarted.
 *     03.07.94  ESF  Made a lean, mean, fighting function.  Look in game.c.
 *     03.17.94  ESF  Fixed remote case.
 *     05.10.94  ESF  Fixed to do nothing unless game has started.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_MapClick(Widget w, XtPointer pClientData, XEvent *pEvent)
{
  Int  x, y, iCountry;

  if (!fRegistrationEnded)
    return;

  if (fPlayingRemotely)
    {
      sprintf(strScratch, "Click as you may, it's %s's turn...", 
	      RISK_GetNameOfPlayer(iCurrentPlayer));
      UTIL_DisplayError(strScratch);
      return;
    }

  x = pEvent->xbutton.x;
  y = pEvent->xbutton.y;
  iCountry = COLOR_ColorToCountry(XGetPixel(pMapImage, x, y));
  
  /* Erase previous errors */
  UTIL_DisplayError("");
  
  switch (iState)
    {
    case STATE_FORTIFY:
      GAME_PlaceClick(iCountry, PLACE_ONE);
      break;

    case STATE_PLACE:
      if (pEvent->xbutton.button == Button1)
 	GAME_PlaceClick(iCountry, PLACE_ONE);
      else if (pEvent->xbutton.button == Button3)
 	GAME_PlaceClick(iCountry, PLACE_MULTIPLE);
      break;

    case STATE_ATTACK:
      GAME_AttackClick(iCountry);
      break;
      
    case STATE_MOVE:
      GAME_MoveClick(iCountry);
      break;
    }
}


/************************************************************************ 
 *  FUNCTION: CBK_Register
 *  HISTORY: 
 *     01.29.94  ESF  Created.
 *     01.31.94  ESF  Delete the strings after registering.
 *     02.05.94  ESF  Remove local registration from here.
 *     02.05.94  ESF  Adding color validity checking.
 *     03.07.94  ESF  Switched to varargs Xt calls.
 *     03.08.94  ESF  Fixed lack of NULL termination in XtVa calls.
 *     05.04.94  ESF  Fixed DistObj changes, added SetNumLivePlayers() 
 *     05.07.94  ESF  Fixed to not let too many players register.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_Register(Widget w, XtPointer pData, XtPointer call_data)
{
  String  strPlayerName, strPlayerColor;
  XColor  colDummy;

  /* Get the two strings */
  XtVaGetValues(wNameText, XtNstring, &strPlayerName, NULL);
  XtVaGetValues(wColorText, XtNstring, &strPlayerColor, NULL);
  
  /* Don't bother doing anything if something's not filled out */
  if (!strlen(strPlayerName) || !strlen(strPlayerColor))
    return;

  /* Check that the color is valid */
  if (!XParseColor(hDisplay, cmapColormap, strPlayerColor, &colDummy))
    UTIL_DisplayError("That color was not valid, try another.");
  else
    {
      MsgMessagePacket  mess;
      Int               iNewPlayer;

      /* See if there are too many players */
      if ((iNewPlayer=RISK_AllocPlayer(CBK_IncomingMessage)) == -1)
	{
	  UTIL_PopupDialog("Error", "Maximum number of players exceeded!", 1,
			   "Ok", NULL, NULL);
	  return;
	}

      /* Initialize the player */
      RISK_SetColorIndexOfPlayer(iNewPlayer, 
				 COLOR_CountryToColor(NUM_COUNTRIES)+2+
				 iNewPlayer);
      RISK_SetColorStringOfPlayer(iNewPlayer, strPlayerColor);
      RISK_SetNameOfPlayer(iNewPlayer, strPlayerName);
      RISK_SetClientOfPlayer(iNewPlayer, iThisClient);

      /* Send a message to the server for broadcast */
      sprintf(strScratch, "%s has joined, with color %s.", strPlayerName, 
	      strPlayerColor);
      mess.strMessage = strScratch;
      mess.strFrom = strClientName;
      mess.iTo = DST_ALLBUTME;
      NET_SendMessage(iWriteSocket, MSG_MESSAGEPACKET, (void *)&mess);

      /* Erase the strings in the registration widget */
      strPlayerName = (String)MEM_Alloc(16); 
      strcpy(strPlayerName, "");
      XtVaSetValues(wNameText, XtNstring, strPlayerName, NULL);

      /* Erase any messages that may have been there */
      UTIL_DisplayError("");
    }

  /* Either way erase the color string */
  strPlayerColor = (String)MEM_Alloc(16);
  strcpy(strPlayerColor, "");
  XtVaSetValues(wColorText, XtNstring, strPlayerColor, NULL);
}


/************************************************************************ 
 *  FUNCTION: CBK_StartGame
 *  HISTORY: 
 *     01.29.94  ESF  Created.
 *     05.19.94  ESF  Added warning popup.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_StartGame(Widget w, XtPointer pData, XtPointer call_data)
{
  String strColor, strName;

  /* Make sure the player and color dialogs are empty, popup
   * warning if not.
   */
  
  XtVaGetValues(wNameText, XtNstring, &strName, NULL);
  XtVaGetValues(wColorText, XtNstring, &strColor, NULL);
  if (strlen(strName) || strlen(strColor))
    if (UTIL_PopupDialog("Warning", "Discard current registration data?", 
			 2, "Yes", "No", NULL) == QUERY_NO)
      return;
  
  /* Let the server know that this client is ready to play */
  NET_SendMessage(iWriteSocket, MSG_STARTGAME, NULL);
  XtPopdown(wRegisterShell);
}


/************************************************************************ 
 *  FUNCTION: CBK_Quit
 *  HISTORY: 
 *     01.29.94  ESF  Created 
 *     04.02.94  ESF  Added confirmation.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_Quit(Widget w, XtPointer pData, XtPointer call_data)
{
  if (UTIL_PopupDialog("Quit", "Really quit Frisk?", 2, "Yes", "No", NULL)
      == QUERY_YES)
    {
      NET_SendMessage(iWriteSocket, MSG_EXIT, NULL);
      close(iWriteSocket);
      close(iReadSocket);
      UTIL_ExitProgram(0);
    }
  else
    return;
}


/************************************************************************ 
 *  FUNCTION: CBK_ShowCards
 *  HISTORY: 
 *     02.19.94  ESF  Created.
 *     03.17.94  ESF  Erase the error display.
 *     05.04.94  ESF  Fixed so that only the owner can see his or her cards.
 *     05.10.94  ESF  Fixed to do nothing unless game has started.
 *     05.12.94  ESF  Fixed to display correct cards.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_ShowCards(Widget w, XtPointer pData, XtPointer call_data)
{
  Int i, iPlayer;

  if (!fRegistrationEnded)
    return;

  UTIL_DisplayError("");

  /* If the player clicking in the current player, show cards, otherwise,
   * if there's more than one player at the client, issue an error, and
   * if there's only one, show his or her cards.
   */

  if (UTIL_PlayerIsLocal(iCurrentPlayer))
    iPlayer = iCurrentPlayer;
  else if (UTIL_NumPlayersAtClient(iThisClient) == 1)
    iPlayer = UTIL_GetNthPlayerAtClient(iThisClient, 0);
  else
    {
      iPlayer = -1;
      (void)UTIL_PopupDialog("Error", 
			     "Wait until your turn!", 
			     1, "Ok", "Cancel", NULL);
    }

  /* If it's valid, display cards */
  if (iPlayer >= 0)
    {
      XtPopup(wCardShell, XtGrabExclusive);
      
      for (i=0; i!=RISK_GetNumCardsOfPlayer(iPlayer); i++)
	CARD_RenderCard(RISK_GetCardOfPlayer(iPlayer, i), i);
    }
}


/************************************************************************ 
 *  FUNCTION: CBK_CancelCards
 *  HISTORY: 
 *     02.19.94  ESF  Created.
 *     03.17.94  ESF  Added clearing of selections and unmappings.
 *     04.01.94  ESF  Fixed clearing of error when player cancels.
 *     04.11.94  ESF  Added not popping down the shell when !fForceExchange.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_CancelCards(Widget w, XtPointer pData, XtPointer call_data)
{
  Int i;

  if (fForceExchange == TRUE)
    UTIL_PopupDialog("Error", "You must exchange cards!", 1, "Ok", NULL, NULL);
  else
    {
      UTIL_DisplayError("");
      XtPopdown(wCardShell);
      
      /* Unmap the cards and clear the selections */
      for (i=0; i!=MAX_CARDS; i++)
	{
	  XtUnmanageChild(wCardToggle[i]);
	  XtVaSetValues(wCardToggle[i], XtNstate, False, NULL);
	}
    }
}


/************************************************************************ 
 *  FUNCTION: CBK_Attack
 *  HISTORY: 
 *     03.03.94  ESF  Stubbed.
 *     03.05.94  ESF  Coded.
 *     05.07.94  ESF  Modified to work with Distributed RiskGame Object.
 *     05.10.94  ESF  Fixed to do nothing unless game has started.
 *     05.19.94  ESF  Fixed bug, not refering to DiceMode.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_Attack(Widget w, XtPointer pData, XtPointer call_data)
{
  XawListReturnStruct *pItem;

  if (!fRegistrationEnded)
    return;

  /* Don't do anything if current player is not local */
  if (!UTIL_PlayerIsLocal(iCurrentPlayer))
    return;
    
  /* I know this is silly, but hey... (We should get the data from pData?) */
  pItem = XawListShowCurrent(wAttackList);
  RISK_SetDiceModeOfPlayer(iCurrentPlayer, pItem->list_index);
}


/************************************************************************ 
 *  FUNCTION: CBK_Action
 *  HISTORY: 
 *     03.03.94  ESF  Stubbed.
 *     03.05.94  ESF  Coded.
 *     03.29.94  ESF  Added player attack mode history.
 *     05.07.94  ESF  Modified to work with Distributed RiskGame Object.
 *     05.10.94  ESF  Fixed to do nothing unless game has started.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_Action(Widget w, XtPointer pData, XtPointer call_data)
{
  XawListReturnStruct *pItem;

  if (!fRegistrationEnded)
    return;

  /* Don't do anything if current player is not local */
  if (!UTIL_PlayerIsLocal(iCurrentPlayer))
    return;

  /* I know this is silly, but hey... (We should get the data from pData?) */
  pItem = XawListShowCurrent(wActionList);

  switch (pItem->list_index)
    {
    /*****************/
    case ACTION_PLACE:
    /*****************/
      if (iState == STATE_PLACE || iState == STATE_FORTIFY)
	{
	  UTIL_DisplayError("");
	  iActionState = pItem->list_index;
	}
      else /* Invalid */
	{
	  UTIL_DisplayError("You've already placed your armies.");
	  XawListHighlight(wActionList, ACTION_ATTACK);
	}

      break;

    /*******************/
    case ACTION_ATTACK:
    case ACTION_DOORDIE:
    /*******************/
      if (iState == STATE_ATTACK)
	{
	  UTIL_DisplayError("");
	  iActionState = pItem->list_index;

	  /* Keep track of the type of attack selected */
	  RISK_SetAttackModeOfPlayer(iCurrentPlayer, pItem->list_index);
	}	
      else if (iState == STATE_PLACE || iState == STATE_FORTIFY)
	{
	  UTIL_DisplayError("You must finish placing your armies.");
	  XawListHighlight(wActionList, ACTION_PLACE);
	}
      else if (iState == STATE_MOVE) /* Must not have moved yet */
	{
	  iState = STATE_ATTACK;
	  UTIL_ServerEnterState(iState);
	  UTIL_DisplayActionString(iState, iCurrentPlayer);
	}

      break;
      
    /****************/
    case ACTION_MOVE:
    /****************/
      if (iState == STATE_MOVE)
	{
	  UTIL_DisplayError("");
	  iActionState = pItem->list_index;
	}
      else if (iState == STATE_FORTIFY || iState == STATE_PLACE)
	{
	  UTIL_DisplayError("You must finish placing your armies.");
	  XawListHighlight(wActionList, ACTION_PLACE);
	}
      else /* STATE_ATTACK */
	{
	  iState = STATE_MOVE;
	  UTIL_ServerEnterState(iState);
	  UTIL_DisplayActionString(iState, iCurrentPlayer);
	}

      break;
    }
}


/************************************************************************ 
 *  FUNCTION: CBK_MsgDest
 *  HISTORY: 
 *     03.03.94  ESF  Stubbed.
 *     05.07.94  ESF  Created and modified to work with Distributed Object.
 *     05.19.94  ESF  Fixed a bug, wasn't refering to MsgDestList.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_MsgDest(Widget w, XtPointer pData, XtPointer call_data)
{
  XawListReturnStruct *pItem;

  if (!UTIL_PlayerIsLocal(iCurrentPlayer))
    return;
  
  /* I know this is silly, but hey... (We should get the data from pData?) */
  pItem = XawListShowCurrent(wMsgDestList);
  RISK_SetMsgDstModeOfPlayer(iCurrentPlayer, pItem->list_index);
}


/************************************************************************ 
 *  FUNCTION: CBK_CancelAttack
 *  HISTORY: 
 *     03.03.94  ESF  Stubbed.
 *     03.04.94  ESF  Coded.
 *     05.10.94  ESF  Fixed to do nothing unless game has started.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_CancelAttack(Widget w, XtPointer pData, XtPointer call_data)
{
  if (!fRegistrationEnded)
    return;

  if (fPlayingRemotely)
    {
      UTIL_DisplayError("Nice try, but it isn't your turn...");
      return;
    }

  UTIL_DisplayError("");

  if (iState == STATE_ATTACK)
    {
      if (iAttackSrc >=0)
	UTIL_DarkenCountry(iAttackSrc);
      iAttackSrc = -1;
    }
  else if (iState == STATE_MOVE)
    {
      if (iMoveSrc >=0)
	UTIL_DarkenCountry(iMoveSrc);
      iMoveSrc = -1;
    }

  UTIL_DisplayActionString(iState, iCurrentPlayer);
}


/************************************************************************ 
 *  FUNCTION: CBK_Repeat
 *  HISTORY: 
 *     03.03.94  ESF  Stubbed.
 *     03.06.94  ESF  Coded.
 *     03.07.94  ESF  Fixed state bug.
 *     03.08.94  ESF  Added DOORDIE recognition.
 *     03.16.94  ESF  Fixed bug in DOORDIE.
 *     04.02.94  ESF  Added server notification caching for do-or-die.
 *     05.10.94  ESF  Fixed to do nothing unless game has started.
 *     05.19.94  ESF  Added verbose message across network.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_Repeat(Widget w, XtPointer pData, XtPointer call_data)
{
  MsgNetMessage mess;

  if (!fRegistrationEnded)
    return;

  if (fPlayingRemotely)
    {
      sprintf(strScratch, "I'm afraid that's up to %s to decide...", 
	      RISK_GetNameOfPlayer(iCurrentPlayer));
      UTIL_DisplayError(strScratch);
      return;
    }

  /* Send a message if an attack is going down */
  if (iActionState == ACTION_ATTACK || iActionState == ACTION_DOORDIE)
    {
      /* send a verbose message */
      sprintf(strScratch, "%s is attacking from %s to %s\n",
	      RISK_GetNameOfPlayer(iCurrentPlayer),
	      RISK_GetNameOfCountry(iLastAttackSrc),
	      RISK_GetNameOfCountry(iLastAttackDst));
      mess.strMessage = strScratch;
      NET_SendMessage(iWriteSocket, MSG_NETMESSAGE, &mess);
    }

  /* Check the validity of the things in the buffer */
  if (iLastAttackSrc == -1 || iLastAttackDst == -1)
    UTIL_DisplayError("You can't repeat an uncomplete attack.");
  else if (iState != STATE_ATTACK)
    UTIL_DisplayError("You can't do any attacking at the moment.");
  else if (iActionState == ACTION_ATTACK)
    {
      if (GAME_ValidAttackSrc(iLastAttackSrc, TRUE) &&
	  GAME_ValidAttackDst(iLastAttackSrc, iLastAttackDst, TRUE))
	{
	  GAME_Attack(iLastAttackSrc, iLastAttackDst, FALSE);
	  UTIL_DisplayActionString(iState, iCurrentPlayer);
	}
    }
  else if (iActionState == ACTION_DOORDIE)
    {
      Int iDice = RISK_GetDiceModeOfPlayer(iCurrentPlayer);

      while (GAME_ValidAttackSrc(iLastAttackSrc, FALSE) &&
	     GAME_ValidAttackDst(iLastAttackSrc, iLastAttackDst, FALSE) &&
	     GAME_ValidAttackDice(iDice, iLastAttackSrc))
	GAME_Attack(iLastAttackSrc, iLastAttackDst, TRUE);
    }
}


/************************************************************************ 
 *  FUNCTION: CBK_Help
 *  HISTORY: 
 *     04.01.94  ESF  Coded.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_Help(Widget w, XtPointer pData, XtPointer call_data)
{
  Int x, y;

  /* Center popup */
  UTIL_CenterShell(wHelpShell, wToplevel, &x, &y);
  XtVaSetValues(wHelpShell, 
		XtNallowShellResize, False,
		XtNx, x, 
		XtNy, y, 
		XtNborderWidth, 1,
		XtNtitle, "Frisk Help",
		NULL);

  XtPopup(wHelpShell, XtGrabNone);
}


/************************************************************************ 
 *  FUNCTION: CBK_
 *  HISTORY: 
 *     04.01.94  ESF  Coded.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_HelpSelectTopic(Widget w, XtPointer pData, XtPointer call_data)
{
  XawListReturnStruct *pItem;
  
  /* I know this is sill, but hey... (We should get the data from pData) */
  pItem = XawListShowCurrent(wHelpTopicList);
  HELP_IndexPopupHelp(pItem->list_index);
}


/************************************************************************ 
 *  FUNCTION: CBK_
 *  HISTORY: 
 *     04.01.94  ESF  Coded.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_HelpOk(Widget w, XtPointer pData, XtPointer call_data)
{
  XtPopdown(wHelpShell);
}


/************************************************************************ 
 *  FUNCTION: CBK_ExchangeCards
 *  HISTORY: 
 *     03.03.94  ESF  Stubbed.
 *     03.29.94  ESF  Coded.
 *     05.03.94  ESF  Fixed for when the player has many cards.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_ExchangeCards(Widget w, XtPointer pData, XtPointer call_data)
{
  Int      i, iNumCardsSelected;
  Boolean  iCardState; 
  Int      piCards[3], piCardValues[3], piCardTypes[3];
  
  /* Get the cards that have been pressed, check for validity */
  for (i=iNumCardsSelected=0; i!=MAX_CARDS; i++)
    {
      XtVaGetValues(wCardToggle[i], XtNstate, &iCardState, NULL);
      if (iCardState == True)
	{
	  if (iNumCardsSelected<3)
	    piCards[iNumCardsSelected] = i;
	  iNumCardsSelected++;
	}
    }

  /* Clear the selections of the cards */
  for (i=0; i!=MAX_CARDS; i++)
    XtVaSetValues(wCardToggle[i], XtNstate, False, NULL);

  /* Valid number of cards selected */
  if (fCanExchange == FALSE)
    {
      UTIL_DisplayError("You can only exchange cards at the beginning of "
			"your turn.");
      return;
    }
  else if (iNumCardsSelected!=3)
    {
      UTIL_DisplayError("You must select three cards to exchange.");
      return;
    }
  
  /* Substitute the indices for the card values */
  for (i=0; i!=3; i++)
    {
      piCardValues[i] = RISK_GetCardOfPlayer(iCurrentPlayer, piCards[i]);
      if (piCardValues[i]<NUM_COUNTRIES)
	piCardTypes[i] = piCardValues[i] % 3;
      else
	piCardTypes[i] = -1;  /* Joker */
    }
  
  /* Now see if the cards form a valid triple */
  if ((piCardTypes[0]==piCardTypes[1] && 
       piCardTypes[1]==piCardTypes[2] && 
       piCardTypes[0]==piCardTypes[2]) ||
      (piCardTypes[0]!=piCardTypes[1] && 
       piCardTypes[1]!=piCardTypes[2] && 
       piCardTypes[0]!=piCardTypes[2]) ||
      (piCardTypes[0]==-1 || piCardTypes[1]==-1 || piCardTypes[2]==-1))
    {
      /* Actually perform the exchange */
      GAME_ExchangeCards(piCards);

      /* If this is the only exchange, then pop down the window */
      if (RISK_GetNumCardsOfPlayer(iCurrentPlayer) <= 5)
	{
	  /* We're through exchanging sets of cards */
	  fForceExchange = fCanExchange = FALSE;

	  XtPopdown(wCardShell);
	  
	  /* Unmanage the cards */
	  for (i=0; i!=MAX_CARDS; i++)
	    XtUnmanageChild(wCardToggle[i]);
	}
      else
	{
	  /* Redisplay cards after erasing and unselecting the old ones */
	  for (i=0; i!=MAX_CARDS; i++)
	    {
	      XtUnmanageChild(wCardToggle[i]);
	      XtVaSetValues(wCardToggle[i], XtNstate, False, NULL);
	    }
	  
	  for (i=0; i!=RISK_GetNumCardsOfPlayer(iCurrentPlayer); i++)
	    CARD_RenderCard(RISK_GetCardOfPlayer(iCurrentPlayer, i), i);
	}

      /* Display the appropriate message for the state */
      UTIL_DisplayActionString(iState, iCurrentPlayer);
    }
  else
    UTIL_DisplayError("You must pick three cards of the same type or one of "
		      "each type.");
}


/************************************************************************ 
 *  FUNCTION: CBK_EndTurn
 *  HISTORY: 
 *     03.05.94  ESF  Created.
 *     05.10.94  ESF  Fixed to do nothing unless game has started.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_EndTurn(Widget w, XtPointer pData, XtPointer call_data)
{
  if (!fRegistrationEnded)
    return;

  if (fPlayingRemotely)
    {
      sprintf(strScratch, "Come on, let %s finish playing...", 
	      RISK_GetNameOfPlayer(iCurrentPlayer));
      UTIL_DisplayError(strScratch);
      return;
    }
  
  if (iState == STATE_PLACE || iState == STATE_FORTIFY)
    UTIL_DisplayError("You must finish placing your armies.");
  else
    {
      UTIL_DisplayError("");
      UTIL_ServerEndTurn();
    }
}


/************************************************************************ 
 *  FUNCTION: CBK_RefreshDice
 *  HISTORY: 
 *     04.11.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_RefreshDice(Widget w, XtPointer pData, XtPointer call_data)
{
  DICE_Refresh();
}


/************************************************************************ 
 *  FUNCTION: CBK_ObjectMessage
 *  HISTORY: 
 *     05.03.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void CBK_ObjectMessage(Int iMessType, void *pvMess)
{
  switch (iMessType)
    {
    case MSG_OBJINTUPDATE:
      {
	MsgObjIntUpdate *pMess = (MsgObjIntUpdate *)pvMess;
	switch(pMess->iField)
	  {
	  /*************/
	  case CNT_OWNER:
	  /*************/
	    {
	      /* Recolor the country */
	      if (pMess->iNewValue >= 0)
		COLOR_ColorCountry(pMess->iIndex1, pMess->iNewValue);
	    }
	    break;
	    
	  /*****************/
	  case CNT_NUMARMIES:
	  /*****************/
	    {
	      /* Redraw the number of armies on the country */
	      UTIL_PrintArmies(pMess->iIndex1, pMess->iNewValue,
			       BlackPixel(hDisplay, 0));
	    }
	    break;
	    
	  /*************/
	  case PLR_STATE:
	  /*************/
	    {
	      if (pMess->iNewValue == FALSE)
		{
		  /* A player has died, so handle this */
		  sprintf(strScratch, "%s has been destroyed.", 
			  RISK_GetNameOfPlayer(pMess->iIndex1));
		  UTIL_DisplayMessage("Server", strScratch);
		}
	    }
	    break;
	  }
	break;
      }

    case MSG_OBJSTRUPDATE:
      {
	MsgObjStrUpdate *pMess = (MsgObjStrUpdate *)pvMess;
	switch (pMess->iField)
	  {
	  /************/
	  case PLR_NAME:
	  /************/
	    {
	      if (pMess->strNewValue != NULL)
		{
		  /* A new player, add this to the list, update listbox */
		  pstrMsgDstString[iIndexMD] = 
		    (String)MEM_Alloc(strlen(pMess->strNewValue)+1);
		  piMsgDstPlayerID[iIndexMD] = pMess->iIndex1;
		  strcpy(pstrMsgDstString[iIndexMD], pMess->strNewValue);
		  UTIL_RefreshMsgDest(++iIndexMD);

		  /* Also create a message that tells of the new player */
		  sprintf(strScratch, "%s has joined, with color %s.", 
			  pMess->strNewValue, 
			  RISK_GetColorStringOfPlayer(pMess->iIndex1));
		  UTIL_DisplayMessage("Server", strScratch);
		}
	    }
	    break;

	  /*******************/
	  case PLR_COLORSTRING:
	  /*******************/
	    {
	      /* Allocate the color */
	      COLOR_StoreNamedColor(pMess->strNewValue, pMess->iIndex1);
	    }
	    break;
	  }
      }
    }
}
