/*
 This file is part of GNU Taler
 (C) 2022-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  AbsoluteTime,
  AvailableMeasureSummary,
  KycRule,
  LegitimizationMeasures,
  TalerExchangeApi,
} from "@gnu-taler/taler-util";
import { Time, useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { RulesInfo } from "./RulesInfo.js";
import { CurrentMeasureTable } from "./MeasuresTable.js";
import { computeMeasureInformation } from "../utils/computeAvailableMesaures.js";
import { CheckCircleIcon } from "@heroicons/react/20/solid";

const TALER_SCREEN_ID = 120;

// function ShowLegistimizationInfo({info}:{info: LegitimizationMeasureDetails}): VNode {
//   return <CurrentMeasureTable measures={res[0]} hideMeasureNames />;
// }

export function ShowLegistimizationInfo({
  since,
  startOpen,
  fixed,
  legitimization,
  completed,
  serverMeasures,
}: {
  since: AbsoluteTime;
  startOpen?: boolean;
  fixed?: boolean;
  legitimization: LegitimizationMeasures;
  completed: boolean;
  serverMeasures: AvailableMeasureSummary | undefined;
}): VNode {
  const { i18n } = useTranslationContext();
  const [opened, setOpened] = useState(startOpen ?? false);

  function Header() {
    return (
      <div
        data-fixed={!!fixed}
        class="p-4 relative bg-gray-200 flex justify-between data-[fixed=false]:cursor-pointer"
        onClick={() => {
          if (!fixed) {
            setOpened((o) => !o);
          }
        }}
      >
        <div class="flex min-w-0 gap-x-4">
          <div class="flex rounded-md  shadow-sm border-0 ring-1 ring-inset ring-gray-300 focus:ring-2 focus:ring-inset focus:ring-indigo-600">
            <div class="pointer-events-none bg-gray-300 inset-y-0 flex items-center px-3">
              <i18n.Translate>Since</i18n.Translate>
            </div>
            <div class="p-2  bg-gray-50 text-right rounded-md rounded-l-none data-[left=true]:text-left w-full py-1.5 pl-3 text-gray-900  placeholder:text-gray-50  sm:text-sm sm:leading-6">
              <Time format="dd/MM/yyyy HH:mm:ss" timestamp={since} />
            </div>
          </div>
        </div>
        <div class="flex shrink-0 items-center gap-x-4">
          {completed ? (
            <CheckCircleIcon title={i18n.str`Already completed.`} />
          ) : (
            <ClockIcon title={i18n.str`Waiting to be completed.`} />
          )}
          {fixed ? (
            <Fragment />
          ) : (
            <div class="rounded-full bg-gray-50 p-2">
              <svg
                xmlns="http://www.w3.org/2000/svg"
                fill="none"
                viewBox="0 0 24 24"
                stroke-width="1.5"
                stroke="currentColor"
                class="size-6 w-6 h-6"
              >
                {opened ? (
                  <path
                    stroke-linecap="round"
                    stroke-linejoin="round"
                    d="m19.5 8.25-7.5 7.5-7.5-7.5"
                  />
                ) : (
                  <path
                    stroke-linecap="round"
                    stroke-linejoin="round"
                    d="m4.5 15.75 7.5-7.5 7.5 7.5"
                  />
                )}
              </svg>
            </div>
          )}
        </div>
      </div>
    );
  }

  if (!opened) {
    return (
      <div class="overflow-hidden border border-gray-800  rounded-xl">
        <Header />
      </div>
    );
  }

  const info = computeMeasureInformation(serverMeasures, {
    measureList: legitimization.measures,
  });

  return (
    <div class="overflow-hidden border border-gray-800 rounded-xl">
      <Header />
      <div class="p-4 grid gap-y-4">
        <CurrentMeasureTable measures={info} hideMeasureNames />
      </div>
    </div>
  );
}

function CircleCheckIcon({ title }: { title: string }): VNode {
  return (
    <svg
      xmlns="http://www.w3.org/2000/svg"
      fill="none"
      viewBox="0 0 24 24"
      stroke-width="1.5"
      stroke="currentColor"
      class="size-6"
      title={title}
    >
      <path
        stroke-linecap="round"
        stroke-linejoin="round"
        d="M9 12.75 11.25 15 15 9.75M21 12a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z"
      />
    </svg>
  );
}

function ClockIcon({ title }: { title: string }): VNode {
  return (
    <svg
      xmlns="http://www.w3.org/2000/svg"
      fill="none"
      viewBox="0 0 24 24"
      stroke-width="1.5"
      stroke="currentColor"
      class="size-6"
      title={title}
    >
      <path
        stroke-linecap="round"
        stroke-linejoin="round"
        d="M12 6v6h4.5m4.5 0a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z"
      />
    </svg>
  );
}
