// SPDX-License-Identifier: GPL-2.0-or-later
/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2012  Intel Corporation. All rights reserved.
 *
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <unistd.h>
#include <stdlib.h>
#include <stdbool.h>
#include <inttypes.h>
#include <string.h>
#include <fcntl.h>
#include <sys/socket.h>

#include <glib.h>

#include "src/shared/util.h"
#include "src/shared/queue.h"
#include "src/shared/tester.h"
#include "src/log.h"

#include "unit/avdtp.h"

#define MAX_SEID 0x3E

struct test_pdu {
	bool valid;
	bool fragmented;
	uint8_t *data;
	size_t size;
};

struct test_data {
	char *test_name;
	struct test_pdu *pdu_list;
};

#define data(args...) ((const unsigned char[]) { args })

#define raw_pdu(args...) \
	{							\
		.valid = true,					\
		.data = util_memdup(data(args), sizeof(data(args))), \
		.size = sizeof(data(args)),			\
	}

#define frg_pdu(args...) \
	{							\
		.valid = true,					\
		.fragmented = true,				\
		.data = util_memdup(data(args), sizeof(data(args))), \
		.size = sizeof(data(args)),			\
	}

#define define_test(name, function, args...) \
	do {								\
		const struct test_pdu pdus[] = {			\
			args, { }					\
		};							\
		static struct test_data data;				\
		data.test_name = g_strdup(name);			\
		data.pdu_list = util_memdup(pdus, sizeof(pdus));	\
		tester_add(name, &data, NULL, function, NULL);	\
	} while (0)

struct context {
	struct avdtp *session;
	struct avdtp_local_sep *sep;
	struct avdtp_stream *stream;
	struct queue *lseps;
	guint source;
	guint process;
	int fd;
	int mtu;
	gboolean pending_open;
	gboolean pending_suspend;
	unsigned int pdu_offset;
	const struct test_data *data;
};

static void test_free(gconstpointer user_data)
{
	const struct test_data *data = user_data;
	struct test_pdu *pdu;
	int i;

	for (i = 0; (pdu = &data->pdu_list[i]) && pdu->valid; i++)
		g_free(pdu->data);

	g_free(data->test_name);
	g_free(data->pdu_list);
}

static void unregister_sep(void *data)
{
	struct avdtp_local_sep *sep = data;

	/* Removed from the queue by caller */
	avdtp_unregister_sep(NULL, sep);
}

static void destroy_context(struct context *context)
{
	if (context->source > 0)
		g_source_remove(context->source);
	avdtp_unref(context->session);

	test_free(context->data);
	queue_destroy(context->lseps, unregister_sep);

	g_free(context);
}

static gboolean context_quit(gpointer user_data)
{
	struct context *context = user_data;

	if (context->process > 0)
		g_source_remove(context->process);

	destroy_context(context);

	tester_test_passed();

	return FALSE;
}

static gboolean send_pdu(gpointer user_data)
{
	struct context *context = user_data;
	const struct test_pdu *pdu;
	ssize_t len;

	pdu = &context->data->pdu_list[context->pdu_offset++];

	len = write(context->fd, pdu->data, pdu->size);

	tester_monitor('<', 0x0000, 0x0019, pdu->data, len);

	g_assert_cmpint(len, ==, pdu->size);

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-02-C"))
		g_timeout_add(100, context_quit, context);

	if (pdu->fragmented)
		return send_pdu(user_data);

	context->process = 0;
	return FALSE;
}

static void context_process(struct context *context)
{
	if (!context->data->pdu_list[context->pdu_offset].valid) {
		context_quit(context);
		return;
	}

	context->process = g_idle_add(send_pdu, context);
}

static gboolean transport_open(struct avdtp_stream *stream)
{
	int fd;

	fd = open("/dev/null", O_RDWR, 0);
	if (fd < 0)
		g_assert_not_reached();

	return avdtp_stream_set_transport(stream, fd, 672, 672);
}

static gboolean test_handler(GIOChannel *channel, GIOCondition cond,
							gpointer user_data)
{
	struct context *context = user_data;
	const struct test_pdu *pdu;
	unsigned char buf[512];
	ssize_t len;
	int fd;

	pdu = &context->data->pdu_list[context->pdu_offset++];

	if (cond & (G_IO_NVAL | G_IO_ERR | G_IO_HUP)) {
		context->source = 0;
		return FALSE;
	}

	fd = g_io_channel_unix_get_fd(channel);

	len = read(fd, buf, sizeof(buf));

	g_assert(len > 0);

	tester_monitor('>', 0x0000, 0x0019, buf, len);

	g_assert_cmpint(len, ==, pdu->size);

	g_assert(memcmp(buf, pdu->data, pdu->size) == 0);

	if (context->pending_open) {
		context->pending_open = FALSE;
		g_assert(transport_open(context->stream));
	}

	if (context->pending_suspend) {
		int ret;

		context->pending_suspend = FALSE;
		ret = avdtp_suspend(context->session, context->stream);
		g_assert_cmpint(ret, ==, 0);
	}

	if (!pdu->fragmented)
		context_process(context);

	return TRUE;
}

static struct context *context_new(uint16_t version, uint16_t imtu,
					uint16_t omtu, gconstpointer data)
{
	struct context *context = g_new0(struct context, 1);
	GIOChannel *channel;
	int err, sv[2];

	err = socketpair(AF_UNIX, SOCK_SEQPACKET | SOCK_CLOEXEC, 0, sv);
	g_assert(err == 0);

	context->lseps = queue_new();
	g_assert(context->lseps);

	context->session = avdtp_new(sv[0], imtu, omtu, version,
								context->lseps);
	g_assert(context->session != NULL);

	channel = g_io_channel_unix_new(sv[1]);

	g_io_channel_set_close_on_unref(channel, TRUE);
	g_io_channel_set_encoding(channel, NULL, NULL);
	g_io_channel_set_buffered(channel, FALSE);

	context->source = g_io_add_watch(channel,
				G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
				test_handler, context);
	g_assert(context->source > 0);

	g_io_channel_unref(channel);

	context->fd = sv[1];
	context->data = data;

	return context;
}

static struct context *create_context(uint16_t version, gconstpointer data)
{
	return context_new(version, 672, 672, data);
}

static gboolean sep_getcap_ind(struct avdtp *session,
					struct avdtp_local_sep *sep,
					GSList **caps, uint8_t *err,
					void *user_data)
{
	struct avdtp_service_capability *media_transport, *media_codec;
	struct avdtp_media_codec_capability *codec_caps;
	uint8_t cap[4] = { 0xff, 0xff, 2, 64 };

	*caps = NULL;

	media_transport = avdtp_service_cap_new(AVDTP_MEDIA_TRANSPORT,
						NULL, 0);

	*caps = g_slist_append(*caps, media_transport);

	codec_caps = g_malloc0(sizeof(*codec_caps) + sizeof(cap));
	codec_caps->media_type = AVDTP_MEDIA_TYPE_AUDIO;
	codec_caps->media_codec_type = 0x00;
	memcpy(codec_caps->data, cap, sizeof(cap));

	media_codec = avdtp_service_cap_new(AVDTP_MEDIA_CODEC, codec_caps,
					sizeof(*codec_caps) + sizeof(cap));

	*caps = g_slist_append(*caps, media_codec);
	g_free(codec_caps);

	return TRUE;
}

static gboolean sep_open_ind(struct avdtp *session, struct avdtp_local_sep *sep,
				struct avdtp_stream *stream, uint8_t *err,
				void *user_data)
{
	struct context *context = user_data;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-18-C")) {
		*err = 0xc0;
		return FALSE;
	}

	context->pending_open = TRUE;
	context->stream = stream;

	return TRUE;
}

static gboolean sep_setconf_ind(struct avdtp *session,
						struct avdtp_local_sep *sep,
						struct avdtp_stream *stream,
						GSList *caps,
						avdtp_set_configuration_cb cb,
						void *user_data)
{
	struct context *context = user_data;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-09-C"))
		return FALSE;

	cb(session, stream, NULL);

	return TRUE;
}

static gboolean sep_start_ind(struct avdtp *session,
						struct avdtp_local_sep *sep,
						struct avdtp_stream *stream,
						uint8_t *err,
						void *user_data)
{
	struct context *context = user_data;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-21-C")) {
		*err = 0xc0;
		return FALSE;
	}

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-25-C"))
		context->pending_suspend = TRUE;

	return TRUE;
}

static gboolean sep_close_ind(struct avdtp *session,
						struct avdtp_local_sep *sep,
						struct avdtp_stream *stream,
						uint8_t *err,
						void *user_data)
{
	struct context *context = user_data;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-24-C")) {
		*err = 0xc0;
		return FALSE;
	}

	return TRUE;
}

static gboolean sep_suspend_ind(struct avdtp *session,
						struct avdtp_local_sep *sep,
						struct avdtp_stream *stream,
						uint8_t *err,
						void *user_data)
{
	struct context *context = user_data;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-27-C")) {
		*err = 0xc0;
		return FALSE;
	}

	return TRUE;
}

static struct avdtp_sep_ind sep_ind = {
	.get_capability		= sep_getcap_ind,
	.set_configuration	= sep_setconf_ind,
	.open			= sep_open_ind,
	.close			= sep_close_ind,
	.start			= sep_start_ind,
	.suspend		= sep_suspend_ind,
};

static void sep_setconf_cfm(struct avdtp *session, struct avdtp_local_sep *sep,
				struct avdtp_stream *stream,
				struct avdtp_error *err, void *user_data)
{
	struct context *context = user_data;
	int ret;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BV-09-C")) {
		context_quit(context);
		return;
	}

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-07-C")) {
		g_assert(err != NULL);
		g_assert_cmpint(avdtp_error_error_code(err), ==, 0x13);
		context_quit(context);
		return;
	}

	g_assert(err == NULL);

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BV-11-C") ||
		g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-10-C"))
		ret = avdtp_get_configuration(session, stream);
	else if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BV-23-C"))
		ret = avdtp_abort(session, stream);
	else
		ret = avdtp_open(session, stream);

	g_assert_cmpint(ret, ==, 0);
}

static void sep_getconf_cfm(struct avdtp *session, struct avdtp_local_sep *sep,
				struct avdtp_stream *stream,
				struct avdtp_error *err, void *user_data)
{
	struct context *context = user_data;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-10-C")) {
		g_assert(err != NULL);
		g_assert_cmpint(avdtp_error_error_code(err), ==, 0x12);
	} else
		g_assert(err == NULL);

	context_quit(context);
}

static void sep_open_cfm(struct avdtp *session, struct avdtp_local_sep *sep,
			struct avdtp_stream *stream, struct avdtp_error *err,
			void *user_data)
{
	int ret;

	g_assert(err == NULL);

	g_assert(transport_open(stream));

	ret = avdtp_start(session, stream);
	g_assert_cmpint(ret, ==, 0);
}

static void sep_start_cfm(struct avdtp *session, struct avdtp_local_sep *sep,
			struct avdtp_stream *stream, struct avdtp_error *err,
			void *user_data)
{
	struct context *context = user_data;
	int ret;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-19-C") ||
		g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-22-C")) {
		g_assert(err != NULL);
		g_assert_cmpint(avdtp_error_error_code(err), ==, 0x31);
		context_quit(context);
		return;
	}

	g_assert(err == NULL);

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BV-19-C"))
		ret = avdtp_close(session, stream, FALSE);
	else
		ret = avdtp_suspend(session, stream);

	g_assert_cmpint(ret, ==, 0);
}

static void sep_suspend_cfm(struct avdtp *session, struct avdtp_local_sep *sep,
			struct avdtp_stream *stream, struct avdtp_error *err,
			void *user_data)
{
	struct context *context = user_data;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-25-C")) {
		g_assert(err != NULL);
		g_assert_cmpint(avdtp_error_error_code(err), ==, 0x31);
		context_quit(context);
	}
}

static struct avdtp_sep_cfm sep_cfm = {
	.set_configuration	= sep_setconf_cfm,
	.get_configuration	= sep_getconf_cfm,
	.open			= sep_open_cfm,
	.start			= sep_start_cfm,
	.suspend		= sep_suspend_cfm,
};

static void test_server(gconstpointer data)
{
	struct context *context = create_context(0x0100, data);
	struct avdtp_local_sep *sep;

	sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SOURCE,
					AVDTP_MEDIA_TYPE_AUDIO,
					0x00, FALSE, &sep_ind, &sep_cfm,
					context);
	g_assert(sep);

	g_idle_add(send_pdu, context);
}

static void test_server_1_3(gconstpointer data)
{
	struct context *context = create_context(0x0103, data);
	struct avdtp_local_sep *sep;

	sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SOURCE,
					AVDTP_MEDIA_TYPE_AUDIO,
					0x00, TRUE, &sep_ind, NULL, context);
	g_assert(sep);

	g_idle_add(send_pdu, context);
}

static void test_server_1_3_sink(gconstpointer data)
{
	struct context *context = create_context(0x0103, data);
	struct avdtp_local_sep *sep;

	sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SINK,
					AVDTP_MEDIA_TYPE_AUDIO,
					0x00, TRUE, &sep_ind, NULL, context);
	g_assert(sep);

	g_idle_add(send_pdu, context);
}

static void test_server_0_sep(gconstpointer data)
{
	struct context *context = create_context(0x0100, data);

	g_idle_add(send_pdu, context);
}

static void test_server_seid(gconstpointer data)
{
	struct context *context = create_context(0x0103, data);
	struct avdtp_local_sep *sep;
	unsigned int i;

	for (i = 0; i < MAX_SEID; i++) {
		sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SINK,
						AVDTP_MEDIA_TYPE_AUDIO,
						0x00, TRUE, &sep_ind, NULL,
						context);
		g_assert(sep);
	}

	/* Now add (MAX_SEID + 1) SEP -> it shall fail */
	sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SINK,
						AVDTP_MEDIA_TYPE_AUDIO,
						0x00, TRUE, &sep_ind, NULL,
						context);
	g_assert(!sep);

	context_quit(context);
}

static void test_server_seid_duplicate(gconstpointer data)
{
	struct context *context = create_context(0x0103, data);
	struct avdtp_local_sep *sep;
	int i;

	for (i = 0; i < 2; i++) {
		sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SINK,
						AVDTP_MEDIA_TYPE_AUDIO,
						0x00, TRUE, &sep_ind, NULL,
						context);
		g_assert(sep);
	}

	/* Remove 1st element */
	sep = queue_peek_head(context->lseps);
	g_assert(sep);

	avdtp_unregister_sep(context->lseps, sep);

	/* Now register new element */
	sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SINK,
						AVDTP_MEDIA_TYPE_AUDIO,
						0x00, TRUE, &sep_ind, NULL,
						context);
	g_assert(sep);

	/* Check SEID ids with DISCOVER */

	g_idle_add(send_pdu, context);
}

static gboolean sep_getcap_ind_frg(struct avdtp *session,
					struct avdtp_local_sep *sep,
					GSList **caps, uint8_t *err,
					void *user_data)
{
	struct avdtp_service_capability *media_transport, *media_codec;
	struct avdtp_service_capability *content_protection;
	struct avdtp_media_codec_capability *codec_caps;
	uint8_t cap[4] = { 0xff, 0xff, 2, 64 };
	uint8_t frg_cap[96] = {};

	*caps = NULL;

	media_transport = avdtp_service_cap_new(AVDTP_MEDIA_TRANSPORT,
						NULL, 0);

	*caps = g_slist_append(*caps, media_transport);

	codec_caps = g_malloc0(sizeof(*codec_caps) + sizeof(cap));
	codec_caps->media_type = AVDTP_MEDIA_TYPE_AUDIO;
	codec_caps->media_codec_type = 0x00;
	memcpy(codec_caps->data, cap, sizeof(cap));

	media_codec = avdtp_service_cap_new(AVDTP_MEDIA_CODEC, codec_caps,
					sizeof(*codec_caps) + sizeof(cap));

	*caps = g_slist_append(*caps, media_codec);
	g_free(codec_caps);

	content_protection = avdtp_service_cap_new(AVDTP_CONTENT_PROTECTION,
						frg_cap, sizeof(frg_cap));
	*caps = g_slist_append(*caps, content_protection);

	return TRUE;
}

static struct avdtp_sep_ind sep_ind_frg = {
	.get_capability		= sep_getcap_ind_frg,
};

static void test_server_frg(gconstpointer data)
{
	struct context *context = context_new(0x0100, 48, 48, data);
	struct avdtp_local_sep *sep;

	sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SOURCE,
						AVDTP_MEDIA_TYPE_AUDIO,
						0x00, TRUE, &sep_ind_frg,
						NULL, context);
	g_assert(sep);

	g_idle_add(send_pdu, context);
}

static void discover_cb(struct avdtp *session, GSList *seps,
				struct avdtp_error *err, void *user_data)
{
	struct context *context = user_data;
	struct avdtp_stream *stream;
	struct avdtp_remote_sep *rsep;
	struct avdtp_service_capability *media_transport, *media_codec;
	struct avdtp_media_codec_capability *cap;
	GSList *caps;
	uint8_t data[4] = { 0x21, 0x02, 2, 32 };
	int ret;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BV-05-C") ||
		g_str_equal(context->data->test_name, "/TP/SIG/SMG/BV-07-C") ||
		g_str_equal(context->data->test_name, "/TP/SIG/SMG/BV-25-C"))
		return;

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-01-C")) {
		g_assert(err != NULL);
		g_assert_cmpint(avdtp_error_error_code(err), ==, 0x01);
		context_quit(context);
		return;
	}

	if (g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-04-C") ||
		g_str_equal(context->data->test_name, "/TP/SIG/SMG/BI-32-C")) {
		g_assert(err != NULL);
		g_assert_cmpint(avdtp_error_error_code(err), ==, 0x11);
		context_quit(context);
		return;
	}

	g_assert(err == NULL);
	g_assert_cmpint(g_slist_length(seps), !=, 0);

	if (g_str_equal(context->data->test_name, "/TP/SIG/FRA/BV-02-C")) {
		g_assert(err == NULL);
		context_quit(context);
		return;
	}

	rsep = avdtp_find_remote_sep(session, context->sep);
	g_assert(rsep != NULL);

	media_transport = avdtp_service_cap_new(AVDTP_MEDIA_TRANSPORT,
						NULL, 0);

	caps = g_slist_append(NULL, media_transport);

	cap = g_malloc0(sizeof(*cap) + sizeof(data));
	cap->media_type = AVDTP_MEDIA_TYPE_AUDIO;
	cap->media_codec_type = 0x00;
	memcpy(cap->data, data, sizeof(data));

	media_codec = avdtp_service_cap_new(AVDTP_MEDIA_CODEC, cap,
						sizeof(*cap) + sizeof(data));

	caps = g_slist_append(caps, media_codec);
	g_free(cap);

	ret = avdtp_set_configuration(session, rsep, context->sep, caps,
								&stream);
	g_assert_cmpint(ret, ==, 0);

	g_slist_free_full(caps, g_free);
}

static void test_client(gconstpointer data)
{
	struct context *context = create_context(0x0100, data);
	struct avdtp_local_sep *sep;

	sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SINK,
					AVDTP_MEDIA_TYPE_AUDIO,
					0x00, FALSE, NULL, &sep_cfm, context);

	context->sep = sep;

	avdtp_discover(context->session, discover_cb, context);
}

static void test_client_1_3(gconstpointer data)
{
	struct context *context = create_context(0x0103, data);
	struct avdtp_local_sep *sep;

	sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SINK,
					AVDTP_MEDIA_TYPE_AUDIO,
					0x00, TRUE, NULL, &sep_cfm, context);

	context->sep = sep;

	avdtp_discover(context->session, discover_cb, context);
}

static void test_client_frg(gconstpointer data)
{
	struct context *context = context_new(0x0100, 48, 48, data);
	struct avdtp_local_sep *sep;

	sep = avdtp_register_sep(context->lseps, AVDTP_SEP_TYPE_SINK,
					AVDTP_MEDIA_TYPE_AUDIO,
					0x00, TRUE, NULL, &sep_cfm, context);

	context->sep = sep;

	avdtp_discover(context->session, discover_cb, context);
}

int main(int argc, char *argv[])
{
	tester_init(&argc, &argv);

	__btd_log_init("*", 0);

	/*
	 * Stream Management Service
	 *
	 * To verify that the following procedures are implemented according to
	 * their specification in AVDTP.
	 */
	define_test("/TP/SIG/SMG/BV-06-C-SEID-1", test_server_seid,
			raw_pdu(0x00));
	define_test("/TP/SIG/SMG/BV-06-C-SEID-2", test_server_seid_duplicate,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x08, 0x08, 0x04, 0x08));
	define_test("/TP/SIG/SMG/BV-05-C", test_client,
			raw_pdu(0x00, 0x01));
	define_test("/TP/SIG/SMG/BV-06-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00));
	define_test("/TP/SIG/SMG/BV-07-C", test_client,
			raw_pdu(0x10, 0x01),
			raw_pdu(0x12, 0x01, 0x04, 0x00),
			raw_pdu(0x20, 0x02, 0x04));
	define_test("/TP/SIG/SMG/BV-08-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40));
	define_test("/TP/SIG/SMG/BV-09-C", test_client,
			raw_pdu(0x30, 0x01),
			raw_pdu(0x32, 0x01, 0x04, 0x00),
			raw_pdu(0x40, 0x02, 0x04),
			raw_pdu(0x42, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x50, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x52, 0x03));
	define_test("/TP/SIG/SMG/BV-10-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03));
	define_test("/TP/SIG/SMG/BV-11-C", test_client,
			raw_pdu(0x60, 0x01),
			raw_pdu(0x62, 0x01, 0x04, 0x00),
			raw_pdu(0x70, 0x02, 0x04),
			raw_pdu(0x72, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x80, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x82, 0x03),
			raw_pdu(0x90, 0x04, 0x04));
	define_test("/TP/SIG/SMG/BV-12-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x04, 0x04),
			raw_pdu(0x32, 0x04, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0x21, 0x02, 0x02, 0x20));
	define_test("/TP/SIG/SMG/BV-15-C", test_client,
			raw_pdu(0xa0, 0x01),
			raw_pdu(0xa2, 0x01, 0x04, 0x00),
			raw_pdu(0xb0, 0x02, 0x04),
			raw_pdu(0xb2, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0xc0, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0xc2, 0x03),
			raw_pdu(0xd0, 0x06, 0x04));
	define_test("/TP/SIG/SMG/BV-16-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06));
	define_test("/TP/SIG/SMG/BV-17-C", test_client,
			raw_pdu(0xe0, 0x01),
			raw_pdu(0xe2, 0x01, 0x04, 0x00),
			raw_pdu(0xf0, 0x02, 0x04),
			raw_pdu(0xf2, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x00, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x02, 0x03),
			raw_pdu(0x10, 0x06, 0x04),
			raw_pdu(0x12, 0x06),
			raw_pdu(0x20, 0x07, 0x04));
	define_test("/TP/SIG/SMG/BV-18-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x07, 0x04),
			raw_pdu(0x42, 0x07));
	define_test("/TP/SIG/SMG/BV-19-C", test_client,
			raw_pdu(0x30, 0x01),
			raw_pdu(0x32, 0x01, 0x04, 0x00),
			raw_pdu(0x40, 0x02, 0x04),
			raw_pdu(0x42, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x50, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x52, 0x03),
			raw_pdu(0x60, 0x06, 0x04),
			raw_pdu(0x62, 0x06),
			raw_pdu(0x70, 0x07, 0x04),
			raw_pdu(0x72, 0x07),
			raw_pdu(0x80, 0x08, 0x04));
	define_test("/TP/SIG/SMG/BV-20-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x07, 0x04),
			raw_pdu(0x42, 0x07),
			raw_pdu(0x50, 0x08, 0x04),
			raw_pdu(0x52, 0x08));
	define_test("/TP/SIG/SMG/BV-21-C", test_client,
			raw_pdu(0x90, 0x01),
			raw_pdu(0x92, 0x01, 0x04, 0x00),
			raw_pdu(0xa0, 0x02, 0x04),
			raw_pdu(0xa2, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0xb0, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0xb2, 0x03),
			raw_pdu(0xc0, 0x06, 0x04),
			raw_pdu(0xc2, 0x06),
			raw_pdu(0xd0, 0x07, 0x04),
			raw_pdu(0xd2, 0x07),
			raw_pdu(0xe0, 0x09, 0x04));
	define_test("/TP/SIG/SMG/BV-22-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x07, 0x04),
			raw_pdu(0x42, 0x07),
			raw_pdu(0x50, 0x09, 0x04),
			raw_pdu(0x52, 0x09));
	define_test("/TP/SIG/SMG/BV-23-C", test_client,
			raw_pdu(0xf0, 0x01),
			raw_pdu(0xf2, 0x01, 0x04, 0x00),
			raw_pdu(0x00, 0x02, 0x04),
			raw_pdu(0x02, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x10, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x12, 0x03),
			raw_pdu(0x20, 0x0a, 0x04));
	define_test("/TP/SIG/SMG/BV-24-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x0a, 0x04),
			raw_pdu(0x32, 0x0a));
	define_test("/TP/SIG/SMG/BV-25-C", test_client_1_3,
			raw_pdu(0x30, 0x01),
			raw_pdu(0x32, 0x01, 0x04, 0x00),
			raw_pdu(0x40, 0x0c, 0x04));
	define_test("/TP/SIG/SMG/BV-26-C", test_server_1_3,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x0c, 0x04),
			raw_pdu(0x12, 0x0c, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40, 0x08, 0x00));
	define_test("/TP/SIG/SMG/BV-27-C", test_server_1_3,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40));
	define_test("/TP/SIG/SMG/BV-28-C", test_client_1_3,
			raw_pdu(0x50, 0x01),
			raw_pdu(0x52, 0x01, 0x04, 0x00),
			raw_pdu(0x60, 0x0c, 0x04),
			raw_pdu(0x62, 0x0c, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40, 0x0f, 0x00),
			raw_pdu(0x70, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20));
	define_test("/TP/SIG/SMG/BV-31-C", test_client_1_3,
			raw_pdu(0x80, 0x01),
			raw_pdu(0x82, 0x01, 0x04, 0x00),
			raw_pdu(0x90, 0x0c, 0x04),
			raw_pdu(0x92, 0x0c, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00,
				0x04, 0x00, 0x05, 0x00, 0x06, 0x00, 0x07, 0x06,
				0x00, 0x00, 0xff, 0xff, 0x02, 0x40, 0x08, 0x00),
			raw_pdu(0xa0, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20, 0x08,
				0x00));
	define_test("/TP/SIG/SMG/BI-01-C", test_client,
			raw_pdu(0xb0, 0x01),
			raw_pdu(0xb3, 0x01, 0x01));
	define_test("/TP/SIG/SMG/BI-02-C", test_server,
			raw_pdu(0x01, 0x01));
	define_test("/TP/SIG/SMG/BI-03-C", test_server_0_sep,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x03, 0x01, 0x19));
	define_test("/TP/SIG/SMG/BI-04-C", test_client,
			raw_pdu(0xc0, 0x01),
			raw_pdu(0xc2, 0x01, 0x04, 0x00),
			raw_pdu(0xd0, 0x02, 0x04),
			raw_pdu(0xd3, 0x02, 0x11));
	define_test("/TP/SIG/SMG/BI-05-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02),
			raw_pdu(0x13, 0x02, 0x11));
	define_test("/TP/SIG/SMG/BI-06-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x00),
			raw_pdu(0x13, 0x02, 0x12));
	define_test("/TP/SIG/SMG/BI-07-C", test_client,
			raw_pdu(0xe0, 0x01),
			raw_pdu(0xe2, 0x01, 0x04, 0x00),
			raw_pdu(0xf0, 0x02, 0x04),
			raw_pdu(0xf2, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x00, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x03, 0x03, 0x00, 0x13));
	define_test("/TP/SIG/SMG/BI-08-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x33, 0x03, 0x00, 0x13));
	define_test("/TP/SIG/SMG/BI-09-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x23, 0x03, 0x00, 0x29));
	define_test("/TP/SIG/SMG/BI-10-C", test_client,
			raw_pdu(0x10, 0x01),
			raw_pdu(0x12, 0x01, 0x04, 0x00),
			raw_pdu(0x20, 0x02, 0x04),
			raw_pdu(0x22, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x30, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x32, 0x03),
			raw_pdu(0x40, 0x04, 0x04),
			raw_pdu(0x43, 0x04, 0x12));
	define_test("/TP/SIG/SMG/BI-11-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x04, 0x00),
			raw_pdu(0x33, 0x04, 0x12));
	define_test("/TP/SIG/SMG/BI-17-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x33, 0x06, 0x31));
	define_test("/TP/SIG/SMG/BI-18-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x33, 0x06, 0xc0));
	define_test("/TP/SIG/SMG/BI-19-C", test_client,
			raw_pdu(0x50, 0x01),
			raw_pdu(0x52, 0x01, 0x04, 0x00),
			raw_pdu(0x60, 0x02, 0x04),
			raw_pdu(0x62, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x70, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x72, 0x03),
			raw_pdu(0x80, 0x06, 0x04),
			raw_pdu(0x82, 0x06),
			raw_pdu(0x90, 0x07, 0x04),
			raw_pdu(0x93, 0x07, 0x04, 0x31));
	define_test("/TP/SIG/SMG/BI-20-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x07, 0x04),
			raw_pdu(0x33, 0x07, 0x04, 0x31));
	define_test("/TP/SIG/SMG/BI-21-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x07, 0x04),
			raw_pdu(0x43, 0x07, 0x04, 0xc0));
	define_test("/TP/SIG/SMG/BI-22-C", test_client,
			raw_pdu(0xa0, 0x01),
			raw_pdu(0xa2, 0x01, 0x04, 0x00),
			raw_pdu(0xb0, 0x02, 0x04),
			raw_pdu(0xb2, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0xc0, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0xc2, 0x03),
			raw_pdu(0xd0, 0x06, 0x04),
			raw_pdu(0xd2, 0x06),
			raw_pdu(0xe0, 0x07, 0x04),
			raw_pdu(0xe3, 0x07, 0x04, 0x31));
	define_test("/TP/SIG/SMG/BI-23-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x08, 0x00),
			raw_pdu(0x43, 0x08, 0x12));
	define_test("/TP/SIG/SMG/BI-24-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x08, 0x04),
			raw_pdu(0x43, 0x08, 0xc0));
	define_test("/TP/SIG/SMG/BI-25-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x07, 0x04),
			raw_pdu(0x42, 0x07),
			raw_pdu(0xf0, 0x09, 0x04),
			raw_pdu(0xf3, 0x09, 0x04, 0x31));
	define_test("/TP/SIG/SMG/BI-26-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x09, 0x04),
			raw_pdu(0x43, 0x09, 0x04, 0x31));
	define_test("/TP/SIG/SMG/BI-27-C", test_server,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x07, 0x04),
			raw_pdu(0x42, 0x07),
			raw_pdu(0x50, 0x09, 0x04),
			raw_pdu(0x53, 0x09, 0x04, 0xc0));
	define_test("/TP/SIG/SMG/BI-28-C", test_server,
			raw_pdu(0x00, 0xff),
			raw_pdu(0x01, 0x3f));
	define_test("/TP/SIG/SMG/BI-30-C", test_client,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			raw_pdu(0x12, 0x02, 0xee, 0x01, 0x00, 0x01, 0x00, 0x07,
				0x06, 0x00, 0x00, 0xff, 0xff, 0x02, 0x40),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20));
	define_test("/TP/SIG/SMG/ESR04/BI-28-C", test_server,
			raw_pdu(0x00, 0x3f),
			raw_pdu(0x01, 0x3f));
	define_test("/TP/SIG/SMG/BI-32-C", test_client_1_3,
			raw_pdu(0x30, 0x01),
			raw_pdu(0x32, 0x01, 0x04, 0x00),
			raw_pdu(0x40, 0x0c, 0x04),
			raw_pdu(0x43, 0x0c, 0x11));
	define_test("/TP/SIG/SMG/BI-33-C", test_server_1_3,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x0c),
			raw_pdu(0x13, 0x0c, 0x11));
	define_test("/TP/SIG/SMG/BI-35-C", test_client_1_3,
			raw_pdu(0x50, 0x01),
			raw_pdu(0x52, 0x01, 0x04, 0x00),
			raw_pdu(0x60, 0x0c, 0x04),
			raw_pdu(0x62, 0x0c, 0x01, 0x00, 0x02, 0x00, 0x03, 0x00,
				0x04, 0x00, 0x05, 0x00, 0x06, 0x00, 0x07, 0x06,
				0x00, 0x00, 0xff, 0xff, 0x02, 0x40, 0x08, 0x00),
			raw_pdu(0x70, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20, 0x08,
				0x00));
	define_test("/TP/SIG/SMG/BI-36-C", test_client_1_3,
			raw_pdu(0x80, 0x01),
			raw_pdu(0x82, 0x01, 0x04, 0x00),
			raw_pdu(0x90, 0x0c, 0x04),
			raw_pdu(0x92, 0x0c, 0xee, 0x01, 0x00, 0x01, 0x00, 0x07,
				0x06, 0x00, 0x00, 0xff, 0xff, 0x02, 0x40),
			raw_pdu(0xa0, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20));

	/*
	 * Signaling Message Fragmentation Service
	 *
	 * verify that the IUT (INT and ACP) fragments the signaling messages
	 * that cannot fit in a single L2CAP packet.
	 */
	define_test("/TP/SIG/FRA/BV-01-C", test_server_frg,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x02, 0x04),
			frg_pdu(0x16, 0x03, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00,
				0x00, 0xff, 0xff, 0x02, 0x40, 0x04, 0x60, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00),
			frg_pdu(0x1a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00),
			raw_pdu(0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00));
	define_test("/TP/SIG/FRA/BV-02-C", test_client_frg,
			raw_pdu(0xb0, 0x01),
			raw_pdu(0xb2, 0x01, 0x04, 0x00),
			raw_pdu(0xc0, 0x02, 0x04),
			frg_pdu(0xc6, 0x03, 0x02, 0x01, 0x00, 0x07, 0x06, 0x00,
				0x00, 0xff, 0xff, 0x02, 0x40, 0x04, 0x60, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00),
			frg_pdu(0xca, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00),
			raw_pdu(0xce, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00));

	/*
	 * Delay Reporting
	 *
	 * Verify that the stream management signaling procedure of delay
	 * reporting is implemented according to its specification in AVDTP.
	 */
	define_test("/TP/SIG/SYN/BV-01-C", test_server_1_3_sink,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x08),
			raw_pdu(0x10, 0x0c, 0x04),
			raw_pdu(0x12, 0x0c, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40, 0x08, 0x00));
	define_test("/TP/SIG/SYN/BV-02-C", test_client_1_3,
			raw_pdu(0xd0, 0x01),
			raw_pdu(0xd2, 0x01, 0x04, 0x00),
			raw_pdu(0xe0, 0x0c, 0x04),
			raw_pdu(0xe2, 0x0c, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40, 0x0f, 0x00, 0x08, 0x00),
			raw_pdu(0xf0, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20, 0x08,
				0x00));
	define_test("/TP/SIG/SYN/BV-03-C", test_server_1_3_sink,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x08),
			raw_pdu(0x10, 0x0c, 0x04),
			raw_pdu(0x12, 0x0c, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40, 0x08, 0x00),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20, 0x08,
				0x00),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x00, 0x0d, 0x04, 0x00, 0x00));
	define_test("/TP/SIG/SYN/BV-04-C", test_client_1_3,
			raw_pdu(0x10, 0x01),
			raw_pdu(0x12, 0x01, 0x04, 0x00),
			raw_pdu(0x20, 0x0c, 0x04),
			raw_pdu(0x22, 0x0c, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40, 0x0f, 0x00, 0x08, 0x00),
			raw_pdu(0x30, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20, 0x08,
				0x00),
			raw_pdu(0x32, 0x03),
			raw_pdu(0x40, 0x0d, 0x04, 0x00, 0x00));
	define_test("/TP/SIG/SYN/BV-05-C", test_server_1_3,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x0c, 0x04),
			raw_pdu(0x12, 0x0c, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40, 0x08, 0x00),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20, 0x08,
				0x00),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x0d, 0x04, 0x00, 0x00),
			raw_pdu(0x42, 0x0d));
	define_test("/TP/SIG/SYN/BV-06-C", test_server_1_3,
			raw_pdu(0x00, 0x01),
			raw_pdu(0x02, 0x01, 0x04, 0x00),
			raw_pdu(0x10, 0x0c, 0x04),
			raw_pdu(0x12, 0x0c, 0x01, 0x00, 0x07, 0x06, 0x00, 0x00,
				0xff, 0xff, 0x02, 0x40, 0x08, 0x00),
			raw_pdu(0x20, 0x03, 0x04, 0x04, 0x01, 0x00, 0x07, 0x06,
				0x00, 0x00, 0x21, 0x02, 0x02, 0x20, 0x08,
				0x00),
			raw_pdu(0x22, 0x03),
			raw_pdu(0x30, 0x06, 0x04),
			raw_pdu(0x32, 0x06),
			raw_pdu(0x40, 0x07, 0x04),
			raw_pdu(0x42, 0x07),
			raw_pdu(0x50, 0x0d, 0x04, 0x00, 0x00),
			raw_pdu(0x52, 0x0d));

	return tester_run();
}
