/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set ts=2 sw=2 sts=2 et cindent: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef mozilla_H264Converter_h
#define mozilla_H264Converter_h

#include "PlatformDecoderModule.h"
#include "mozilla/Maybe.h"

namespace mozilla {

class DecoderDoctorDiagnostics;

DDLoggedTypeDeclNameAndBase(H264Converter, MediaDataDecoder);

// H264Converter is a MediaDataDecoder wrapper used to ensure that
// only AVCC or AnnexB is fed to the underlying MediaDataDecoder.
// The H264Converter allows playback of content where the SPS NAL may not be
// provided in the init segment (e.g. AVC3 or Annex B)
// H264Converter will monitor the input data, and will delay creation of the
// MediaDataDecoder until a SPS and PPS NALs have been extracted.

class H264Converter
  : public MediaDataDecoder
  , public DecoderDoctorLifeLogger<H264Converter>
{
public:

  H264Converter(PlatformDecoderModule* aPDM,
                const CreateDecoderParams& aParams);
  virtual ~H264Converter();

  RefPtr<InitPromise> Init() override;
  RefPtr<DecodePromise> Decode(MediaRawData* aSample) override;
  RefPtr<DecodePromise> Drain() override;
  RefPtr<FlushPromise> Flush() override;
  RefPtr<ShutdownPromise> Shutdown() override;
  bool IsHardwareAccelerated(nsACString& aFailureReason) const override;
  nsCString GetDescriptionName() const override
  {
    if (mDecoder) {
      return mDecoder->GetDescriptionName();
    }
    return NS_LITERAL_CSTRING("H264Converter decoder (pending)");
  }
  void SetSeekThreshold(const media::TimeUnit& aTime) override;
  bool SupportDecoderRecycling() const override
  {
    if (mDecoder) {
      return mDecoder->SupportDecoderRecycling();
    }
    return false;
  }

  ConversionRequired NeedsConversion() const override
  {
    if (mDecoder) {
      return mDecoder->NeedsConversion();
    }
    // Default so no conversion is performed.
    return ConversionRequired::kNeedAVCC;
  }
  MediaResult GetLastError() const { return mLastError; }

private:
  // Will create the required MediaDataDecoder if need AVCC and we have a SPS NAL.
  // Returns NS_ERROR_FAILURE if error is permanent and can't be recovered and
  // will set mError accordingly.
  MediaResult CreateDecoder(const VideoInfo& aConfig,
                         DecoderDoctorDiagnostics* aDiagnostics);
  MediaResult CreateDecoderAndInit(MediaRawData* aSample);
  MediaResult CheckForSPSChange(MediaRawData* aSample);
  void UpdateConfigFromExtraData(MediaByteBuffer* aExtraData);

  bool CanRecycleDecoder() const;

  void DecodeFirstSample(MediaRawData* aSample);
  void DrainThenFlushDecoder(MediaRawData* aPendingSample);
  void FlushThenShutdownDecoder(MediaRawData* aPendingSample);
  RefPtr<ShutdownPromise> ShutdownDecoder();

  RefPtr<PlatformDecoderModule> mPDM;
  const VideoInfo mOriginalConfig;
  VideoInfo mCurrentConfig;
  // Current out of band extra data (as found in metadata's VideoInfo).
  RefPtr<MediaByteBuffer> mOriginalExtraData;
  RefPtr<layers::KnowsCompositor> mKnowsCompositor;
  RefPtr<layers::ImageContainer> mImageContainer;
  const RefPtr<TaskQueue> mTaskQueue;
  RefPtr<MediaDataDecoder> mDecoder;
  MozPromiseRequestHolder<InitPromise> mInitPromiseRequest;
  MozPromiseRequestHolder<DecodePromise> mDecodePromiseRequest;
  MozPromiseHolder<DecodePromise> mDecodePromise;
  MozPromiseRequestHolder<FlushPromise> mFlushRequest;
  MediaDataDecoder::DecodedData mPendingFrames;
  MozPromiseRequestHolder<DecodePromise> mDrainRequest;
  MozPromiseRequestHolder<ShutdownPromise> mShutdownRequest;
  RefPtr<ShutdownPromise> mShutdownPromise;
  MozPromiseHolder<FlushPromise> mFlushPromise;

  RefPtr<GMPCrashHelper> mGMPCrashHelper;
  Maybe<bool> mNeedAVCC;
  MediaResult mLastError;
  bool mNeedKeyframe = true;
  const TrackInfo::TrackType mType;
  MediaEventProducer<TrackInfo::TrackType>* const mOnWaitingForKeyEvent;
  const CreateDecoderParams::OptionSet mDecoderOptions;
  const CreateDecoderParams::VideoFrameRate mRate;
  Maybe<bool> mCanRecycleDecoder;
};

} // namespace mozilla

#endif // mozilla_H264Converter_h
