// META: title=test WebNN API element-wise min operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-binary
// Compute the element-wise binary minimum of the two input tensors.
// MLOperand min(MLOperand a, MLOperand b);


const getMinPrecisionTolerance = (graphResources) => {
  const toleranceValueDict = {float32: 0, float16: 0};
  const expectedDataType =
      getExpectedDataTypeOfSingleOutput(graphResources.expectedOutputs);
  return {metricType: 'ULP', value: toleranceValueDict[expectedDataType]};
};

const minTests = [
  {
    'name': 'min float32 1D constant tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   62.07444381713867,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, 96.94400787353516,   -40.39130401611328,
            74.14437103271484,   0.03283197432756424, 38.79835510253906,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        },
        'inputB': {
          'data': [
            -40.10139083862305, 86.25190734863281,  51.280174255371094,
            -57.64906311035156, -97.56107330322266, -28.881731033325195,
            80.49571228027344,  46.6654052734375,   62.80685806274414,
            49.81534194946289,  -76.52043151855469, 84.5990982055664,
            50.47281265258789,  -18.01728630065918, 5.198459148406982,
            -47.82608413696289, 46.450077056884766, -71.25122833251953,
            -69.85066223144531, 40.676490783691406, -18.700122833251953,
            20.14988136291504,  41.95068359375,     23.482912063598633
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.10139083862305,  -38.2254524230957,   51.280174255371094,
            -57.64906311035156,  -97.56107330322266,  -28.881731033325195,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, -76.52043151855469,  -40.39130401611328,
            50.47281265258789,   -18.01728630065918,  5.198459148406982,
            -47.82608413696289,  17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'min float32 1D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   62.07444381713867,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, 96.94400787353516,   -40.39130401611328,
            74.14437103271484,   0.03283197432756424, 38.79835510253906,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -40.10139083862305, 86.25190734863281,  51.280174255371094,
            -57.64906311035156, -97.56107330322266, -28.881731033325195,
            80.49571228027344,  46.6654052734375,   62.80685806274414,
            49.81534194946289,  -76.52043151855469, 84.5990982055664,
            50.47281265258789,  -18.01728630065918, 5.198459148406982,
            -47.82608413696289, 46.450077056884766, -71.25122833251953,
            -69.85066223144531, 40.676490783691406, -18.700122833251953,
            20.14988136291504,  41.95068359375,     23.482912063598633
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.10139083862305,  -38.2254524230957,   51.280174255371094,
            -57.64906311035156,  -97.56107330322266,  -28.881731033325195,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, -76.52043151855469,  -40.39130401611328,
            50.47281265258789,   -18.01728630065918,  5.198459148406982,
            -47.82608413696289,  17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'min float32 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   62.07444381713867,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, 96.94400787353516,   -40.39130401611328,
            74.14437103271484,   0.03283197432756424, 38.79835510253906,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -40.10139083862305, 86.25190734863281,  51.280174255371094,
            -57.64906311035156, -97.56107330322266, -28.881731033325195,
            80.49571228027344,  46.6654052734375,   62.80685806274414,
            49.81534194946289,  -76.52043151855469, 84.5990982055664,
            50.47281265258789,  -18.01728630065918, 5.198459148406982,
            -47.82608413696289, 46.450077056884766, -71.25122833251953,
            -69.85066223144531, 40.676490783691406, -18.700122833251953,
            20.14988136291504,  41.95068359375,     23.482912063598633
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.10139083862305,  -38.2254524230957,   51.280174255371094,
            -57.64906311035156,  -97.56107330322266,  -28.881731033325195,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, -76.52043151855469,  -40.39130401611328,
            50.47281265258789,   -18.01728630065918,  5.198459148406982,
            -47.82608413696289,  17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'min float32 3D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   62.07444381713867,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, 96.94400787353516,   -40.39130401611328,
            74.14437103271484,   0.03283197432756424, 38.79835510253906,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -40.10139083862305, 86.25190734863281,  51.280174255371094,
            -57.64906311035156, -97.56107330322266, -28.881731033325195,
            80.49571228027344,  46.6654052734375,   62.80685806274414,
            49.81534194946289,  -76.52043151855469, 84.5990982055664,
            50.47281265258789,  -18.01728630065918, 5.198459148406982,
            -47.82608413696289, 46.450077056884766, -71.25122833251953,
            -69.85066223144531, 40.676490783691406, -18.700122833251953,
            20.14988136291504,  41.95068359375,     23.482912063598633
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.10139083862305,  -38.2254524230957,   51.280174255371094,
            -57.64906311035156,  -97.56107330322266,  -28.881731033325195,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, -76.52043151855469,  -40.39130401611328,
            50.47281265258789,   -18.01728630065918,  5.198459148406982,
            -47.82608413696289,  17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'min float32 4D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   62.07444381713867,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, 96.94400787353516,   -40.39130401611328,
            74.14437103271484,   0.03283197432756424, 38.79835510253906,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -40.10139083862305, 86.25190734863281,  51.280174255371094,
            -57.64906311035156, -97.56107330322266, -28.881731033325195,
            80.49571228027344,  46.6654052734375,   62.80685806274414,
            49.81534194946289,  -76.52043151855469, 84.5990982055664,
            50.47281265258789,  -18.01728630065918, 5.198459148406982,
            -47.82608413696289, 46.450077056884766, -71.25122833251953,
            -69.85066223144531, 40.676490783691406, -18.700122833251953,
            20.14988136291504,  41.95068359375,     23.482912063598633
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.10139083862305,  -38.2254524230957,   51.280174255371094,
            -57.64906311035156,  -97.56107330322266,  -28.881731033325195,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, -76.52043151855469,  -40.39130401611328,
            50.47281265258789,   -18.01728630065918,  5.198459148406982,
            -47.82608413696289,  17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'min float32 5D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   62.07444381713867,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, 96.94400787353516,   -40.39130401611328,
            74.14437103271484,   0.03283197432756424, 38.79835510253906,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -40.10139083862305, 86.25190734863281,  51.280174255371094,
            -57.64906311035156, -97.56107330322266, -28.881731033325195,
            80.49571228027344,  46.6654052734375,   62.80685806274414,
            49.81534194946289,  -76.52043151855469, 84.5990982055664,
            50.47281265258789,  -18.01728630065918, 5.198459148406982,
            -47.82608413696289, 46.450077056884766, -71.25122833251953,
            -69.85066223144531, 40.676490783691406, -18.700122833251953,
            20.14988136291504,  41.95068359375,     23.482912063598633
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.10139083862305,  -38.2254524230957,   51.280174255371094,
            -57.64906311035156,  -97.56107330322266,  -28.881731033325195,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, -76.52043151855469,  -40.39130401611328,
            50.47281265258789,   -18.01728630065918,  5.198459148406982,
            -47.82608413696289,  17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'min float32 broadcast 1D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [34.42634582519531],
          'descriptor': {shape: [1], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   62.07444381713867,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, 96.94400787353516,   -40.39130401611328,
            74.14437103271484,   0.03283197432756424, 38.79835510253906,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   34.42634582519531,
            -16.610267639160156, 34.42634582519531,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  34.42634582519531,
            -39.096099853515625, 34.42634582519531,   -40.39130401611328,
            34.42634582519531,   0.03283197432756424, 34.42634582519531,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'min float32 broadcast 2D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   62.07444381713867,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, 96.94400787353516,   -40.39130401611328,
            74.14437103271484,   0.03283197432756424, 38.79835510253906,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -19.072668075561523, -78.27516174316406, -13.436244010925293,
            -93.01346588134766, -72.27899169921875, 63.14110565185547
          ],
          'descriptor': {shape: [2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -36.06953048706055,  -78.27516174316406, -13.436244010925293,
            -93.01346588134766,  -72.27899169921875, -17.77212905883789,
            -76.01380920410156,  -78.27516174316406, -13.436244010925293,
            -93.01346588134766,  -72.27899169921875, -40.39130401611328,
            -19.072668075561523, -78.27516174316406, -13.436244010925293,
            -93.01346588134766,  -72.27899169921875, -82.3099365234375,
            -80.47379302978516,  -78.27516174316406, -73.2723617553711,
            -93.01346588134766,  -72.27899169921875, 4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'min float32 broadcast 3D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   62.07444381713867,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, 96.94400787353516,   -40.39130401611328,
            74.14437103271484,   0.03283197432756424, 38.79835510253906,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            23.231731414794922, 84.62673950195312, -83.33529663085938,
            -22.82455825805664
          ],
          'descriptor': {shape: [2, 2, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   23.231731414794922,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -83.33529663085938,  -83.33529663085938,  -83.33529663085938,
            -39.096099853515625, -22.82455825805664,  -40.39130401611328,
            23.231731414794922,  0.03283197432756424, 23.231731414794922,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -83.33529663085938,  -83.33529663085938,  -83.33529663085938,
            -33.74562072753906,  -22.82455825805664,  -22.82455825805664
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'min float32 broadcast 4D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [34.42634582519531],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   62.07444381713867,
            -16.610267639160156, 65.99324798583984,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  53.60376739501953,
            -39.096099853515625, 96.94400787353516,   -40.39130401611328,
            74.14437103271484,   0.03283197432756424, 38.79835510253906,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -36.06953048706055,  -38.2254524230957,   34.42634582519531,
            -16.610267639160156, 34.42634582519531,   -17.77212905883789,
            -76.01380920410156,  -69.59134674072266,  34.42634582519531,
            -39.096099853515625, 34.42634582519531,   -40.39130401611328,
            34.42634582519531,   0.03283197432756424, 34.42634582519531,
            -17.720787048339844, 17.383201599121094,  -82.3099365234375,
            -80.47379302978516,  -31.389848709106445, -73.2723617553711,
            -33.74562072753906,  -21.70152473449707,  4.945605278015137
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'min float16 1D constant tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.0625,   -38.21875, 62.0625,    -16.609375,      66,
            -17.765625, -76,       -69.5625,   53.59375,        -39.09375,
            96.9375,    -40.40625, 74.125,     0.0328369140625, 38.8125,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        },
        'inputB': {
          'data': [
            -40.09375, 86.25,      51.28125,   -57.65625, -97.5625, -28.875,
            80.5,      46.65625,   62.8125,    49.8125,   -76.5,    84.625,
            50.46875,  -18.015625, 5.19921875, -47.8125,  46.4375,  -71.25,
            -69.875,   40.6875,    -18.703125, 20.15625,  41.9375,  23.484375
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.09375, -38.21875, 51.28125,   -57.65625,  -97.5625,
            -28.875,   -76,       -69.5625,   53.59375,   -39.09375,
            -76.5,     -40.40625, 50.46875,   -18.015625, 5.19921875,
            -47.8125,  17.390625, -82.3125,   -80.5,      -31.390625,
            -73.25,    -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'min float16 1D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.0625,   -38.21875, 62.0625,    -16.609375,      66,
            -17.765625, -76,       -69.5625,   53.59375,        -39.09375,
            96.9375,    -40.40625, 74.125,     0.0328369140625, 38.8125,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -40.09375, 86.25,      51.28125,   -57.65625, -97.5625, -28.875,
            80.5,      46.65625,   62.8125,    49.8125,   -76.5,    84.625,
            50.46875,  -18.015625, 5.19921875, -47.8125,  46.4375,  -71.25,
            -69.875,   40.6875,    -18.703125, 20.15625,  41.9375,  23.484375
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.09375, -38.21875, 51.28125,   -57.65625,  -97.5625,
            -28.875,   -76,       -69.5625,   53.59375,   -39.09375,
            -76.5,     -40.40625, 50.46875,   -18.015625, 5.19921875,
            -47.8125,  17.390625, -82.3125,   -80.5,      -31.390625,
            -73.25,    -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'min float16 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.0625,   -38.21875, 62.0625,    -16.609375,      66,
            -17.765625, -76,       -69.5625,   53.59375,        -39.09375,
            96.9375,    -40.40625, 74.125,     0.0328369140625, 38.8125,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -40.09375, 86.25,      51.28125,   -57.65625, -97.5625, -28.875,
            80.5,      46.65625,   62.8125,    49.8125,   -76.5,    84.625,
            50.46875,  -18.015625, 5.19921875, -47.8125,  46.4375,  -71.25,
            -69.875,   40.6875,    -18.703125, 20.15625,  41.9375,  23.484375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.09375, -38.21875, 51.28125,   -57.65625,  -97.5625,
            -28.875,   -76,       -69.5625,   53.59375,   -39.09375,
            -76.5,     -40.40625, 50.46875,   -18.015625, 5.19921875,
            -47.8125,  17.390625, -82.3125,   -80.5,      -31.390625,
            -73.25,    -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'min float16 3D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.0625,   -38.21875, 62.0625,    -16.609375,      66,
            -17.765625, -76,       -69.5625,   53.59375,        -39.09375,
            96.9375,    -40.40625, 74.125,     0.0328369140625, 38.8125,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -40.09375, 86.25,      51.28125,   -57.65625, -97.5625, -28.875,
            80.5,      46.65625,   62.8125,    49.8125,   -76.5,    84.625,
            50.46875,  -18.015625, 5.19921875, -47.8125,  46.4375,  -71.25,
            -69.875,   40.6875,    -18.703125, 20.15625,  41.9375,  23.484375
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.09375, -38.21875, 51.28125,   -57.65625,  -97.5625,
            -28.875,   -76,       -69.5625,   53.59375,   -39.09375,
            -76.5,     -40.40625, 50.46875,   -18.015625, 5.19921875,
            -47.8125,  17.390625, -82.3125,   -80.5,      -31.390625,
            -73.25,    -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'min float16 4D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.0625,   -38.21875, 62.0625,    -16.609375,      66,
            -17.765625, -76,       -69.5625,   53.59375,        -39.09375,
            96.9375,    -40.40625, 74.125,     0.0328369140625, 38.8125,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -40.09375, 86.25,      51.28125,   -57.65625, -97.5625, -28.875,
            80.5,      46.65625,   62.8125,    49.8125,   -76.5,    84.625,
            50.46875,  -18.015625, 5.19921875, -47.8125,  46.4375,  -71.25,
            -69.875,   40.6875,    -18.703125, 20.15625,  41.9375,  23.484375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.09375, -38.21875, 51.28125,   -57.65625,  -97.5625,
            -28.875,   -76,       -69.5625,   53.59375,   -39.09375,
            -76.5,     -40.40625, 50.46875,   -18.015625, 5.19921875,
            -47.8125,  17.390625, -82.3125,   -80.5,      -31.390625,
            -73.25,    -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'min float16 5D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.0625,   -38.21875, 62.0625,    -16.609375,      66,
            -17.765625, -76,       -69.5625,   53.59375,        -39.09375,
            96.9375,    -40.40625, 74.125,     0.0328369140625, 38.8125,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -40.09375, 86.25,      51.28125,   -57.65625, -97.5625, -28.875,
            80.5,      46.65625,   62.8125,    49.8125,   -76.5,    84.625,
            50.46875,  -18.015625, 5.19921875, -47.8125,  46.4375,  -71.25,
            -69.875,   40.6875,    -18.703125, 20.15625,  41.9375,  23.484375
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -40.09375, -38.21875, 51.28125,   -57.65625,  -97.5625,
            -28.875,   -76,       -69.5625,   53.59375,   -39.09375,
            -76.5,     -40.40625, 50.46875,   -18.015625, 5.19921875,
            -47.8125,  17.390625, -82.3125,   -80.5,      -31.390625,
            -73.25,    -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'min float16 broadcast 1D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [34.4375],
          'descriptor': {shape: [1], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -36.0625,   -38.21875, 62.0625,    -16.609375,      66,
            -17.765625, -76,       -69.5625,   53.59375,        -39.09375,
            96.9375,    -40.40625, 74.125,     0.0328369140625, 38.8125,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -36.0625,   -38.21875, 34.4375,    -16.609375,      34.4375,
            -17.765625, -76,       -69.5625,   34.4375,         -39.09375,
            34.4375,    -40.40625, 34.4375,    0.0328369140625, 34.4375,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'min float16 broadcast 2D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.0625,   -38.21875, 62.0625,    -16.609375,      66,
            -17.765625, -76,       -69.5625,   53.59375,        -39.09375,
            96.9375,    -40.40625, 74.125,     0.0328369140625, 38.8125,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [-19.078125, -78.25, -13.4375, -93, -72.25, 63.15625],
          'descriptor': {shape: [2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -36.0625,   -78.25, -13.4375, -93, -72.25, -17.765625,
            -76,        -78.25, -13.4375, -93, -72.25, -40.40625,
            -19.078125, -78.25, -13.4375, -93, -72.25, -82.3125,
            -80.5,      -78.25, -73.25,   -93, -72.25, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'min float16 broadcast 3D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -36.0625,   -38.21875, 62.0625,    -16.609375,      66,
            -17.765625, -76,       -69.5625,   53.59375,        -39.09375,
            96.9375,    -40.40625, 74.125,     0.0328369140625, 38.8125,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [23.234375, 84.625, -83.3125, -22.828125],
          'descriptor': {shape: [2, 2, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -36.0625,   -38.21875, 23.234375,  -16.609375,      66,
            -17.765625, -83.3125,  -83.3125,   -83.3125,        -39.09375,
            -22.828125, -40.40625, 23.234375,  0.0328369140625, 23.234375,
            -17.71875,  17.390625, -82.3125,   -83.3125,        -83.3125,
            -83.3125,   -33.75,    -22.828125, -22.828125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'min float16 broadcast 4D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [34.4375],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -36.0625,   -38.21875, 62.0625,    -16.609375,      66,
            -17.765625, -76,       -69.5625,   53.59375,        -39.09375,
            96.9375,    -40.40625, 74.125,     0.0328369140625, 38.8125,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'min',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -36.0625,   -38.21875, 34.4375,    -16.609375,      34.4375,
            -17.765625, -76,       -69.5625,   34.4375,         -39.09375,
            34.4375,    -40.40625, 34.4375,    0.0328369140625, 34.4375,
            -17.71875,  17.390625, -82.3125,   -80.5,           -31.390625,
            -73.25,     -33.75,    -21.703125, 4.9453125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  }
];

if (navigator.ml) {
  minTests.forEach((test) => {
    webnn_conformance_test(
        buildAndExecuteGraph, getMinPrecisionTolerance, test);
  });
} else {
  test(() => assert_implements(navigator.ml, 'missing navigator.ml'));
}
