/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.eviction;

import org.jboss.cache.CacheSPI;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.Region;
import org.jboss.cache.RegionManager;
import org.jboss.cache.factories.UnitTestCacheConfigurationFactory;
import org.jboss.cache.config.EvictionPolicyConfig;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.lock.IsolationLevel;
import static org.testng.AssertJUnit.assertEquals;
import static org.testng.AssertJUnit.assertTrue;
import org.testng.annotations.Test;

import java.net.URL;

/**
 * Unit test to test Eviction configuration types.
 *
 * @author Daniel Huang (dhuang@jboss.org)
 * @version $Revision: 5238 $
 */
@Test(groups = {"functional"})
public class EvictionConfigurationTest
{
   CacheSPI<Object, Object> cache;
   RegionManager regionManager;

   public void testPolicyPerRegion() throws Exception
   {
      this.setupCache("META-INF/conf-test/policyPerRegion-eviction-service.xml");
      assertEquals(5, cache.getConfiguration().getEvictionConfig().getWakeupIntervalSeconds());

      Region region = regionManager.getRegion("/org/jboss/data", true);
      EvictionPolicy policy = region.getEvictionPolicy();
      EvictionPolicyConfig configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/org/jboss/data"), region.getFqn());
      assertTrue(policy instanceof LFUPolicy);
      assertTrue(configuration instanceof LFUConfiguration);
      assertEquals(5000, ((LFUConfiguration) configuration).getMaxNodes());
      assertEquals(1000, ((LFUConfiguration) configuration).getMinNodes());

      region = regionManager.getRegion("/org/jboss/test/data", true);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/org/jboss/test/data"), region.getFqn());
      assertTrue(policy instanceof FIFOPolicy);
      assertTrue(configuration instanceof FIFOConfiguration);
      assertEquals(5, ((FIFOConfiguration) configuration).getMaxNodes());

      region = regionManager.getRegion("/test", true);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/test"), region.getFqn());
      assertTrue(policy instanceof MRUPolicy);
      assertTrue(configuration instanceof MRUConfiguration);
      assertEquals(10000, ((MRUConfiguration) configuration).getMaxNodes());

      region = regionManager.getRegion("/maxAgeTest", true);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/maxAgeTest"), region.getFqn());
      assertTrue(policy instanceof LRUPolicy);
      assertTrue(configuration instanceof LRUConfiguration);
      assertEquals(10000, ((LRUConfiguration) configuration).getMaxNodes());
      assertEquals(8, ((LRUConfiguration) configuration).getTimeToLiveSeconds());
      assertEquals(10, ((LRUConfiguration) configuration).getMaxAgeSeconds());

      // test the default region. use a region name that isn't defined explicitly in conf file.
      region = regionManager.getRegion("/a/b/c", false);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.ROOT, region.getFqn());
      assertTrue(policy instanceof LRUPolicy);
      assertTrue(configuration instanceof LRUConfiguration);
      assertEquals(5000, ((LRUConfiguration) configuration).getMaxNodes());
      assertEquals(1000, ((LRUConfiguration) configuration).getTimeToLiveSeconds());
      assertEquals(0, ((LRUConfiguration) configuration).getMaxAgeSeconds());

      cache.stop();
   }

   public void testMixedPolicies() throws Exception
   {
      this.setupCache("META-INF/conf-test/mixedPolicy-eviction-service.xml");
      assertEquals(5, cache.getConfiguration().getEvictionConfig().getWakeupIntervalSeconds());

      Region region = regionManager.getRegion("/org/jboss/data", true);
      EvictionPolicy policy = region.getEvictionPolicy();
      EvictionPolicyConfig configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/org/jboss/data/"), region.getFqn());
      assertTrue(policy instanceof FIFOPolicy);
      assertTrue(configuration instanceof FIFOConfiguration);
      assertEquals(5000, ((FIFOConfiguration) configuration).getMaxNodes());

      region = regionManager.getRegion("/test", true);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/test/"), region.getFqn());
      assertTrue(policy instanceof MRUPolicy);
      assertTrue(configuration instanceof MRUConfiguration);
      assertEquals(10000, ((MRUConfiguration) configuration).getMaxNodes());

      // test the default region. use a region name that isn't defined explicitly in conf file.
      region = regionManager.getRegion("/a/b/c", false);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.ROOT, region.getFqn());
      assertTrue(policy instanceof LRUPolicy);
      assertTrue(configuration instanceof LRUConfiguration);
      assertEquals(5000, ((LRUConfiguration) configuration).getMaxNodes());
      assertEquals(1000, ((LRUConfiguration) configuration).getTimeToLiveSeconds());
      assertEquals(0, ((LRUConfiguration) configuration).getMaxAgeSeconds());

      region = regionManager.getRegion("/maxAgeTest", false);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/maxAgeTest/"), region.getFqn());
      assertTrue(policy instanceof LRUPolicy);
      assertTrue(configuration instanceof LRUConfiguration);
      assertEquals(10000, ((LRUConfiguration) configuration).getMaxNodes());
      assertEquals(8, ((LRUConfiguration) configuration).getTimeToLiveSeconds());
      assertEquals(10, ((LRUConfiguration) configuration).getMaxAgeSeconds());

      cache.stop();
   }

   public void testLegacyPolicyConfiguration() throws Exception
   {
      this.setupCache("META-INF/conf-test/local-lru-eviction-service.xml");
      assertEquals(5, cache.getConfiguration().getEvictionConfig().getWakeupIntervalSeconds());

      Region region = regionManager.getRegion("/org/jboss/data", false);
      EvictionPolicy policy = region.getEvictionPolicy();
      EvictionPolicyConfig configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/org/jboss/data/"), region.getFqn());
      assertTrue(policy instanceof LRUPolicy);
      assertTrue(configuration instanceof LRUConfiguration);
      assertEquals(5000, ((LRUConfiguration) configuration).getMaxNodes());
      assertEquals(1000, ((LRUConfiguration) configuration).getTimeToLiveSeconds());

      region = regionManager.getRegion("/org/jboss/test/data", false);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/org/jboss/test/data/"), region.getFqn());
      assertTrue(policy instanceof LRUPolicy);
      assertTrue(configuration instanceof LRUConfiguration);
      assertEquals(5, ((LRUConfiguration) configuration).getMaxNodes());
      assertEquals(4, ((LRUConfiguration) configuration).getTimeToLiveSeconds());

      region = regionManager.getRegion("/test", true);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/test/"), region.getFqn());
      assertTrue(policy instanceof LRUPolicy);
      assertTrue(configuration instanceof LRUConfiguration);
      assertEquals(10000, ((LRUConfiguration) configuration).getMaxNodes());
      assertEquals(4, ((LRUConfiguration) configuration).getTimeToLiveSeconds());

      region = regionManager.getRegion("/maxAgeTest", true);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.fromString("/maxAgeTest/"), region.getFqn());
      assertTrue(policy instanceof LRUPolicy);
      assertTrue(configuration instanceof LRUConfiguration);
      assertEquals(10000, ((LRUConfiguration) configuration).getMaxNodes());
      assertEquals(8, ((LRUConfiguration) configuration).getTimeToLiveSeconds());
      assertEquals(10, ((LRUConfiguration) configuration).getMaxAgeSeconds());

      // test the default region. use a region name that isn't defined explicitly in conf file.
      region = regionManager.getRegion("/a/b/c", false);
      policy = region.getEvictionPolicy();
      configuration = region.getEvictionPolicyConfig();
      assertEquals(Fqn.ROOT, region.getFqn());
      assertTrue(policy instanceof LRUPolicy);
      assertTrue(configuration instanceof LRUConfiguration);
      assertEquals(5000, ((LRUConfiguration) configuration).getMaxNodes());
      assertEquals(1000, ((LRUConfiguration) configuration).getTimeToLiveSeconds());
      assertEquals(0, ((LRUConfiguration) configuration).getMaxAgeSeconds());

      cache.stop();

   }

   public void testTwoCacheInstanceConfiguration() throws Exception
   {
      this.setupCache("META-INF/conf-test/local-lru-eviction-service.xml");
      this.setupCache("META-INF/conf-test/local-lru-eviction-service.xml");
   }

   public void testNoEviction() throws Exception
   {
      cache = (CacheSPI<Object, Object>) new DefaultCacheFactory<Object, Object>().createCache();
      regionManager = cache.getRegionManager();
      assertEquals(0, regionManager.getAllRegions(Region.Type.ANY).size());
   }

   private void setupCache(String configurationName)
   {
      cache = (CacheSPI<Object, Object>) new DefaultCacheFactory<Object, Object>().createCache(configurationName, false);
      cache.getConfiguration().setTransactionManagerLookupClass("org.jboss.cache.transaction.DummyTransactionManagerLookup");
      cache.getConfiguration().setIsolationLevel(IsolationLevel.SERIALIZABLE);
      cache.start();
      regionManager = cache.getRegionManager();
   }
}
