/*
 * @(#)TestClassParserUTest.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.junit.v1.parser;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import java.io.IOException;
import java.lang.reflect.Method;
import java.util.Vector;


/**
 * Tests the TestClassParser class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since     March 1, 2002
 * @version   $Date: 2003/05/24 17:15:47 $
 */
public class TestClassParserUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = TestClassParserUTest.class;
//    private static final IJUnitDocumentor LOG = (new JUnitLog(THIS_CLASS)).getDocumentor();
    
    public TestClassParserUTest( String name )
    {
        super( name );
    }

    


    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testConstructor1()
    {
        try
        {
            new TestClassParser( null );
        }
        catch (IllegalArgumentException e)
        {
            // test exception?
        }
    }
    
    
    public void testConstructor2()
    {
        new TestClassParser( getClass() );
    }
    
    
    public static class TesterNoTestMethods implements Test
    {
        public int countTestCases() { return 0; }
        public void run( junit.framework.TestResult tr ) {}
    }
    
    
    public static class TesterOneTestMethod implements Test
    {
        public int countTestCases() { return 0; }
        public void run( junit.framework.TestResult tr ) {}
        
        public void testMyTest() {}
    }


    
    
    public static class TesterBadMethod implements Test
    {
        public int countTestCases() { return 0; }
        public void run( junit.framework.TestResult tr ) {}
        
        public void usedForTesting()
        {
        }
        
        protected void testUsedForTesting()
        {}
    }

    
    
    private class StaticClass {}
    public class InnerClass {}
    
    public void testGetTestMethods1()
    {
        TestClassParser tcp = new TestClassParser( String.class );
        Method m[] = tcp.getTestMethods();
        assertNotNull(
            "Must never return null.",
            m );
        assertEquals(
            "String should have no test methods.",
            0,
            m.length );
        assertTrue(
            "Must never return the same array, but rather a copy.",
            m != tcp.getTestMethods() );
    }
    
    public void testGetTestMethods2()
    {
        TestClassParser tcp = new TestClassParser( Runnable.class );
        Method m[] = tcp.getTestMethods();
        assertNotNull(
            "Must never return null.",
            m );
        assertEquals(
            "Runnable should have no test methods.",
            0,
            m.length );
        assertTrue(
            "Must never return the same array, but rather a copy.",
            m != tcp.getTestMethods() );
    }
    
    public void testGetTestMethods3()
    {
        TestClassParser tcp = new TestClassParser( StaticClass.class );
        Method m[] = tcp.getTestMethods();
        assertNotNull(
            "Must never return null.",
            m );
        assertEquals(
            "Runnable should have no test methods.",
            0,
            m.length );
        assertTrue(
            "Must never return the same array, but rather a copy.",
            m != tcp.getTestMethods() );
    }
    
    public void testGetTestMethods4()
    {
        TestClassParser tcp = new TestClassParser( InnerClass.class );
        Method m[] = tcp.getTestMethods();
        assertNotNull(
            "Must never return null.",
            m );
        assertEquals(
            "Runnable should have no test methods.",
            0,
            m.length );
        assertTrue(
            "Must never return the same array, but rather a copy.",
            m != tcp.getTestMethods() );
    }
    
    public void testGetTestMethods5()
    {
        TestClassParser tcp = new TestClassParser( TesterNoTestMethods.class );
        Method m[] = tcp.getTestMethods();
        assertNotNull(
            "Must never return null.",
            m );
        assertEquals(
            "Runnable should have no test methods.",
            0,
            m.length );
        assertTrue(
            "Must never return the same array, but rather a copy.",
            m != tcp.getTestMethods() );
    }
    
    public void testGetTestMethods6()
    {
        TestClassParser tcp = new TestClassParser( TesterOneTestMethod.class );
        Method m[] = tcp.getTestMethods();
        assertNotNull(
            "Must never return null.",
            m );
        assertEquals(
            "Runnable should have one test method.",
            1,
            m.length );
        assertTrue(
            "Must never return the same array, but rather a copy.",
            m != tcp.getTestMethods() );
    }
    
    public void testClearWarnings1()
    {
        TestClassParser tcp = new TestClassParser( TesterOneTestMethod.class );
        tcp.clearWarnings();
    }
    
    public void testClearWarnings2()
    {
        TestClassParser tcp = new TestClassParser( Object.class );
        tcp.clearWarnings();
    }
    
    public void testGetName1()
    {
        TestClassParser tcp = new TestClassParser( TesterOneTestMethod.class );
        String name = tcp.getName();
        assertEquals(
            "Returned invalid test name.",
            TesterOneTestMethod.class.getName(),
            name );
    }
    
    public void testGetName2()
    {
        TestClassParser tcp = new TestClassParser( Object.class );
        String name = tcp.getName();
        assertEquals(
            "Returned invalid test name.",
            Object.class.getName(),
            name );
    }
    
    public void testAddTestMethod1() throws Exception
    {
        TestClassParser tcp = new TestClassParser( TesterOneTestMethod.class );
        int warningCount = tcp.getWarnings().length;
        Vector v = new Vector();
        Method m = THIS_CLASS.getMethod( "testAddTestMethod1", new Class[0] );
        v.addElement( "testAddTestMethod1" );
        tcp.addTestMethod( m, v );
        
        // ensure that there are more warnings
        assertEquals(
            "Incorrectly changed the warnings.",
            warningCount,
            tcp.getWarnings().length );
    }
    
    public void testAddTestMethod2() throws Exception
    {
        TestClassParser tcp = new TestClassParser( TesterBadMethod.class );
        int warningCount = tcp.getWarnings().length;
        Vector v = new Vector();
        Method m = TesterBadMethod.class.getMethod( "usedForTesting", new Class[0] );
        tcp.addTestMethod( m, v );
        
        // ensure that there are more warnings
        assertEquals(
            "Incorrectly changed the warnings.",
            warningCount,
            tcp.getWarnings().length );
    }
    
    /* this is tested elsewhere.
    public void testAddTestMethod3() throws Exception
    {
        TestClassParser tcp = new TestClassParser( TesterBadMethod.class );
        int warningCount = tcp.getWarnings().length;
        Vector v = new Vector();
        Method m = TesterBadMethod.class.getMethod( "testUsedForTesting", new Class[0] );
        tcp.addTestMethod( m, v );
        
        // ensure that there are more warnings
        assertEquals(
            "Did not add an additional warning.",
            warningCount + 1,
            tcp.getWarnings().length );
    }
    
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

