/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/gamma/cdf' );
import Gamma = require( './../../../../../base/dists/gamma/ctor' );
import entropy = require( './../../../../../base/dists/gamma/entropy' );
import kurtosis = require( './../../../../../base/dists/gamma/kurtosis' );
import logcdf = require( './../../../../../base/dists/gamma/logcdf' );
import logpdf = require( './../../../../../base/dists/gamma/logpdf' );
import mean = require( './../../../../../base/dists/gamma/mean' );
import mgf = require( './../../../../../base/dists/gamma/mgf' );
import mode = require( './../../../../../base/dists/gamma/mode' );
import pdf = require( './../../../../../base/dists/gamma/pdf' );
import quantile = require( './../../../../../base/dists/gamma/quantile' );
import skewness = require( './../../../../../base/dists/gamma/skewness' );
import stdev = require( './../../../../../base/dists/gamma/stdev' );
import variance = require( './../../../../../base/dists/gamma/variance' );

/**
* Interface describing the `gamma` namespace.
*/
interface Namespace {
	/**
	* Gamma distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 8.0, 3.0 );
	* // returns ~0.256
	*
	* y = ns.cdf( 0.0, 1.0, 1.0 );
	* // returns 0.0
	*
	* var mycdf = ns.cdf.factory( 2.0, 0.5 );
	* y = mycdf( 6.0 );
	* // returns ~0.801
	*
	* y = mycdf( 2.0 );
	* // returns ~0.264
	*/
	cdf: typeof cdf;

	/**
	* Gamma distribution.
	*/
	Gamma: typeof Gamma;

	/**
	* Returns the differential entropy of a gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns entropy
	*
	* @example
	* var v = ns.entropy( 1.0, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.entropy( 4.0, 12.0 );
	* // returns ~-0.462
	*
	* @example
	* var v = ns.entropy( 8.0, 2.0 );
	* // returns ~1.723
	*
	* @example
	* var v = ns.entropy( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.entropy( NaN, 2.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis of a gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns kurtosis
	*
	* @example
	* var v = ns.kurtosis( 1.0, 1.0 );
	* // returns 6.0
	*
	* @example
	* var v = ns.kurtosis( 4.0, 12.0 );
	* // returns 1.5
	*
	* @example
	* var v = ns.kurtosis( 8.0, 2.0 );
	* // returns 0.75
	*
	* @example
	* var v = ns.kurtosis( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Gamma distribution natural logarithm of the cumulative distribution function (logCDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 2.0, 0.5, 1.0 );
	* // returns ~-0.047
	*
	* var mylogcdf = ns.logcdf.factory( 3.0, 1.5 );
	* var y = mylogcdf( 1.0 );
	* // returns ~-1.655
	*/
	logcdf: typeof logcdf;

	/**
	* Gamma distribution natural logarithm of the probability density function (logPDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 0.5, 1.0 );
	* // returns ~-2.919
	*
	* var mylogpdf = ns.logpdf.factory( 6.0, 7.0 );
	* var y = mylogpdf( 2.0 );
	* // returns ~-3.646
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 1.0, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.mean( 4.0, 12.0 );
	* // returns ~0.333
	*
	* @example
	* var v = ns.mean( 8.0, 2.0 );
	* // returns 4.0
	*
	* @example
	* var v = ns.mean( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Gamma distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.5, 0.5, 1.0 );
	* // returns ~1.414
	*
	* y = ns.mgf( 0.1, 1.0, 1.0 );
	* // returns ~1.111
	*
	* y = ns.mgf( -1.0, 4.0, 2.0 );
	* // returns ~0.198
	*
	* var mymgf = ns.mgf.factory( 3.0, 1.5 );
	*
	* y = mymgf( 1.0 );
	* // returns ~26.999
	*
	* y = mymgf( 0.5 );
	* // returns ~3.375
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode of a gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns mode
	*
	* @example
	* var v = ns.mode( 1.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var v = ns.mode( 4.0, 12.0 );
	* // returns 0.25
	*
	* @example
	* var v = ns.mode( 8.0, 2.0 );
	* // returns 3.5
	*
	* @example
	* var v = ns.mode( 0.5, 2.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 2.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Gamma distribution probability density function (PDF).
	*
	* @param x - input value
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 0.5, 1.0 );
	* // returns ~0.054
	*
	* var myPDF = ns.pdf.factory( 6.0, 7.0 );
	* y = myPDF( 2.0 );
	* // returns ~0.026
	*/
	pdf: typeof pdf;

	/**
	* Gamma distribution quantile function.
	*
	* @param p - input value
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 1.0, 1.0 );
	* // returns ~1.609
	*
	* var myQuantile = ns.quantile.factory( 2.0, 2.0 );
	* y = myQuantile( 0.8 );
	* // returns ~1.497
	*
	* y = myQuantile( 0.4 );
	* // returns ~0.688
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 1.0, 1.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.skewness( 8.0, 2.0 );
	* // returns ~0.707
	*
	* @example
	* var v = ns.skewness( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 1.0, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0 );
	* // returns ~0.167
	*
	* @example
	* var v = ns.stdev( 8.0, 2.0 );
	* // returns ~1.414
	*
	* @example
	* var v = ns.stdev( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a gamma distribution.
	*
	* ## Notes
	*
	* -   If `alpha <= 0` or `beta <= 0`, the function returns `NaN`.
	*
	* @param alpha - shape parameter
	* @param beta - rate parameter
	* @returns variance
	*
	* @example
	* var v = ns.variance( 1.0, 1.0 );
	* // returns 1.0
	*
	* @example
	* var v = ns.variance( 4.0, 12.0 );
	* // returns ~0.028
	*
	* @example
	* var v = ns.variance( 8.0, 2.0 );
	* // returns 2.0
	*
	* @example
	* var v = ns.variance( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Gamma distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
