/* orafns.c
 *
 * Simple C interface to Oracle, intended to be linked to Perl.
 */
/* Copyright 1991, 1992, 1993 Kevin Stock.
 *
 * You may copy this under the terms of the GNU General Public License,
 * or the Artistic License, copies of which should have accompanied your
 * Perl kit.
 */

#include	"INTERN.h"
#include	<stdio.h>
#include	<ctype.h>
#include	"orafns.h"
#include	"patchlevel.h"


/* address[] is used to return cursor addresses to the perl program
 * it is used so that we can get the addresses exactly right, without
 * worrying about rounding errors or playing with oracle.mus
 */

char	address[20];


#ifdef	DEBUGGING
	/* table of known ORACLE datatypes */

	char	*types[] =
	{
	/* 00 */	NULL,
	/* 01 */	"character array",
	/* 02 */	"number",
	/* 03 */	"signed integer",
	/* 04 */	"float",
	/* 05 */	"null terminated string",
	/* 06 */	NULL,
	/* 07 */	"packed decimal",
	/* 08 */	"long string",
	/* 09 */	"varchar",
	/* 10 */	NULL,
	/* 11 */	"rowid",
	/* 12 */	"date",
	/* 13 */	NULL,
	/* 14 */	NULL,
	/* 15 */	"varraw",
	/* 16 */	NULL,
	/* 17 */	NULL,
	/* 18 */	NULL,
	/* 19 */	NULL,
	/* 20 */	NULL,
	/* 21 */	NULL,
	/* 22 */	NULL,
	/* 23 */	"raw",
	/* 24 */	"long raw",
	};
#endif	/* DEBUGGING */

/* oracle_sid is just used so that we don't keep repeating the string */

static char *oracle_sid		= "ORACLE_SID";
static char *two_task		= "TWO_TASK";


/* set_sid() uses my_setenv() to set ORACLE_SID to the required database.
 * It preserves the old value of ORACLE_SID so that it can be restored
 * (by calling set_sid() with NULL as its parameter.
 *
 * If db contains a colon, then TWO_TASK is used instead of ORACLE_SID.
 */

set_sid(db)
char *db;
{
	char *h, *var;
	static char *oldsid		= NULL;

	DBUG_ENTER("set_sid");
	DBUG_PRINT("entry", ("set_sid(%s)", db ? db : "<NULL>"));

	ora_err.no = 0;
	ora_err.lda = NULL;

	if (db == NULL)
	{
		if (oldsid != NULL)
		{
			var = (strchr(oldsid, ':') == NULL) ? oracle_sid
							     : two_task;

			DBUG_PRINT("info", ("resetting %s to %s", var, oldsid));
			my_setenv(var, oldsid);
			DBUG_RETURN(1);		/* assume it worked */
		}
		else
		{
			DBUG_PRINT("info", ("no value to reset %s/%s",
			    oracle_sid, two_task));
			DBUG_RETURN(1);
		}
	}
	else
	{
		if (oldsid != NULL)
		{
			/* already have a saved value - dispose of it */
			DBUG_PRINT("free", ("freeing oldsid %lx",(long)oldsid));
			free(oldsid);
		}

		var = (strchr(db, ':') == NULL) ? oracle_sid : two_task;

		if ((h = getenv(var)) == NULL)
		{
			/* no previous value to save */
			oldsid = NULL;
		}
		else
		{
			if ((oldsid = malloc(strlen(h) + 1)) == NULL)
			{
				DBUG_PRINT("malloc",
				    ("insufficient memory for oldsid"));
				ora_err.no = ORAP_NOMEM;
				DBUG_RETURN(0);
			}
			else
			{
				DBUG_PRINT("malloc",
				    ("got oldsid %d bytes at %#lx",
				     strlen(h) + 1, (long) oldsid));
				strcpy(oldsid, h);
			}
		}

		DBUG_PRINT("info", ("setting %s to %s", var, db));
		my_setenv(var, db);

		if (((h = getenv(var)) == NULL) || (strcmp(h, db) != 0))
		{
			DBUG_PRINT("info", ("%s misset to %s", var, h));
			DBUG_RETURN(0);
		}
	}

	DBUG_RETURN(1);
}


/* ora_login(database, name, password)
 *
 * logs into the current database under the given name and password.
 */

char *ora_login(database, name, password)
char *database, *name, *password;
{
	int logged;
	char *tmp;
	struct cursor *lda;

	DBUG_ENTER("ora_login");
	DBUG_PRINT("entry",
	    ("ora_login(\"%s\", \"%s\", \"%s\")", database, name, password));

	if ((lda = ora_getlda()) == NULL)
	{
		DBUG_PRINT("exit", ("couldn't get an lda, returning NULL"));
		DBUG_RETURN(NULL);
	}

	if (*database != '\0')
	{
		if (set_sid(database) == 0)
		{
			(void) ora_dropcursor(lda);
			ora_err.no = ORAP_NOSID;
			DBUG_PRINT("exit", ("couldn't set database ID"));
			DBUG_RETURN(NULL);
		}
	}
	/* if no database was specified, use $ORACLE_SID by default */

	logged = orlon(lda->csr, lda->hda, name, -1, password, -1, 0);

	if (*database != '\0')
	{
		set_sid(NULL);	/* reset ORACLE_SID to what it was before
				 * don't really care if this fails
				 */
	}

	if (logged == 0)
	{
		sprintf(address, "%#lx", (long) lda);
		DBUG_PRINT("conv", ("lda %#lx converted to string \"%s\"",
		    (long) lda, address));
		ora_err.no = 0;
		ora_err.lda = NULL;
		DBUG_PRINT("exit", ("returning lda %s", address));
		DBUG_RETURN(address);
	}
	else
	{
		ora_err.no = lda->csr->csrrc;
		ora_err.lda = NULL;	/* this ought to be lda, not NULL
					 * but as we're about to drop the
					 * lda, that wouldn't make sense
					 */
		(void) ora_droplda(lda);
		DBUG_PRINT("exit", ("orlon failed (error %d)", ora_err.no));
		DBUG_RETURN((char *) NULL);
	}
}


/* ora_open(lda, stmt, cache)
 *
 * sets and executes the specified sql statement
 */

char *ora_open(lda_s, stmt, cache)
char *lda_s;
char *stmt;
int cache;
{
	int i;
	struct cursor *csr;
	struct cursor *lda = (struct cursor *)strtoul(lda_s, (char **) NULL, 0);
	short dsize;

	DBUG_ENTER("ora_open");
	DBUG_PRINT("entry", ("ora_open(%s, \"%s\", %d)", lda_s, stmt, cache));
	DBUG_PRINT("conv", ("string \"%s\" converted to lda %#lx", lda_s, lda));

	if (check_lda(lda) == 0)
	{
		ora_err.no = ORAP_INVLDA;
		DBUG_PRINT("exit", ("invalid lda, returning NULL"));
		DBUG_RETURN((char *) NULL);
	}

	if ((csr = ora_getcursor()) == NULL)
	{
		DBUG_PRINT("exit", ("can't get a cursor, returning NULL"));
		DBUG_RETURN((char *) NULL);
	}
	csr->parent = lda;

	/* Check whether there are any substitution variables in the statement
	 * If there are, we don't execute the statement yet.
	 */
	if ((csr->varfields = count_colons(stmt)) < 0)
	{
		ora_err.no = ORAP_BADVAR;
		DBUG_PRINT("exit", ("invalid variable sequence"));
		DBUG_RETURN((char *) NULL);
	}
	DBUG_PRINT("info", ("statement contains %d colons", csr->varfields));

	/* maybe we should separate these statements so that a debugging
	 * trace can show exactly where any failure occurred
	 */
	if ((oopen(csr->csr, lda->csr, (char *)-1, -1, -1, (char *)-1, -1) != 0)
	    || (osql3(csr->csr, stmt, -1) != 0)
	    || ((csr->varfields == 0) && (oexec(csr->csr) != 0)))
	{
		ora_err.no = csr->csr->csrrc;
		ora_err.lda = lda;
		oclose(csr->csr);
		(void) ora_dropcursor(csr);
		DBUG_PRINT("exit",
		    ("couldn't run SQL statement (error %d)", ora_err.no));
		DBUG_RETURN((char *) NULL);
	}

	/* count the number of fields which will be returned */

	i = 0;
	do
	{
		odsc(csr->csr, ++i, (short *) 0, (short *) 0, (short *) 0,
			(short *) 0, (char *) 0, (short *) 0, (short *) 0);
	} while (csr->csr->csrrc == 0);
	--i;
	ora_err.no = 0;
	ora_err.lda = NULL;

	/* set up csr->data  to receive the information when we do a fetch
	 *	  csr->rcode to receive the column return codes
	 *	  csr->len   with the data lengths
	 */

	if (cache < 1)
	{
		DBUG_PRINT("info", ("cache size %d too small - set to %d",
		    cache, ora_cache));
		cache = ora_cache;
	}

	if (i > 0)
	{
		DBUG_PRINT("info", ("statement returns %d fields", i));

		if ((csr->data = (char **) malloc(i * sizeof(char *))) == NULL)
		{
			DBUG_PRINT("malloc", ("insufficient memory for data"));
			oclose(csr->csr);
			(void) ora_dropcursor(csr);
			ora_err.no = ORAP_NOMEM;
			DBUG_PRINT("exit", ("returning NULL"));
			DBUG_RETURN((char *) NULL);
		}
		DBUG_PRINT("malloc",
		    ("got data array %d items %d bytes at %#lx",
		     i, i * sizeof(char *), (long) csr->data));
		*csr->data = (char *) NULL;

		if ((csr->len = (short *) malloc(i * sizeof(short))) == NULL)
		{
			DBUG_PRINT("malloc", ("insufficient memory for len"));
			oclose(csr->csr);
			(void) ora_dropcursor(csr);
			ora_err.no = ORAP_NOMEM;
			DBUG_PRINT("exit", ("returning NULL"));
			DBUG_RETURN((char *) NULL);
		}
		DBUG_PRINT("malloc", ("got len array %d items %d bytes at %#lx",
		     i, i * sizeof(short), (long) csr->len));

		if ((csr->rcode = (short **) malloc(i*sizeof(short *))) == NULL)
		{
			DBUG_PRINT("malloc", ("insufficient memory for rcode"));
			oclose(csr->csr);
			(void) ora_dropcursor(csr);
			ora_err.no = ORAP_NOMEM;
			DBUG_PRINT("exit", ("returning NULL"));
			DBUG_RETURN((char *) NULL);
		}
		DBUG_PRINT("malloc",
		    ("got rcode array %d items %d bytes at %#lx",
		     i, i * sizeof(short *), (long) csr->rcode));

		if ((csr->type = (short *) malloc(i * sizeof(short))) == NULL)
		{
			DBUG_PRINT("malloc", ("insufficient memory for type"));
			oclose(csr->csr);
			(void) ora_dropcursor(csr);
			ora_err.no = ORAP_NOMEM;
			DBUG_PRINT("exit", ("returning NULL"));
			DBUG_RETURN((char *) NULL);
		}
		DBUG_PRINT("malloc",("got type array %d items %d bytes at %#lx",
		     i, i * sizeof(short), (long) csr->type));

		csr->nfields = i;

		for (i = 0 ; i < csr->nfields ; i++)
		{
			odsc(csr->csr, i + 1, (short *) 0, (short *) 0,
			    (short *) 0, &csr->type[i], (char *) 0,
			    (short *) 0, &dsize);

			if ((csr->type[i] == 8) || (csr->type[i] == 24))
			{
				/* either LONG or LONGRAW */
				if (dsize < ora_long)
				{
					dsize = ora_long;
				}
			}

			if ((csr->data[i] =
			    (char *) malloc((dsize + 1) * cache)) == NULL)
			{
				DBUG_PRINT("malloc",
				    ("insufficient memory for data[%d]", i));
				oclose(csr->csr);
				(void) ora_dropcursor(csr);
				ora_err.no = ORAP_NOMEM;
				DBUG_PRINT("exit", ("returning NULL"));
				DBUG_RETURN((char *) NULL);
			}
			DBUG_PRINT("malloc", ("got field %d, %d bytes at %#lx",
			     i, (dsize + 1) * cache, csr->data[i]));

			if ((csr->rcode[i] =
			    (short *) malloc(sizeof(short) * cache)) == NULL)
			{
				DBUG_PRINT("malloc",
				    ("insufficient memory for rcode[%d]", i));
				oclose(csr->csr);
				(void) ora_dropcursor(csr);
				ora_err.no = ORAP_NOMEM;
				DBUG_PRINT("exit", ("returning NULL"));
				DBUG_RETURN((char *) NULL);
			}
			DBUG_PRINT("malloc", ("got rcode %d, %d bytes at %#lx",
			     i, sizeof(short) * cache, csr->rcode[i]));

			odefin(csr->csr, i + 1, csr->data[i], dsize + 1, 5, 0,
				(short *) 0, (char *) 0, 0, 0, (short *) 0,
				csr->rcode[i]);
			csr->len[i] = dsize;

			DBUG_PRINT("info", ("Field %d, length %d, type %d (%s)",
			    i, dsize, csr->type[i], types[csr->type[i]]));
		}

		if (csr->nfields > ora_nfields)
		{
			if (ora_result != NULL)
			{
				DBUG_PRINT("free", ("freeing ora_result %lx",
				    (long) ora_result));
				free(ora_result);
			}

			if ((ora_result =
			    (char **) malloc(csr->nfields * (sizeof(char *))))
				== NULL)
			{
				ora_nfields = 0;
				DBUG_PRINT("malloc",
				    ("insufficient memory for ora_result"));
				oclose(csr->csr);
				(void) ora_dropcursor(csr);
				ora_err.no = ORAP_NOMEM;
				DBUG_PRINT("exit", ("returning NULL"));
				DBUG_RETURN((char *) NULL);
			}
			DBUG_PRINT("malloc", ("got ora_result %d bytes at %#lx",
			    csr->nfields * sizeof(char *), ora_result));
			ora_nfields = csr->nfields;
		}
	}
	else
	{
		DBUG_PRINT("info", ("statement returns no data"));
		csr->data = NULL;
	}

	csr->cache_size = cache;
	csr->in_cache = 0;
	csr->end_of_data = 0;

	sprintf(address, "%#lx", (long) csr);
	DBUG_PRINT("conv", ("csr %#lx converted to string \"%s\"",csr,address));
	DBUG_PRINT("exit", ("returning csr \"%s\"", address));
	DBUG_RETURN(address);
}


/* ora_titles(csr)
 *
 * returns the column headers for the query referenced by csr
 *
 * As of version 2, patch 2, the column titles are written into a locally
 * allocated data area. This allows the entire name to be returned without
 * truncation. If the truncate flag is set, however, then the title is
 * truncated to the length of the data, to allow compatibility with the
 * old form.
 */

int ora_titles(csr_s, truncate)
char *csr_s;
int truncate;
{
	int i;
	short len;
	struct cursor *csr = (struct cursor *)strtoul(csr_s, (char **) NULL, 0);

	static	int	 n_titles	= 0;
	static	char	*titles		= NULL;

	DBUG_ENTER("ora_titles");
	DBUG_PRINT("entry", ("ora_titles(%s)", csr_s));
	DBUG_PRINT("conv", ("string \"%s\" converted to csr %#lx", csr_s, csr));

	if (check_csr(csr) == 0)
	{
		ora_err.no = ORAP_INVCSR;
		DBUG_PRINT("exit", ("not a csr"));
		DBUG_RETURN(0);
	}
	else if (csr->nfields == 0)
	{
		ora_err.no = ORAP_NODATA;
		DBUG_PRINT("exit", ("nothing to return"));
		DBUG_RETURN(0);
	}
	else if ((ora_result == NULL) || (ora_nfields < csr->nfields))
	{
		ora_err.no = ORAP_NOMEM;
		DBUG_PRINT("exit", ("ora_result is not assigned correctly"));
		DBUG_RETURN(0);
	}

	if (n_titles < csr->nfields)
	{
		n_titles = csr->nfields;

		/* we use free/malloc rather than realloc because we don't
		 * need to preserve the contents of titles
		 */

		if (titles != NULL)
		{
			DBUG_PRINT("free", ("freeing titles %lx",(long)titles));
			free(titles);
		}

		/* according to the manual, the title is a maximum of 240
		 * characters. Here we allocate 256, which may lead to better
		 * blocking?
		 */

		if ((titles = (char *) malloc(256 * n_titles)) == NULL)
		{
			n_titles = 0;
			DBUG_PRINT("malloc",("insufficient memory for titles"));
			DBUG_PRINT("exit", ("returning 0"));
			DBUG_RETURN(0);
		}
		else
		{
			DBUG_PRINT("malloc", ("got titles %d bytes at %lx",
				(256 * n_titles), (long) titles));
		}
	}

	for (i = 0 ; i < csr->nfields ; i++)
	{
		len = (truncate) ? csr->len[i] : 256;
		oname(csr->csr, i + 1, (char *) -1, (short *) -1,
		      &titles[256 * i], &len);
		ora_result[i] = &titles[256 * i];
		ora_result[i][len] = '\0';
		DBUG_PRINT("info", ("field %4d (%lx) title \"%s\"",
		    i, (long) ora_result[i], ora_result[i]));
	}

	ora_err.no = 0;
	ora_err.lda = NULL;
	DBUG_PRINT("exit", ("returning %d items", csr->nfields));
	DBUG_RETURN(csr->nfields);
}

/* ora_lengths(csr)
 *
 * returns the return field lengths for the query referenced by csr
 *
 * overwrites the first position in each data array,
 * but this is acceptable since these values have always been returned.
 */

int ora_lengths(csr_s)
char *csr_s;
{
	int i;
	short len;
	struct cursor *csr = (struct cursor *)strtoul(csr_s, (char **) NULL, 0);

	DBUG_ENTER("ora_lengths");
	DBUG_PRINT("entry", ("ora_lengths(%s)", csr_s));
	DBUG_PRINT("conv", ("string \"%s\" converted to csr %#lx", csr_s, csr));

	if (check_csr(csr) == 0)
	{
		ora_err.no = ORAP_INVCSR;
		DBUG_PRINT("exit", ("not a csr"));
		DBUG_RETURN(0);
	}
	else if (csr->nfields == 0)
	{
		ora_err.no = ORAP_NODATA;
		DBUG_PRINT("exit", ("nothing to return"));
		DBUG_RETURN(0);
	}
	else if ((ora_result == NULL) || (ora_nfields < csr->nfields))
	{
		ora_err.no = ORAP_NOMEM;
		DBUG_PRINT("exit", ("ora_result is not assigned correctly"));
		DBUG_RETURN(0);
	}

	for (i = 0 ; i < csr->nfields ; i++)
	{
		/* write the data length into the first entry in the cache */
		sprintf(csr->data[i], "%d", csr->len[i]);
		ora_result[i] = csr->data[i];
		DBUG_PRINT("info", ("field %4d (%lx) length %s",
		    i, (long) csr->data[i], csr->data[i]));
	}

	ora_err.no = 0;
	ora_err.lda = csr->parent;
	DBUG_PRINT("exit", ("returning %d items", csr->nfields));
	DBUG_RETURN(csr->nfields);
}


/* ora_types(csr)
 *
 * returns the return field types for the query referenced by csr
 * see the OCI manuals to work out what the return values mean
 *
 * overwrites the first position in each data array,
 * but this is acceptable since these values have always been returned.
 */

int ora_types(csr_s)
char *csr_s;
{
	int i;
	struct cursor *csr = (struct cursor *)strtoul(csr_s, (char **) NULL, 0);

	DBUG_ENTER("ora_types");
	DBUG_PRINT("entry", ("ora_types(%s)", csr_s));
	DBUG_PRINT("conv", ("string \"%s\" converted to csr %#lx", csr_s, csr));

	if (check_csr(csr) == 0)
	{
		ora_err.no = ORAP_INVCSR;
		DBUG_PRINT("exit", ("not a csr"));
		DBUG_RETURN(0);
	}
	else if (csr->nfields == 0)
	{
		ora_err.no = ORAP_NODATA;
		DBUG_PRINT("exit", ("nothing to return"));
		DBUG_RETURN(0);
	}
	else if ((ora_result == NULL) || (ora_nfields < csr->nfields))
	{
		ora_err.no = ORAP_NOMEM;
		DBUG_PRINT("exit", ("ora_result is not assigned correctly"));
		DBUG_RETURN(0);
	}

	for (i = 0 ; i < csr->nfields ; i++)
	{
		/* write the type into the first cache entry */
		sprintf(csr->data[i], "%d", csr->type[i]);
		ora_result[i] = csr->data[i];
		DBUG_PRINT("info", ("field %4d (%lx) type %s (%s)",
		    i, (long) csr->data[i], csr->data[i],
		    types[csr->type[i]] ? types[csr->type[i]] : "unknown"));
	}

	ora_err.no = 0;
	ora_err.lda = NULL;
	DBUG_PRINT("exit", ("returning %d items", csr->nfields));
	DBUG_RETURN(csr->nfields);
}


/* ora_fetch(csr)
 *
 * returns the next set of data from the cursor
 *
 * at exit, the first entry must be returned or have already been returned,
 * otherwise it may be overwritten by ora_titles().
 */

int ora_fetch(csr_s, trunc)
char *csr_s;
int trunc;
{
	int i;
	struct cursor *csr = (struct cursor *)strtoul(csr_s, (char **) NULL, 0);

	DBUG_ENTER("ora_fetch");
	DBUG_PRINT("entry", ("ora_fetch(%s, %d)", csr_s, trunc));
	DBUG_PRINT("conv", ("string \"%s\" converted to csr %#lx", csr_s, csr));

	if (check_csr(csr) == 0)
	{
		ora_err.no = ORAP_INVCSR;
		DBUG_PRINT("exit", ("not a csr"));
		DBUG_RETURN(0);
	}
	else if (csr->nfields == 0)
	{
		ora_err.no = ORAP_NODATA;
		DBUG_PRINT("exit", ("no data to return"));
		DBUG_RETURN(0);
	}
	else if ((ora_result == NULL) || (ora_nfields < csr->nfields))
	{
		ora_err.no = ORAP_NOMEM;
		DBUG_PRINT("exit", ("ora_result is not assigned"));
		DBUG_RETURN(0);
	}

	if (csr->in_cache == 0)
	{
		unsigned long rowcount = csr->csr->csrrpc;

		if (csr->end_of_data)
		{
			ora_err.no = 0;
			ora_err.lda = csr->parent;
			DBUG_PRINT("exit", ("end of data"));
			DBUG_RETURN(0);
		}

		if (i = ofen(csr->csr, csr->cache_size) == 4)
		{
			DBUG_PRINT("info", ("setting end_of_data"));
			csr->end_of_data = 1;
		}

		csr->in_cache = csr->csr->csrrpc - rowcount;
		csr->next_entry = 0;

		DBUG_PRINT("info", ("fetched %d rows", csr->in_cache));

		if (csr->in_cache == 0)
		{
			if ((i == 1) && (csr->csr->csrrc == 4))
			{
				ora_err.no = 0;
				ora_err.lda = csr->parent;
				DBUG_PRINT("exit", ("end of data"));
				DBUG_RETURN(0);
			}
			else
			{
				ora_err.no = csr->csr->csrrc;
				ora_err.lda = csr->parent;
				DBUG_PRINT("exit",
				    ("ofen error (%d)", ora_err.no));
				DBUG_RETURN(0);
			}
		}
	}

	DBUG_PRINT("info", ("returning row %d from cache:", csr->next_entry));

	for (i = 0 ; i < csr->nfields ; i++)
	{
		ora_result[i] =
			&csr->data[i][csr->next_entry * (csr->len[i] + 1)];

		switch (csr->rcode[i][csr->next_entry])
		{
		case 0:		/* no problem */
			break;

		case 1405:
			ora_result[i] = NULL;
			break;

		case 1406:
			ora_err.no = 1406;
			ora_err.lda = csr->parent;

			if (trunc	&&
			    ((csr->type[i] == 8) || (csr->type[i] == 24)))
			{
			    /* truncation is allowed
			     * IF  the truncation flag is set
			     * AND the field type is LONG or LONGRAW
			     */
			    DBUG_PRINT("info", ("LONG%s field %d was truncated",
				((csr->type[i] == 8) ? "" : "RAW"), i));
			}
			else	/* truncation is an error */
			{
			    DBUG_PRINT("exit", ("field %d truncation error",i));
			    DBUG_RETURN(0);
			}
			break;

		default:	/* others should not happen */
			DBUG_PRINT("info", ("ofetch error %d, field %d",
			    csr->rcode[i][csr->next_entry], i));
			ora_err.no = csr->csr->csrrc;
			ora_err.lda = csr->parent;
			DBUG_PRINT("exit", ("returning 0"));
			DBUG_RETURN(0);
		}

		DBUG_PRINT("info", ("field %4d (%lx) data \"%s\"",
		    i, (long) ora_result[i],
		    ora_result[i] ? ora_result[i] : "<NULL>"));
	}

	++csr->next_entry;
	--csr->in_cache;

	ora_err.no = 0;
	ora_err.lda = csr->parent;
	DBUG_PRINT("exit", ("returning %d items", csr->nfields));
	DBUG_RETURN(csr->nfields);
}

/* ora_bind(csr_s, vars, nitems)
 *
 * binds actual values to the SQL statement associated with csr
 */

long ora_bind(csr_s, vars, nitems)
char *csr_s, **vars;
int nitems;
{
	int i;
	short null_flag = -1;
	long rowcount;
#ifndef	NO_BIND_PADDING
	static char small_buf[2] = " ";
#endif
	struct cursor *csr = (struct cursor *)strtoul(csr_s, (char **) NULL, 0);

	DBUG_ENTER("ora_bind");
	DBUG_PRINT("entry", ("ora_bind(%s, %#lx, %d)",
	    csr_s, (long) vars, nitems));
	DBUG_PRINT("conv", ("string \"%s\" converted to csr %#lx", csr_s, csr));

	if (check_csr(csr) == 0)
	{
		ora_err.no = ORAP_INVCSR;
		DBUG_PRINT("exit", ("not a csr"));
		DBUG_RETURN(-1L);
	}
	else if (csr->varfields != nitems)
	{
		ora_err.no = ORAP_NUMVARS;
		DBUG_PRINT("exit", ("expected %d items, got %d",
		    csr->varfields, nitems));
		DBUG_RETURN(-1L);
	}

	for (i = 0 ; i < nitems ; i++)
	{
	    if (vars[i] == NULL)
	    {
		if ((obndrn(csr->csr, i+1, (char *) -1, 0,
			5, -1, &null_flag, (char *) -1, 0, 0)) != 0)
		{
			ora_err.no = csr->csr->csrrc;
			ora_err.lda = csr->parent;
			DBUG_PRINT("exit", ("obndrn failed on field %d, <NULL>",
			    i + 1));
			DBUG_RETURN(-1L);
		}

		DBUG_PRINT("info", ("obndrn %d, <NULL> OK", (i + 1), vars[i]));
	    }
	    else
	    {
#ifndef	NO_BIND_PADDING
		if (vars[i][0] == '\0')
		{
			DBUG_PRINT("info", ("field %d is empty - padding", i));
			vars[i] = small_buf;
		}
#endif
		if ((obndrn(csr->csr, i+1, vars[i], strlen(vars[i])+1,
			5, -1, (short *) -1, (char *) -1, 0, 0)) != 0)
		{
			ora_err.no = csr->csr->csrrc;
			ora_err.lda = csr->parent;
			DBUG_PRINT("exit", ("obndrn failed on field %d, \"%s\"",
			    i + 1, vars[i]));
			DBUG_RETURN(-1L);
		}

		DBUG_PRINT("info", ("obndrn %d, \"%s\" OK", (i + 1), vars[i]));
	    }
	}

	if (oexec(csr->csr) != 0)
	{
		ora_err.no = csr->csr->csrrc;
		ora_err.lda = csr->parent;
		DBUG_PRINT("exit", ("oexec failed"));
		DBUG_RETURN(-1L);
	}

	/* any cached data is now out of date, as is the end_of data flag */
	csr->in_cache = 0;
	csr->end_of_data = 0;

	rowcount = csr->csr->csrrpc;
	DBUG_PRINT("info", ("%ld rows processed", rowcount));

	DBUG_PRINT("exit", ("returning %ld", rowcount));
	DBUG_RETURN(rowcount);
}


char	*OK	= "OK";		/* valid return from ora_close/do/logoff */


/* ora_do(lda, stmt)
 *
 * sets and executes the specified sql statement, without leaving a cursor open
 */

long ora_do(lda_s, stmt)
char *lda_s;
char *stmt;
{
	long rowcount;
	char *csr_s;
	struct cursor *csr;

	DBUG_ENTER("ora_do");
	DBUG_PRINT("entry", ("ora_do(%s, \"%s\")", lda_s, stmt));

	if ((csr_s = ora_open(lda_s, stmt)) == NULL)
	{
		DBUG_PRINT("exit", ("ora_open failed - returning -1"));
		DBUG_RETURN(-1L);
	}

	csr = (struct cursor *) strtoul(csr_s, (char **) NULL, 0);
	DBUG_PRINT("conv", ("string %s converted to address $#lx",
		csr_s, (long) csr));

	rowcount = csr->csr->csrrpc;
	DBUG_PRINT("info", ("%ld rows processed", rowcount));

	if (ora_close(csr_s) == NULL)
	{
		ora_dropcursor(csr);
		DBUG_PRINT("exit", ("ora_close failed - returning -1"));
		DBUG_RETURN(-1L);
	}
	else
	{
		DBUG_PRINT("exit", ("returning %ld", rowcount));
		DBUG_RETURN(rowcount);
	}

	/* NOTREACHED */
}


/* ora_close(csr)
 *
 * Closes an oracle statement, releasing resources
 */

char *ora_close(csr_s)
char *csr_s;
{
	struct cursor *csr = (struct cursor *)strtoul(csr_s, (char **) NULL, 0);

	DBUG_ENTER("ora_close");
	DBUG_PRINT("entry", ("ora_close(%s)", csr_s));
	DBUG_PRINT("conv", ("string \"%s\" converted to csr %#lx", csr_s, csr));

	if (check_csr(csr) == 0)
	{
		ora_err.no = ORAP_INVCSR;
		DBUG_PRINT("exit", ("not a csr"));
		DBUG_RETURN(NULL);
	}
	else if (oclose(csr->csr) != 0)
	{
		ora_err.no = csr->csr->csrrc;
		ora_err.lda = csr->parent;
		DBUG_PRINT("exit", ("oclose failed"));
		DBUG_RETURN(NULL);
	}
	else
	{
		(void) ora_dropcursor(csr);
		DBUG_PRINT("exit", ("returning OK"));
		DBUG_RETURN(OK);
	}
}


/* ora_logoff(lda)
 *
 * Logs the user off of Oracle, releasing all resources
 */

char *ora_logoff(lda_s)
char *lda_s;
{
	struct cursor *lda = (struct cursor *)strtoul(lda_s, (char **) NULL, 0);

	DBUG_ENTER("ora_logoff");
	DBUG_PRINT("entry", ("ora_logoff(%s)", lda_s));
	DBUG_PRINT("conv", ("string \"%s\" converted to lda %#lx", lda_s, lda));

	if (check_lda(lda) == 0)
	{
		ora_err.no = ORAP_INVLDA;
		DBUG_PRINT("exit", ("not an lda"));
		DBUG_RETURN(NULL);
	}
	else if (ologof(lda->csr) != 0)
	{
		ora_err.no = lda->csr->csrrc;
		ora_err.lda = lda;
		DBUG_PRINT("exit", ("ologof failed, error code %d",ora_err.no));
		DBUG_RETURN(NULL);
	}
	else
	{
		(void) ora_droplda(lda);
		DBUG_PRINT("exit", ("returning OK"));
		DBUG_RETURN(OK);
	}
}


/* ora_commit(lda)
 *
 * Commits all pending transactions on the specified lda.
 */

char *ora_commit(lda_s)
char *lda_s;
{
	struct cursor *lda = (struct cursor *)strtoul(lda_s, (char **) NULL, 0);

	DBUG_ENTER("ora_commit");
	DBUG_PRINT("entry", ("ora_commit(%s)", lda_s));
	DBUG_PRINT("conv", ("string \"%s\" converted to lda %#lx", lda_s, lda));

	if (check_lda(lda) == 0)
	{
		ora_err.no = ORAP_INVLDA;
		DBUG_PRINT("exit", ("not an lda"));
		DBUG_RETURN(NULL);
	}
	else if (ocom(lda->csr) != 0)
	{
		ora_err.no = lda->csr->csrrc;
		ora_err.lda = lda;
		DBUG_PRINT("exit", ("ocom failed, error code %d", ora_err.no));
		DBUG_RETURN(NULL);
	}
	else
	{
		DBUG_PRINT("exit", ("returning OK"));
		DBUG_RETURN(OK);
	}
}


/* ora_rollback(lda)
 *
 * rollbacks all pending transactions on the specified lda.
 */

char *ora_rollback(lda_s)
char *lda_s;
{
	struct cursor *lda = (struct cursor *)strtoul(lda_s, (char **) NULL, 0);

	DBUG_ENTER("ora_rollback");
	DBUG_PRINT("entry", ("ora_rollback(%s)", lda_s));
	DBUG_PRINT("conv", ("string \"%s\" converted to lda %#lx", lda_s, lda));

	if (check_lda(lda) == 0)
	{
		ora_err.no = ORAP_INVLDA;
		DBUG_PRINT("exit", ("not an lda", stderr));
		DBUG_RETURN(NULL);
	}
	else if (orol(lda->csr) != 0)
	{
		ora_err.no = lda->csr->csrrc;
		ora_err.lda = lda;
		DBUG_PRINT("exit", ("orol failed, error code %d", ora_err.no));
		DBUG_RETURN(NULL);
	}
	else
	{
		DBUG_PRINT("exit", ("returning OK"));
		DBUG_RETURN(OK);
	}
}


/* ora_autocommit(lda, on_off)
 *
 * Turns autocommit for the specified lda on or off.
 */

char *ora_autocommit(lda_s, on_off)
char *lda_s;
int on_off;
{
	struct cursor *lda = (struct cursor *)strtoul(lda_s, (char **) NULL, 0);

	DBUG_ENTER("ora_autocommit");
	DBUG_PRINT("entry", ("ora_autocommit(%s, %d)", lda_s, on_off));
	DBUG_PRINT("conv", ("string \"%s\" converted to lda %#lx", lda_s, lda));

	if (check_lda(lda) == 0)
	{
		ora_err.no = ORAP_INVLDA;
		DBUG_PRINT("exit", ("not an lda", stderr));
		DBUG_RETURN(NULL);
	}

	if (on_off)	/* turn autocommit on */
	{
		if (ocon(lda->csr) != 0)
		{
			ora_err.no = lda->csr->csrrc;
			ora_err.lda = lda;
			DBUG_PRINT("exit",
				("ocon failed, error code %d", ora_err.no));
			DBUG_RETURN(NULL);
		}
	}
	else		/* turn autocommit off */
	{
		if (ocof(lda->csr) != 0)
		{
			ora_err.no = lda->csr->csrrc;
			ora_err.lda = lda;
			DBUG_PRINT("exit",
				("ocof failed, error code %d", ora_err.no));
			DBUG_RETURN(NULL);
		}
	}

	DBUG_PRINT("exit", ("returning OK"));
	DBUG_RETURN(OK);
}
