/*
 * planar figures
 */

#include <InterViews/brush.h>
#include <InterViews/canvas.h>
#include <InterViews/color.h>
#include <InterViews/event.h>
#include <InterViews/hit.h>
#include <InterViews/listener.h>

#include <OS/math.h>
#include <stream.h>

#include "figure.h"
#include "clip.h"

/*
 * FigureTool
 */

class FigureTool : public Glyph {
public:
  FigureTool(
       const Brush* brush, const Color* stroke, const Color* fill,
       boolean closed, boolean curved, int coords);
  ~FigureTool ();

  void brush(const Brush* b);
  void stroke(const Color* stroke);
  void fill(const Color* fill);

  const Color* stroke();
  
  void request(Requisition&) const;
  void allocate(Canvas*, const Allocation&, Extension&);
  void draw(Canvas*, const Allocation&) const;
  void pick(Canvas*, const Allocation&, int depth, Hit&);

  void clear_points();
  void point(int index, Coord&x, Coord& y);
  void add_point (Coord x, Coord y);
  void add_curve (Coord x, Coord y, Coord x1, Coord y1, Coord x2, Coord y2);
  void Bspline_move_to (
       Coord x, Coord y, Coord x1, Coord y1, Coord x2, Coord y2
  );
  void Bspline_curve_to (
       Coord x, Coord y, Coord x1, Coord y1, Coord x2, Coord y2
   );

private:

  Coord* _x;
  Coord* _y;

  const Brush* _brush;
  const Color* _stroke;
  const Color* _fill;
  
  boolean _closed;
  boolean _curved;
  int _count;
  
  Coord _xmin;
  Coord _xmax;
  Coord _ymin;
  Coord _ymax;
};

FigureTool::FigureTool(
	const Brush* brush, const Color* stroke, const Color* fill,
	boolean closed, boolean curved, int coords)
     : Glyph()
{
  Resource::ref(_brush = brush);
  Resource::ref(_stroke = stroke);
  Resource::ref(_fill = fill);

  _closed = closed;
  _curved = curved;
  _count = 0;
  _x = new Coord[coords];
  _y = new Coord[coords];

}

FigureTool::~FigureTool () 
{
//  printf("FigureTool::~FigureTool\n");

    Resource::unref(_brush);
    Resource::unref(_stroke);
    Resource::unref(_fill);
    delete _x;
    delete _y;

}

void FigureTool::brush(const Brush* brush)
{
  Resource::unref(_brush);
  Resource::ref(_brush = brush);
}

void FigureTool::stroke(const Color* stroke)
{
  Resource::unref(_stroke);
  Resource::ref(_stroke = stroke);
}

const Color* FigureTool::stroke()
{
  return(_stroke);
}

void FigureTool::fill(const Color* fill)
{
  Resource::unref(_fill);
  Resource::ref(_fill = fill);
}

void FigureTool::request(Requisition& requisition) const {
    if (_count > 0) {
        Requirement rx(-_xmin, -_xmin, -_xmin, _xmax, _xmax, _xmax);
        Requirement ry(-_ymin, -_ymin, -_ymin, _ymax, _ymax, _ymax);
        requisition.require(Dimension_X, rx);
        requisition.require(Dimension_Y, ry);
    }
}

void FigureTool::allocate(Canvas*, const Allocation& a, Extension& ext) {
    if (_count > 0) {
        Coord w = _brush == nil ? 0 : _brush->width();
        Coord x = a.x();
        Coord y = a.y();
	ext.xy_extents(
	    x + _xmin - w, x + _xmax + w,
	    y + _ymin - w, y + _ymax + w
	);
    }
}

void FigureTool::draw(Canvas* c, const Allocation& allocation) const {
    if (c != nil && _count > 0) {
        Coord x = allocation.x();
        Coord y = allocation.y();
        c->new_path();
        c->move_to(x + _x[0], y + _y[0]);
        if (_curved) {
            for (int i = 1; i < _count; i += 3) {
                c->curve_to(
                    x + _x[i + 2], y + _y[i + 2],
                    x + _x[i], y + _y[i],
                    x + _x[i + 1], y + _y[i + 1]
                );
            }
        } else {
            for (int i = 1; i < _count; ++i) {
                c->line_to(x + _x[i], y + _y[i]);
            }
        }
        if (_closed) {
            c->close_path();
        }
        if (_fill != nil) {
	    c->fill(_fill);
	}
	if (_brush != nil && _stroke != nil) {
	    c->stroke(_stroke, _brush);
	}
    }
}

void FigureTool::pick(Canvas*, const Allocation& a, int depth, Hit& h)
{
  int i;

  Coord x1 = h.left() - a.x();
  Coord y1 = h.bottom() - a.y();
  Coord x2 = h.right() - a.x();
  Coord y2 = h.top() - a.y();

//  printf("FigureTool::pick()\n");

  if (_closed) {
    if (poly_clip(x1, y1, x2, y2,
		  _x, _y, _count)) {
//      printf("\picked\n");
      h.target(depth, this, 0);
    }
  } else {
    for (i = 0 ; i < _count-1 ; ++i)
      if (line_clip(x1, y1, x2, y2,
		    _x[i], _y[i], _x[i+1], _y[i+1])) {
//	printf("\picked\n");
	h.target(depth, this, 0);
	return;
      }
  }
}

void FigureTool::clear_points()
{
  _count = 0;
  delete _x;
  delete _y;
}

void FigureTool::point(int index, Coord&x, Coord& y)
{
  x = _x[index];
  y = _y[index];
}

void FigureTool::add_point(Coord x, Coord y) {
    if (_count == 0) {
        _xmin = x - 1;
        _xmax = x + 1;
        _ymin = y - 1;
        _ymax = y + 1;
    } else {
        _xmin = Math::min(_xmin, x);
        _xmax = Math::max(_xmax, x);
        _ymin = Math::min(_ymin, y);
        _ymax = Math::max(_ymax, y);
    }
    _x[_count] = x;
    _y[_count] = y;
    _count += 1;
}

void FigureTool::add_curve(
    Coord x, Coord y, Coord x1, Coord y1, Coord x2, Coord y2
) {
    add_point(x1, y1);
    add_point(x2, y2);
    add_point(x, y);
}

void FigureTool::Bspline_move_to (
    Coord x, Coord y, Coord x1, Coord y1, Coord x2, Coord y2
) {
    Coord p1x = (x + x + x1) / 3;
    Coord p1y = (y + y + y1) / 3;
    Coord p2x = (x + x + x2) / 3;
    Coord p2y = (y + y + y2) / 3;
    add_point((p1x + p2x) / 2, (p1y + p2y) / 2);
}

void FigureTool::Bspline_curve_to (
    Coord x, Coord y, Coord x1, Coord y1, Coord x2, Coord y2
) {
    Coord p1x = (x + x + x1) / 3;
    Coord p1y = (y + y + y1) / 3;
    Coord p2x = (x + x + x2) / 3;
    Coord p2y = (y + y + y2) / 3;
    Coord p3x = (x1 + x1 + x) / 3;
    Coord p3y = (y1 + y1 + y) / 3;
    add_curve((p1x + p2x) / 2, (p1y + p2y) / 2, p3x, p3y, p1x, p1y);
}

/*
 * Figure
 */

// A listener ref's it's handler which increase the refence count
// of the object being passed by one.
// This creates a circular depenency in that the listener in the figure
// references the Figure, so the Figure does not behave as resources
// should and probably will never be deleted.
// For this reason don't reference figures ones they are created!

Figure::Figure(
       const Brush* brush, const Color* stroke, const Color* fill,
       boolean closed, boolean curved, int coords)
     : Patch(nil), PointerHandler()
{
  _tool = new FigureTool(brush, stroke, fill, closed, curved, coords);

  _listener = new Listener(_tool, this);
  body(_listener);

}

Figure::~Figure()
{
  Listener* l = _listener;
//  printf("Figure::~Figure\n");

  // The reference count for the figure is now 0.  Since the handler
  // of _listener is this Figure, it will unref it and cause this 
  // Figure to be deleted twice.
  // We need to increase the ref count of this Figure by two so that
  // _listener does not delete the figure twice, causing a core dump.

  ref(this);
  ref(this);
}

void Figure::press(Event& e) 
{
//  printf("Figure::press\n");
//  printf("\tx=%f y=%f\n", e.pointer_x(), e.pointer_y());

  if (_listener->picks(e.pointer_x(), e.pointer_y())) 
    printf("\tselected\n");
}

void Figure::brush(const Brush* brush)
{
  _tool->brush(brush);
}

void Figure::stroke(const Color* stroke)
{
  _tool->stroke(stroke);
}

const Color* Figure::stroke()
{ 
  return(_tool->stroke());
}

void Figure::fill(const Color* fill)
{
  _tool->fill(fill);
}

void Figure::button(boolean b, EventButton button)
   { _listener->button(b, button); }

void Figure::motion(boolean b)
   { _listener->motion(b); }

void Figure::key(boolean b)
   { _listener->key(b); }

void Figure::clear_points() 
   { _tool->clear_points(); }

void Figure::add_point (Coord x, Coord y)
   { _tool->add_point(x, y); }

void Figure::point(int index, Coord& x, Coord& y)
   { _tool->point(index, x, y); }

/*
 * Point
 */

Point::Point(const Brush* brush, const Color* stroke, const Color* fill,
	     Coord x, Coord y)
       : Figure(brush, stroke, fill, false, false, 1)
{
  add_point(x, y);
}

Point::~Point() { }

/*
 * Line
 */

Line::Line(const Brush* brush, const Color* stroke, const Color* fill,
	   Coord x1, Coord y1, Coord x2, Coord y2)
	: Figure(brush, stroke, fill, false, false, 2)
{
    add_point(x1, y1);
    add_point(x2, y2);
}

Line::~Line () { }

/*
 * Rectangle
 */

Rectangle::Rectangle (const Brush* brush, const Color* stroke, const Color* fill,
		      Coord l, Coord b, Coord r, Coord t)
	: Figure(brush, stroke, fill, true, false, 4)
{
    add_point(l, b);
    add_point(l, t);
    add_point(r, t);
    add_point(r, b);
}

Rectangle::~Rectangle ()
{
//  printf("Rectangle::~Rectangle\n");
}

/*
 * Circle
 */

static float p0 = 1.01422100;
static float p1 = 0.90932667;
static float p2 = 0.70710678;
static float p3 = 0.52500000;
static float p4 = 0.27176000;

Circle::Circle (const Brush* brush, const Color* stroke, const Color* fill,
		Coord x, Coord y, Coord r)
	: Figure(brush, stroke, fill, true, true, 25)
{
    float px0 = p0 * r, py0 = p0 * r;
    float px1 = p1 * r, py1 = p1 * r;
    float px2 = p2 * r, py2 = p2 * r;
    float px3 = p3 * r, py3 = p3 * r;
    float px4 = p4 * r, py4 = p4 * r;
    
    add_point(x + r, y);
    _tool->add_curve(x + px2, y + py2, x + px0, y + py4, x + px1, y + py3);
    _tool->add_curve(x, y + r, x + px3, y + py1, x + px4, y + py0);
    _tool->add_curve(x - px2, y + py2, x - px4, y + py0, x - px3, y + py1);
    _tool->add_curve(x - r, y, x - px1, y + py3, x - px0, y + py4);
    _tool->add_curve(x - px2, y - py2, x - px0, y - py4, x - px1, y - py3);
    _tool->add_curve(x, y - r, x - px3, y - py1, x - px4, y - py0);
    _tool->add_curve(x + px2, y - py2, x + px4, y - py0, x + px3, y - py1);
    _tool->add_curve(x + r, y, x + px1, y - py3, x + px0, y - py4);
}

Circle::~Circle () { }

/*
 * Ellipse
 */

Ellipse::Ellipse (const Brush* brush, const Color* stroke, const Color* fill,
		  Coord x, Coord y, Coord rx, Coord ry)
	: Figure(brush, stroke, fill, true, true, 25)
{
    float px0 = p0 * rx, py0 = p0 * ry;
    float px1 = p1 * rx, py1 = p1 * ry;
    float px2 = p2 * rx, py2 = p2 * ry;
    float px3 = p3 * rx, py3 = p3 * ry;
    float px4 = p4 * rx, py4 = p4 * ry;
    
    add_point(x + rx, y);
    _tool->add_curve(x + px2, y + py2, x + px0, y + py4, x + px1, y + py3);
    _tool->add_curve(x, y + ry, x + px3, y + py1, x + px4, y + py0);
    _tool->add_curve(x - px2, y + py2, x - px4, y + py0, x - px3, y + py1);
    _tool->add_curve(x - rx, y, x - px1, y + py3, x - px0, y + py4);
    _tool->add_curve(x - px2, y - py2, x - px0, y - py4, x - px1, y - py3);
    _tool->add_curve(x, y - ry, x - px3, y - py1, x - px4, y - py0);
    _tool->add_curve(x + px2, y - py2, x + px4, y - py0, x + px3, y - py1);
    _tool->add_curve(x + rx, y, x + px1, y - py3, x + px0, y - py4);
}

Ellipse::~Ellipse () { }

/*
 * Open_BSpline
 */

Open_BSpline::Open_BSpline (const Brush* brush, const Color* stroke, const Color* fill,
			    Coord* x, Coord* y, int n)
	: Figure(brush, stroke, fill, false, true, (n + 2) * 3 + 1)
{
    _tool->Bspline_move_to(x[0], y[0], x[0], y[0], x[0], y[0]);
    _tool->Bspline_curve_to(x[0], y[0], x[0], y[0], x[1], y[1]);
    for (int i = 1; i < n - 1; ++i) {
        _tool->Bspline_curve_to(x[i], y[i], x[i-1], y[i-1], x[i+1], y[i+1]);
    }
    _tool->Bspline_curve_to(x[n-1], y[n-1], x[n-2], y[n-2], x[n-1], y[n-1]);
    _tool->Bspline_curve_to(x[n-1], y[n-1], x[n-1], y[n-1], x[n-1], y[n-1]);
}

Open_BSpline::~Open_BSpline () { }

/*
 * Close_BSpline
 */

Closed_BSpline::Closed_BSpline (const Brush* brush, const Color* stroke, const Color* fill,
				Coord* x, Coord* y, int n)
	: Figure(brush, stroke, fill, true, true, n * 3 + 1)
{
  _tool->Bspline_move_to(x[0], y[0], x[n-1], y[n-1], x[1], y[1]);
  for (int i = 1; i < n - 1; ++i) {
    _tool->Bspline_curve_to(x[i], y[i], x[i-1], y[i-1], x[i+1], y[i+1]);
  }
  _tool->Bspline_curve_to(x[n-1], y[n-1], x[n-2], y[n-2], x[0], y[0]);
  _tool->Bspline_curve_to(x[0], y[0], x[n-1], y[n-1], x[1], y[1]);
}

Closed_BSpline::~Closed_BSpline () { }

/*
 * Polyline
 */

Polyline::Polyline (const Brush* brush, const Color* stroke, const Color* fill,
		    Coord* x, Coord* y, int n)
	: Figure(brush, stroke, fill, false, false, n)
{
  add_point(x[0], y[0]);
  for (int i = 1; i < n; ++i) {
    add_point(x[i], y[i]);
  }
}

Polyline::~Polyline () { }

/*
 * Polygon
 */

Polygon::Polygon (const Brush* brush, const Color* stroke, const Color* fill,
		  Coord* x, Coord* y, int n)
	: Figure(brush, stroke, fill, true, false, n)
{
  add_point(x[0], y[0]);
  for (int i = 1; i < n; ++i) {
    add_point(x[i], y[i]);
  }
}

Polygon::~Polygon () { }
