/* File: color_scale.c
 * Created: 4/28/92 by John Butare(jb7p@andrew.cmu.edu)
 * Description:
 *	
 * Modifications:
 */

#include <OS/string.h>

#include <InterViews/box.h>
#include <InterViews/style.h>
#include <InterViews/glue.h>
#include <InterViews/background.h>

#include <IV-look/kit.h>

#include <stdio.h>

#include "color_scale.h"
#include "gems.h"

/* 
 * Cubic Spline
 */

class CubicSpline {
public:
  CubicSpline(int n, float* t, float* y);
  ~CubicSpline();
  float f(float x);
  
private:
  int _n;

  float* _z;
  float* _t;
  float* _y;
};


/*
 * from PAGE 277-278: NUMERICAL MATHEMATICS AND COMPUTING, CHENEY/KINCAID, 1985
 *
 */

CubicSpline::CubicSpline(int n, float* t, float* y)
{
  _z = new float[n+1];
  _y = y;
  _t = t;
  _n  = n;

  float* v = new float[n+1];
  float* b = new float[n+1];
  float* h = new float[n+1];
  float* u = new float[n+1];

  int i;

  for (i = 1 ; i <= n-1 ; ++i) {
    h[i] = t[i+1] - t[i];
    b[i] = (y[i+1] -y[i])/h[i];
  }

  u[2] = 2.0*(h[1] + h[2]);
  v[2] = 6.0*(b[2] - b[1]);

  for (i = 3 ; i <= n-1 ; ++i) {
    u[i] = 2.0*(h[i] + h[i-1]) - h[i-1]*h[i-1]/u[i-1];
    v[i] = 6.0*(b[i] - b[i-1]) - h[i-1]*v[i-1]/u[i-1];
  }

  _z[n] = 0.0;
  
  for (i = n-1 ; i >= 2 ; --i) {
    _z[i] = (v[i] - h[i]*_z[i+1])/u[i];
  }
   
  _z[1] = 0.0;

  delete v;
  delete b;
  delete h;
  delete u;

  return;
}

CubicSpline::~CubicSpline()
{
  delete _z;
}
 
float CubicSpline::f(float x)
{
  float diff;

  for (int i = _n-1 ; i >= 2 ; --i) {
    diff = x - _t[i];
    if (diff >= 0.0) {
      break;
    }
  }

  if (diff < 0.0) {
    i = 1;
    diff = x - _t[1];
  }

  float h = _t[i+1] - _t[i];
  float b = (_y[i+1] - _y[i])/h - h*(_z[i+1] + 2.0*_z[i])/6.0;
  float p = 0.5*_z[i] + diff*(_z[i+i] - _z[i])/(6.0*h);
  p = b + diff*p;
		       
  return(_y[i] + diff*p);
}

/*
 * ColorScale
 */

ColorScale::ColorScale(int num_colors)
{
  _num_colors = num_colors;
  _color = new Color*[_num_colors];

  Glyph* display =
       new TBBox(
       	   new VGlue(0.0, 0.0, 0.0)
       );

   Glyph* structure = 
       new Overlay(
       	   new VGlue(0.5),
	   new HGlue(50.0)
       );
  
  boolean color = Gems::instance()->color();
  const Color* bw = Gems::instance()->style()->foreground();

  CubicSpline* r = r_spline();
  CubicSpline* g = g_spline();
  CubicSpline* b = b_spline();

  for (int i = 0 ; i < _num_colors ; ++i) {
    float x = (float) i / (float) (_num_colors-1);
    _color[i] = (color ? new Color(r->f(x), g->f(x), b->f(x)) : bw); 
//    printf("x=%f \t r=%f \t g=%f \t b=%f\n", x, r->f(x), x, g->f(x), x, b->f(x));
    display->append(new Background(structure, _color[i]));
  }
    
  body(display);

  delete r, g, b;
}

ColorScale::~ColorScale()
{
  for (int i = 0 ; i < _num_colors ; ++i)
    delete _color[i];

  delete _color;
}

CubicSpline* ColorScale::r_spline()
{
  int n = 10;
  static float t[] = { 0.0, .1, .2, .3, .4, .5, .6, .7, .8, .9, 1.0 };
  static float y[] = { .8, .8, .78, .74, .52, .26, .12, .1, .08, .06, .06 };

  CubicSpline* f = new CubicSpline(n, t, y); 
  return(f);
}

CubicSpline* ColorScale::g_spline()
{
  int n = 10;
  static float t[] = { 0.0, .1, .2, .3, .4, .5, .6, .7, .8, .9, 1.0 };
  static float y[] = { .308, .316, .42, .62, .8, .74, .6, .36, .26, .12, .12};

  CubicSpline* f = new CubicSpline(n, t, y); 
  return(f);
}

CubicSpline* ColorScale::b_spline()
{
  int n = 10;
  static float t[] = { 0.0, .1, .2, .3, .4, .5, .6, .7, .8, .9, 1.0 };
  static float y[] = { .1, .1, .12, .13, .14, .18, .3, .36, .4, .62, .92 };

  CubicSpline* f = new CubicSpline(n, t, y); 
  return(f);
}
